<?php
namespace app\commands\cron;

use app\core\helpers\SettingsHelper;
use app\core\services\CronMaster;
use Yii;
use yii\console\Controller;
use yii\console\ExitCode;

class CronController extends Controller
{
    /**
     * @var array Конфигурация задач
     */
    private array $tasks = [];

    public function init(): void
    {
        parent::init();

        // Инициализируем задачи
        $this->tasks = [
            'backup-full' => [
                'controller' => function() {
                    $controller = Yii::createObject(BackupController::class);
                    return $controller->actionDaily();
                },
                'schedule' => SettingsHelper::getSetting('cron_backup_full', '0 2 * * *'), // Каждый день в 2:00
                'options' => [
                    'timeout' => 3600, // 1 час
                    'description' => 'Полное резервное копирование',
                    'enabled' => SettingsHelper::getSetting('cron_backup_full_enable', true),
                ],
            ],
            'backup-sql' => [
                'controller' => function() {
                    $controller = Yii::createObject(BackupController::class);
                    return $controller->actionDailySql();
                },
                'schedule' => SettingsHelper::getSetting('cron_backup_sql', '0 */6 * * *'), // Каждые 6 часов
                'options' => [
                    'timeout' => 3600, // 1 час
                    'description' => 'Резервное копирование базы данных',
                    'enabled' => SettingsHelper::getSetting('cron_backup_sql_enable', true),
                ],
            ],
            'backup-restore' => [
                'controller' => function($backupId) {
                    $controller = Yii::createObject(BackupController::class);
                    //file_put_contents(Yii::getAlias('@runtime/rrrr.txt'), $backupId);
                    return $controller->actionRestore($backupId);
                },
                'schedule' => '0 6 * * *',
                'options' => [
                    'timeout' => 3600, // 1 час
                    'description' => 'Восстановление базы данных',
                    'enabled' => false, // для не автоматической задачи = false
                    'no_auto' => true // не автоматическая задача лоя крона, enable игнорируется, запуск обязателен
                ],
            ],
//            'cleanup-temp' => [
//                'controller' => [CleanupController::class, 'actionTemp'],
//                'schedule' => '0 */6 * * *', // Каждые 6 часов
//                'options' => [
//                    'timeout' => 600, // 10 минут
//                    'description' => 'Очистка временных файлов',
//                ],
//            ],
//            'send-newsletter' => [
//                'controller' => [NewsletterController::class, 'actionSend'],
//                'schedule' => '0 9 * * 1', // Каждый понедельник в 9:00
//                'options' => [
//                    'timeout' => 7200, // 2 часа
//                    'description' => 'Рассылка новостей',
//                ],
//            ],
//            'update-exchange-rates' => [
//                'controller' => [FinanceController::class, 'actionUpdateRates'],
//                'schedule' => '*/30 * * * *', // Каждые 30 минут
//                'options' => [
//                    'timeout' => 300, // 5 минут
//                    'description' => 'Обновление курсов валют',
//                ],
//            ],
        ];
    }

    /**
     * Запуск всех задач по расписанию
     */
    public function actionRun(): int
    {
        /** @var CronMaster $cron */
        $cron = Yii::createObject(CronMaster::class);
        $cron->setTasks($this->tasks);
        $result = $cron->runScheduledTasks();

        return ExitCode::OK;
    }

    /**
     * Запуск конкретной задачи
     * @param string $taskName Имя задачи
     */
    public function actionTask($taskName): int
    {
        /** @var CronMaster $cron */
        $cron = Yii::createObject(CronMaster::class);
        $cron->setTasks($this->tasks);

        if ($cron->runSingleTask($taskName)) {
            $this->stdout("✅ Задача '$taskName' выполнена успешно\n");
            return ExitCode::OK;
        } else {
            $this->stderr("❌ Ошибка при выполнении задачи '$taskName'\n");
            return ExitCode::UNSPECIFIED_ERROR;
        }
    }

    /**
     * Список всех задач
     */
    public function actionList(): int
    {
        /** @var CronMaster $cron */
        $cron = Yii::createObject(CronMaster::class);
        $cron->setTasks($this->tasks);

        $tasks = $cron->listTasks();

        $this->stdout("Доступные задачи (" . count($tasks) . "):\n");
        $this->stdout(str_repeat("-", 120) . "\n");

        foreach ($tasks as $name => $task) {
            $status = $task['is_running'] ? '🟢 ВЫПОЛНЯЕТСЯ' : ($task['is_enabled'] ? '🟡 ВКЛЮЧЕНА' : '🔴 ВЫКЛЮЧЕНА');

            $this->stdout(sprintf(
                "  %-25s | %-15s | Расписание: %-15s | Запусков: %-4d | Последний: %s\n",
                $name,
                $status,
                $task['schedule'],
                $task['total_runs'],
                $task['last_run']
            ));

            if (!empty($task['description'])) {
                $this->stdout("     📝 " . $task['description'] . "\n");
            }
        }

        return ExitCode::OK;
    }

    /**
     * Показать логи крона
     * @param int $lines Количество строк (по умолчанию 50)
     */
    public function actionLogs($lines = 50)
    {
        /** @var CronMaster $cron */
        $cron = Yii::createObject(CronMaster::class);

        $logLines = $cron->showLogs($lines);

        if (empty($logLines)) {
            $this->stdout("Логи отсутствуют\n");
            return ExitCode::OK;
        }

        $this->stdout("Последние $lines строк лога:\n");
        $this->stdout(str_repeat("=", 80) . "\n");

        foreach ($logLines as $line) {
            $this->stdout($line . "\n");
        }

        return ExitCode::OK;
    }

    /**
     * Очистка старых логов
     */
    public function actionCleanup()
    {
        /** @var CronMaster $cron */
        $cron = Yii::createObject(CronMaster::class);
        $cron->cleanupLogs();

        $this->stdout("✅ Логи очищены\n");
        return ExitCode::OK;
    }

    /**
     * Проверить, выполняется ли задача
     * @param string $taskName Имя задачи
     */
    public function actionStatus($taskName)
    {
        /** @var CronMaster $cron */
        $cron = Yii::createObject(CronMaster::class);

        if ($cron->isTaskRunning($taskName)) {
            $info = $cron->getTaskInfo($taskName);

            $this->stdout("🟢 Задача '$taskName' выполняется\n");
            $this->stdout("   PID: " . ($info['pid'] ?? 'неизвестно') . "\n");
            $this->stdout("   Время выполнения: " . ($info['running_time'] ?? 0) . " сек\n");
            $this->stdout("   Запущена: " . date('Y-m-d H:i:s', $info['start_time'] ?? time()) . "\n");

            if (isset($info['memory_usage'])) {
                $this->stdout("   Использование памяти: " . round($info['memory_usage'] / 1024 / 1024, 2) . " MB\n");
            }
        } else {
            $this->stdout("🔴 Задача '$taskName' не выполняется\n");
        }

        return ExitCode::OK;
    }

    /**
     * Показать все выполняющиеся задачи
     */
    public function actionRunning()
    {
        /** @var CronMaster $cron */
        $cron = Yii::createObject(CronMaster::class);
        $cron->setTasks($this->tasks);

        $runningTasks = $cron->getRunningTasks();

        if (empty($runningTasks)) {
            $this->stdout("Нет выполняющихся задач\n");
            return ExitCode::OK;
        }

        $this->stdout("Выполняющиеся задачи (" . count($runningTasks) . "):\n");
        $this->stdout(str_repeat("=", 80) . "\n");

        foreach ($runningTasks as $taskName => $info) {
            $runningTime = time() - ($info['start_time'] ?? time());
            $memory = isset($info['memory_usage']) ? round($info['memory_usage'] / 1024 / 1024, 2) . ' MB' : 'неизвестно';

            $this->stdout(sprintf(
                "  %-25s | PID: %-6d | Время: %-6d сек | Память: %s\n",
                $taskName,
                $info['pid'] ?? 0,
                $runningTime,
                $memory
            ));
        }

        return ExitCode::OK;
    }

    /**
     * Остановить задачу
     * @param string $taskName Имя задачи
     * @param string $force Принудительная остановка (y/n)
     */
    public function actionStop($taskName, $force = 'n')
    {
        /** @var CronMaster $cron */
        $cron = Yii::createObject(CronMaster::class);

        $forceFlag = strtolower($force) === 'y';

        if ($cron->stopTask($taskName, $forceFlag)) {
            $this->stdout("✅ Задача '$taskName' остановлена" . ($forceFlag ? ' принудительно' : '') . "\n");
            return ExitCode::OK;
        } else {
            $this->stderr("❌ Не удалось остановить задачу '$taskName'\n");
            return ExitCode::UNSPECIFIED_ERROR;
        }
    }

    /**
     * Остановить все задачи
     * @param string $force Принудительная остановка (y/n)
     */
    public function actionStopAll($force = 'n')
    {
        /** @var CronMaster $cron */
        $cron = Yii::createObject(CronMaster::class);
        $cron->setTasks($this->tasks);

        $forceFlag = strtolower($force) === 'y';
        $result = $cron->stopAllTasks($forceFlag);

        $this->stdout("Остановлено задач: {$result['stopped']} из {$result['total']}\n");

        if ($result['failed'] > 0) {
            $this->stderr("Не удалось остановить: {$result['failed']} задач\n");
            return ExitCode::UNSPECIFIED_ERROR;
        }

        return ExitCode::OK;
    }

    /**
     * Показать зависшие задачи
     */
    public function actionStuck()
    {
        /** @var CronMaster $cron */
        $cron = Yii::createObject(CronMaster::class);
        $cron->setTasks($this->tasks);

        $stuckTasks = $cron->getStuckTasks();

        if (empty($stuckTasks)) {
            $this->stdout("Нет зависших задач\n");
            return ExitCode::OK;
        }

        $this->stdout("Зависшие задачи (" . count($stuckTasks) . "):\n");
        $this->stdout(str_repeat("=", 80) . "\n");

        foreach ($stuckTasks as $taskName => $info) {
            $runningTime = time() - ($info['start_time'] ?? time());
            $timeout = $info['timeout'] ?? 3600;

            $this->stdout(sprintf(
                "  %-25s | Запущена: %s | Время работы: %d сек (таймаут: %d сек)\n",
                $taskName,
                date('Y-m-d H:i:s', $info['start_time'] ?? time()),
                $runningTime,
                $timeout
            ));
        }

        return ExitCode::OK;
    }

    /**
     * Очистить все зависшие задачи
     */
    public function actionCleanupStuck()
    {
        /** @var CronMaster $cron */
        $cron = Yii::createObject(CronMaster::class);
        $cron->setTasks($this->tasks);

        $result = $cron->cleanupAllStuckTasks();

        $this->stdout("Очищено зависших задач: {$result['cleaned']} из {$result['total']}\n");

        if ($result['failed'] > 0) {
            $this->stderr("Не удалось очистить: {$result['failed']} задач\n");
            return ExitCode::UNSPECIFIED_ERROR;
        }

        return ExitCode::OK;
    }

    /**
     * Справка
     */
    public function actionHelp()
    {
        $this->stdout("CronMaster для Yii2 Basic - управление задачами по расписанию\n\n");

        $this->stdout("Использование:\n");
        $this->stdout("  ./yii cron/run                       - запуск всех задач по расписанию\n");
        $this->stdout("  ./yii cron/task <name>               - запуск конкретной задачи\n");
        $this->stdout("  ./yii cron/list                      - список всех задач\n");
        $this->stdout("  ./yii cron/logs [lines]              - показать логи (по умолчанию 50 строк)\n");
        $this->stdout("  ./yii cron/cleanup                   - очистка старых логов\n");
        $this->stdout("  ./yii cron/status <name>             - статус задачи\n");
        $this->stdout("  ./yii cron/running                   - показать выполняющиеся задачи\n");
        $this->stdout("  ./yii cron/stop <name> [force]       - остановить задачу (force=y для принудительной)\n");
        $this->stdout("  ./yii cron/stop-all [force]          - остановить все задачи\n");
        $this->stdout("  ./yii cron/stuck                     - показать зависшие задачи\n");
        $this->stdout("  ./yii cron/cleanup-stuck             - очистить все зависшие задачи\n");
        $this->stdout("  ./yii cron/help                      - эта справка\n\n");

        $this->stdout("Примеры:\n");
        $this->stdout("  ./yii cron/run\n");
        $this->stdout("  ./yii cron/task backup-database\n");
        $this->stdout("  ./yii cron/status backup-database\n");
        $this->stdout("  ./yii cron/stop backup-database y    - принудительно остановить\n");
        $this->stdout("  ./yii cron/logs 100\n\n");

        return ExitCode::OK;
    }
}
