<?php

namespace app\commands\cron;

use Exception;
use yii\console\Controller;
use yii\console\ExitCode;
use Yii;

class UpdateController extends Controller
{
    const string NEED_UPDATE_FILE = '@runtime/update_need.txt';
    const string IN_PROGRESS_FILE = '@runtime/update_process.txt';
    const string UPDATE_ARCHIVE_DIR = '@runtime/updates';
    const string LOG_FILE = '@runtime/update_cron.log';
    const string UPDATE_SERVER_URL = 'https://your-update-server.com';
    const string LAST_CHECK_FILE = '@runtime/last_update_check.txt';
    const string CURRENT_VERSION_FILE = '@app/config/version.php';

    public $verbose = false;

    public function options($actionID): array
    {
        return array_merge(parent::options($actionID), ['verbose']);
    }

    public function optionAliases(): array
    {
        return [
            'v' => 'verbose',
        ];
    }

    /**
     * Основная команда для крона
     */
    public function actionIndex(): int
    {
        $this->log("=== Update check started ===", true);

        // 1. Проверяем, есть ли запрос на обновление
        if (!$this->isUpdateRequested()) {
            $this->log("No update requested. Doing auto-check...", true);
            $this->autoCheckUpdates();
            $this->log("=== Update check completed ===", true);
            return ExitCode::OK;
        }

        // 2. Проверяем, не идет ли уже обновление
        if ($this->isUpdateInProgress()) {
            if ($this->isUpdateStuck()) {
                $this->log("Update process is stuck! Cleaning up...", true);
                $this->cleanupStuckUpdate();
            } else {
                $this->log("Update is already in progress.", true);
                $this->log("=== Update check completed ===", true);
                return ExitCode::OK;
            }
        }

        // 3. Начинаем процесс обновления
        $currentVersion = $this->getCurrentVersion();
        $updateChain = $this->getUpdateChain($currentVersion);

        if (empty($updateChain)) {
            $this->log("No updates available in chain", true);
            $this->clearUpdateRequest();
            $this->log("=== Update check completed ===", true);
            return ExitCode::OK;
        }

        $this->startUpdateProcess($updateChain);

        try {
            $this->processUpdateChain($currentVersion, $updateChain);
            $this->completeUpdateProcess();
            $this->log("=== All updates completed successfully ===", true);
            return ExitCode::OK;

        } catch (Exception $e) {
            $this->handleUpdateFailure($e, $currentVersion);
            $this->log("=== Update chain failed ===", true);
            return ExitCode::UNSPECIFIED_ERROR;
        }
    }

    /**
     * Автопроверка обновлений
     */
    private function autoCheckUpdates(): void
    {
        if (!$this->shouldCheckForUpdates()) {
            return;
        }

        $currentVersion = $this->getCurrentVersion();
        $this->log("Auto-check for updates from version: {$currentVersion}", true);

        try {
            $updates = $this->fetchAvailableUpdates($currentVersion);

            if (!empty($updates)) {
                $this->log("New updates available: " . implode(', ', $updates), true);
                $this->createAutoUpdateRequest($updates);
                $this->sendNotification("New updates available: " . implode(', ', $updates));
            } else {
                $this->log("No new updates available", true);
            }

            $this->updateLastCheckTime();

        } catch (Exception $e) {
            $this->log("Auto-check failed: " . $e->getMessage(), true);
        }
    }

    /**
     * Обрабатывает цепочку обновлений
     */
    private function processUpdateChain(string $currentVersion, array $updateChain): void
    {
        $this->log("Processing update chain: " . implode(' → ', $updateChain), true);
        $this->log("Total steps: " . count($updateChain), true);

        $step = 1;
        foreach ($updateChain as $targetVersion) {
            $this->log("--- Step {$step}/" . count($updateChain) . ": {$currentVersion} → {$targetVersion} ---", true);
            $this->updateProgressStep($step - 1);

            $this->applyVersionUpdate($currentVersion, $targetVersion);

            $currentVersion = $targetVersion;
            $step++;
        }

        $this->log("All steps completed successfully!", true);
    }

    /**
     * Применяет обновление для одной версии
     */
    private function applyVersionUpdate(string $fromVersion, string $toVersion): void
    {
        $this->log("Starting update {$fromVersion} → {$toVersion}");

        // 1. Скачиваем архив
        $archivePath = $this->downloadUpdateArchive($fromVersion, $toVersion);
        $this->log("Archive downloaded: " . basename($archivePath));

        // 2. === МЕСТО ДЛЯ БЭКАПА ===
        $this->log("Pre-update backup...");
        // $backupResult = $this->createBackup($fromVersion, $toVersion);
        // if (!$backupResult) {
        //     throw new \Exception("Backup failed");
        // }

        // 3. Применяем обновление
        $this->log("Applying update...");
        $this->applyUpdateArchive($archivePath);

        // 4. Обновляем версию
        $this->updateSystemVersion($toVersion);

        // 5. Архивируем
        $this->archiveAppliedUpdate($archivePath);

        $this->log("Update {$fromVersion} → {$toVersion} completed successfully");
    }

    /**
     * Скачивает архив обновления
     */
    private function downloadUpdateArchive(string $fromVersion, string $toVersion): string
    {
        $updatesDir = Yii::getAlias(self::UPDATE_ARCHIVE_DIR);
        if (!is_dir($updatesDir)) {
            mkdir($updatesDir, 0755, true);
        }

        $archivePath = $updatesDir . '/' . $toVersion . '-from-' . $fromVersion . '.zip';

        // Если архив уже есть локально
        if (file_exists($archivePath)) {
            $this->log("Using existing archive: " . basename($archivePath));
            return $archivePath;
        }

        // Скачиваем с сервера
        $url = self::UPDATE_SERVER_URL . '/api/updates/download';
        $params = [
            'from' => $fromVersion,
            'to' => $toVersion,
            'app_key' => $this->getAppKey(),
        ];

        $downloadUrl = $url . '?' . http_build_query($params);
        $this->log("Downloading from: " . $downloadUrl);

        $fileContent = @file_get_contents($downloadUrl);
        if ($fileContent === false) {
            throw new Exception("Failed to download update archive from server");
        }

        file_put_contents($archivePath, $fileContent);

        if (!file_exists($archivePath)) {
            throw new Exception("Failed to save update archive");
        }

        // Проверяем хэш
        if (!$this->verifyArchiveHash($archivePath, $fromVersion, $toVersion)) {
            @unlink($archivePath);
            throw new Exception("Archive hash verification failed");
        }

        return $archivePath;
    }

    /**
     * Применяет архив обновления
     */
    private function applyUpdateArchive(string $archivePath): void
    {
        $updateController = new \app\commands\UpdateController('update', Yii::$app);
        $updateController->runAction('apply', [$archivePath]);
    }

    /**
     * Проверяет хэш архива
     */
    private function verifyArchiveHash(string $archivePath, string $fromVersion, string $toVersion): bool
    {
        try {
            $url = self::UPDATE_SERVER_URL . '/api/updates/verify';
            $params = [
                'from' => $fromVersion,
                'to' => $toVersion,
                'app_key' => $this->getAppKey(),
            ];

            $response = $this->makeApiRequest($url, $params);

            if (!$response['success']) {
                $this->log("Hash verification API failed: " . ($response['error'] ?? 'Unknown error'));
                return false;
            }

            $expectedHash = $response['hash'] ?? '';
            if (empty($expectedHash)) {
                $this->log("No hash returned from server");
                return false;
            }

            $actualHash = hash_file('sha256', $archivePath);

            if (!hash_equals($expectedHash, $actualHash)) {
                $this->log("Hash mismatch!");
                $this->log("Expected: {$expectedHash}");
                $this->log("Actual: {$actualHash}");
                return false;
            }

            $this->log("Archive hash verified successfully");
            return true;

        } catch (Exception $e) {
            $this->log("Hash verification exception: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Обновляет версию системы
     */
    private function updateSystemVersion(string $version): void
    {
        $versionFile = Yii::getAlias(self::CURRENT_VERSION_FILE);
        file_put_contents($versionFile, "<?php return ['version' => '$version'];");
        $this->log("System version updated to: {$version}");
    }

    /**
     * Архивирует примененное обновление
     */
    private function archiveAppliedUpdate(string $archivePath): void
    {
        if (!file_exists($archivePath)) {
            return;
        }

        $archiveDir = dirname($archivePath);
        $appliedDir = $archiveDir . '/applied';

        if (!is_dir($appliedDir)) {
            mkdir($appliedDir, 0755, true);
        }

        $newName = $appliedDir . '/' . date('Y-m-d_His_') . basename($archivePath);

        if (rename($archivePath, $newName)) {
            $this->log("Update archived to: " . basename($newName));
        } else {
            $this->log("Failed to archive update file");
        }
    }

    /**
     * Получает текущую версию
     */
    private function getCurrentVersion(): string
    {
        return Yii::$app->params['version'];
    }

    /**
     * Проверяет, нужно ли делать автопроверку
     */
    private function shouldCheckForUpdates(): bool
    {
        $lastCheckFile = Yii::getAlias(self::LAST_CHECK_FILE);

        if (!file_exists($lastCheckFile)) {
            return true;
        }

        $lastCheck = filemtime($lastCheckFile);
        return (time() - $lastCheck) > 3600; // Раз в час
    }

    /**
     * Обновляет время последней проверки
     */
    private function updateLastCheckTime(): void
    {
        $lastCheckFile = Yii::getAlias(self::LAST_CHECK_FILE);
        touch($lastCheckFile);
    }

    /**
     * Проверяет, есть ли запрос на обновление
     */
    private function isUpdateRequested(): bool
    {
        $needFile = Yii::getAlias(self::NEED_UPDATE_FILE);
        return file_exists($needFile);
    }

    /**
     * Проверяет, идет ли обновление
     */
    private function isUpdateInProgress(): bool
    {
        $progressFile = Yii::getAlias(self::IN_PROGRESS_FILE);
        return file_exists($progressFile);
    }

    /**
     * Проверяет, не зависло ли обновление
     */
    private function isUpdateStuck(): bool
    {
        $progressFile = Yii::getAlias(self::IN_PROGRESS_FILE);

        if (!file_exists($progressFile)) {
            return false;
        }

        $fileTime = filemtime($progressFile);
        $stuckTime = 6 * 60 * 60; // 6 часов

        return (time() - $fileTime) > $stuckTime;
    }

    /**
     * Очищает зависший процесс
     */
    private function cleanupStuckUpdate(): void
    {
        $progressFile = Yii::getAlias(self::IN_PROGRESS_FILE);
        if (file_exists($progressFile)) {
            $data = json_decode(file_get_contents($progressFile), true);
            $failedFile = str_replace('update_process.txt', 'update_stuck_' . date('Ymd_His') . '.txt', $progressFile);
            rename($progressFile, $failedFile);
            $this->log("Stuck process moved to: " . basename($failedFile));
        }
    }

    /**
     * Получает цепочку обновлений
     */
    private function getUpdateChain(string $currentVersion): array
    {
        $needFile = Yii::getAlias(self::NEED_UPDATE_FILE);

        if (!file_exists($needFile)) {
            return [];
        }

        $data = json_decode(file_get_contents($needFile), true);

        // Если уже есть готовый массив обновлений
        if (isset($data['updates']) && is_array($data['updates'])) {
            return $data['updates'];
        }

        // Если указана целевая версия
        if (isset($data['target_version'])) {
            $targetVersion = $data['target_version'];
            return $this->fetchUpdateChainFromServer($currentVersion, $targetVersion);
        }

        return [];
    }

    /**
     * Запрашивает цепочку с сервера
     */
    private function fetchUpdateChainFromServer(string $fromVersion, string $toVersion): array
    {
        $this->log("Fetching update chain from {$fromVersion} to {$toVersion}");

        $url = self::UPDATE_SERVER_URL . '/api/updates/chain';
        $params = [
            'from' => $fromVersion,
            'to' => $toVersion,
            'app_key' => $this->getAppKey(),
        ];

        $response = $this->makeApiRequest($url, $params);

        if (!$response['success']) {
            throw new Exception("Failed to fetch update chain: " . ($response['error'] ?? 'Unknown error'));
        }

        return $response['chain'] ?? [];
    }

    /**
     * Запрашивает доступные обновления
     */
    private function fetchAvailableUpdates(string $currentVersion): array
    {
        $url = self::UPDATE_SERVER_URL . '/api/updates/available';
        $params = [
            'current' => $currentVersion,
            'app_key' => $this->getAppKey(),
        ];

        $response = $this->makeApiRequest($url, $params);

        if (!$response['success']) {
            throw new Exception("Failed to fetch available updates: " . ($response['error'] ?? 'Unknown error'));
        }

        return $response['updates'] ?? [];
    }

    /**
     * Создает автоматический запрос
     */
    private function createAutoUpdateRequest(array $updates): void
    {
        $needFile = Yii::getAlias(self::NEED_UPDATE_FILE);

        file_put_contents($needFile, json_encode([
            'requested_at' => date('Y-m-d H:i:s'),
            'requested_by' => 'auto_check',
            'updates' => $updates,
            'target_version' => end($updates),
            'auto' => true
        ], JSON_PRETTY_PRINT));

        $this->log("Auto-update request created for versions: " . implode(', ', $updates));
    }

    /**
     * Начинает процесс обновления
     */
    private function startUpdateProcess(array $updateChain): void
    {
        $progressFile = Yii::getAlias(self::IN_PROGRESS_FILE);

        file_put_contents($progressFile, json_encode([
            'started_at' => date('Y-m-d H:i:s'),
            'pid' => getmypid(),
            'current_chain' => $updateChain,
            'current_step' => 0,
            'total_steps' => count($updateChain),
            'status' => 'in_progress'
        ], JSON_PRETTY_PRINT));

        $this->log("Update process started. PID: " . getmypid());
        $this->log("Chain: " . implode(' → ', $updateChain));
    }

    /**
     * Обновляет текущий шаг в процессе
     */
    private function updateProgressStep(int $step): void
    {
        $progressFile = Yii::getAlias(self::IN_PROGRESS_FILE);

        if (file_exists($progressFile)) {
            $data = json_decode(file_get_contents($progressFile), true);
            $data['current_step'] = $step;
            file_put_contents($progressFile, json_encode($data, JSON_PRETTY_PRINT));
        }
    }

    /**
     * Завершает процесс успешно
     */
    private function completeUpdateProcess(): void
    {
        // Удаляем запрос
        $needFile = Yii::getAlias(self::NEED_UPDATE_FILE);
        if (file_exists($needFile)) {
            unlink($needFile);
        }

        // Удаляем процесс
        $progressFile = Yii::getAlias(self::IN_PROGRESS_FILE);
        if (file_exists($progressFile)) {
            unlink($progressFile);
        }

        $this->log("Update process completed and cleaned up");
    }

    /**
     * Очищает запрос на обновление
     */
    private function clearUpdateRequest(): void
    {
        $needFile = Yii::getAlias(self::NEED_UPDATE_FILE);
        if (file_exists($needFile)) {
            unlink($needFile);
        }
    }

    /**
     * Обрабатывает ошибку обновления
     */
    private function handleUpdateFailure(Exception $e, string $currentVersion): void
    {
        $error = $e->getMessage();
        $this->log("Update chain failed: " . $error, true);
        $this->failUpdateProcess($error, $currentVersion);
        $this->sendNotification("Update FAILED at version {$currentVersion}: " . $error);
    }

    /**
     * Завершает процесс с ошибкой
     */
    private function failUpdateProcess(string $error, string $currentVersion): void
    {
        $progressFile = Yii::getAlias(self::IN_PROGRESS_FILE);

        if (file_exists($progressFile)) {
            $data = json_decode(file_get_contents($progressFile), true) ?: [];
            $data['finished_at'] = date('Y-m-d H:i:s');
            $data['status'] = 'failed';
            $data['error'] = $error;
            $data['failed_at_version'] = $currentVersion;

            file_put_contents($progressFile, json_encode($data, JSON_PRETTY_PRINT));

            $failedFile = str_replace('update_process.txt', 'update_failed_' . date('Ymd_His') . '.txt', $progressFile);
            rename($progressFile, $failedFile);

            $this->log("Update process marked as failed. File: " . basename($failedFile));
        }
    }

    /**
     * API запрос
     */
    private function makeApiRequest(string $url, array $params): array
    {
        $this->log("API request to: {$url}", false);

        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($params));
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/json',
            'X-App-Key: ' . $this->getAppKey(),
        ]);

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $curlError = curl_error($ch);
        curl_close($ch);

        if ($curlError) {
            return ['success' => false, 'error' => "CURL error: {$curlError}"];
        }

        if ($httpCode !== 200) {
            return ['success' => false, 'error' => "HTTP {$httpCode}"];
        }

        $data = json_decode($response, true);

        if (json_last_error() !== JSON_ERROR_NONE) {
            return ['success' => false, 'error' => 'Invalid JSON response: ' . json_last_error_msg()];
        }

        return $data;
    }

    /**
     * Генерирует ключ приложения
     */
    private function getAppKey(): string
    {
        $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
        $secret = Yii::$app->params['appSecret'] ?? 'default_secret_key';
        return hash('sha256', $host . $secret);
    }

    /**
     * Логирование
     */
    private function log(string $message, bool $force = false): void
    {
        if ($this->verbose || $force) {
            $this->stdout($message . "\n");
        }

        $logFile = Yii::getAlias(self::LOG_FILE);
        $timestamp = date('Y-m-d H:i:s');
        $logMessage = "[{$timestamp}] {$message}\n";
        file_put_contents($logFile, $logMessage, FILE_APPEND);
    }

    /**
     * Отправляет уведомление
     */
    private function sendNotification(string $message): void
    {
        $this->log("Notification: {$message}");
        // Реализуйте по необходимости
    }

    /**
     * Команда для ручного запроса обновления до версии
     */
    public function actionRequestVersion($targetVersion)
    {
        $currentVersion = $this->getCurrentVersion();

        $this->stdout("Current version: {$currentVersion}\n");
        $this->stdout("Target version: {$targetVersion}\n");

        if (version_compare($targetVersion, $currentVersion) <= 0) {
            $this->stdout("❌ Target version must be greater than current version\n");
            return ExitCode::DATAERR;
        }

        $needFile = Yii::getAlias(self::NEED_UPDATE_FILE);

        if (file_exists($needFile)) {
            $this->stdout("⚠️  Update is already requested. Overwriting...\n");
        }

        file_put_contents($needFile, json_encode([
            'requested_at' => date('Y-m-d H:i:s'),
            'requested_by' => get_current_user() ?: 'console',
            'target_version' => $targetVersion,
            'auto' => false
        ], JSON_PRETTY_PRINT));

        $this->stdout("✅ Update to version {$targetVersion} requested.\n");
        $this->stdout("Cron will process it within the next minute.\n");

        return ExitCode::OK;
    }

    /**
     * Команда для проверки статуса
     */
    public function actionStatus()
    {
        $needFile = Yii::getAlias(self::NEED_UPDATE_FILE);
        $progressFile = Yii::getAlias(self::IN_PROGRESS_FILE);
        $currentVersion = $this->getCurrentVersion();

        $this->stdout("Current version: {$currentVersion}\n");
        $this->stdout("Last check: " . date('Y-m-d H:i:s', @filemtime($progressFile)) . "\n");

        if (file_exists($progressFile)) {
            $data = json_decode(file_get_contents($progressFile), true);
            $this->stdout("\n=== UPDATE IN PROGRESS ===\n");
            $this->stdout("Started: " . ($data['started_at'] ?? 'unknown') . "\n");
            $this->stdout("PID: " . ($data['pid'] ?? 'unknown') . "\n");
            $this->stdout("Status: " . ($data['status'] ?? 'unknown') . "\n");

            if (isset($data['current_chain'])) {
                $this->stdout("Chain: " . implode(' → ', $data['current_chain']) . "\n");
                $this->stdout("Step: " . (($data['current_step'] ?? 0) + 1) . "/" . ($data['total_steps'] ?? 0) . "\n");
            }

            if (isset($data['status']) && $data['status'] === 'failed') {
                $this->stdout("Error: " . ($data['error'] ?? 'unknown') . "\n");
                $this->stdout("Failed at version: " . ($data['failed_at_version'] ?? 'unknown') . "\n");
            }

        } elseif (file_exists($needFile)) {
            $data = json_decode(file_get_contents($needFile), true);
            $this->stdout("\n=== UPDATE REQUESTED ===\n");
            $this->stdout("Requested at: " . ($data['requested_at'] ?? 'unknown') . "\n");
            $this->stdout("Requested by: " . ($data['requested_by'] ?? 'unknown') . "\n");

            if (isset($data['target_version'])) {
                $this->stdout("Target version: " . $data['target_version'] . "\n");
            }

            if (isset($data['updates'])) {
                $this->stdout("Updates chain: " . implode(' → ', $data['updates']) . "\n");
            }

            $this->stdout("Status: Waiting for cron\n");

        } else {
            $this->stdout("\nNo update requested or in progress\n");
        }

        // Проверяем наличие архивов
        $updatesDir = Yii::getAlias(self::UPDATE_ARCHIVE_DIR);
        if (is_dir($updatesDir)) {
            $archives = glob($updatesDir . '/*.zip');
            $applied = glob($updatesDir . '/applied/*.zip');

            $this->stdout("\nArchives:\n");
            $this->stdout("  Pending: " . count($archives) . "\n");
            $this->stdout("  Applied: " . count($applied) . "\n");
        }

        return ExitCode::OK;
    }

    /**
     * Команда для принудительной проверки обновлений
     */
    public function actionCheck()
    {
        $this->verbose = true;
        $this->autoCheckUpdates();
        return ExitCode::OK;
    }

    /**
     * Команда для очистки запроса на обновление
     */
    public function actionClear()
    {
        $needFile = Yii::getAlias(self::NEED_UPDATE_FILE);

        if (file_exists($needFile)) {
            unlink($needFile);
            $this->stdout("✅ Update request cleared\n");
        } else {
            $this->stdout("No update request found\n");
        }

        return ExitCode::OK;
    }

    /**
     * Команда для сброса зависшего процесса
     */
    public function actionReset(): int
    {
        $progressFile = Yii::getAlias(self::IN_PROGRESS_FILE);

        if (file_exists($progressFile)) {
            $data = json_decode(file_get_contents($progressFile), true);
            if (isset($data['status']) && $data['status'] === 'in_progress') {
                $failedFile = str_replace('update_process.txt', 'update_manually_reset_' . date('Ymd_His') . '.txt', $progressFile);
                rename($progressFile, $failedFile);
                $this->stdout("✅ Update process reset. File moved to: " . basename($failedFile) . "\n");
            } else {
                unlink($progressFile);
                $this->stdout("✅ Update process file removed\n");
            }
        } else {
            $this->stdout("No update process file found\n");
        }

        return ExitCode::OK;
    }
}