<?php
namespace app\controllers;

use app\core\components\JwtAuthBehavior;
use Yii;
use yii\filters\auth\HttpBearerAuth;
use yii\filters\Cors;
use yii\rest\Controller;
use yii\web\ForbiddenHttpException;

class ApiController extends Controller
{
    public array $accessRules = [];

    public static function publicActions(): array
    {
        return [];
    }

    public static function allowedDomains(): array
    {
        return [
            'http://localhost',
            Yii::$app->params['generalDomain']
            //'https://id-pass.ru'
            //'*',                        // star allows all domains
            //'http://test1.example.com',
            //'http://test2.example.com',
        ];
    }

    public function behaviors(): array
    {
        $behaviors = parent::behaviors();

        // Добавляем JWT behavior для автоматического обновления токена
        $behaviors['jwtAuth'] = [
            'class' => JwtAuthBehavior::class,
        ];

        $behaviors['authenticator'] = [
            'class' => HttpBearerAuth::class,
            'optional' => array_merge(['options'], static::publicActions()),
            //'except' => ['options'],
        ];

        $behaviors['corsFilter'] = [
            'class' => Cors::class,
            'cors'  => [
                // restrict access to domains:
                'Origin'                           => static::allowedDomains(),
                'Access-Control-Request-Method'    => ['GET','POST','PATCH','PUT','DELETE'],
                'Access-Control-Allow-Credentials' => true,
                'Access-Control-Max-Age'           => 3600,                 // Cache (seconds)
                'Access-Control-Allow-Headers' => ['content-type', 'authorization', 'x-seed', 'x-new-token', 'x-total-count'],
                'Access-Control-Expose-Headers' => ['X-Total-Count', 'x-total-count', 'X-New-Token', 'x-new-token', 'x-seed'],
                'Access-Control-Request-Headers' => ['*'],
            ],
        ];

        return $behaviors;
    }

    public function actions(): array
    {
        return [
            'options' => [
                'class' => 'yii\rest\OptionsAction',
            ],
        ];
    }

    public function beforeAction($action): bool
    {
        if (!parent::beforeAction($action)) {
            return false;
        }

        $this->checkAccess($action);
        return true;
    }

    /**
     * Проверяет доступ к действию на основе accessRules
     */
    protected function checkAccess($action): void
    {
        $actionId = $action->id;

        // Если правила не заданы для действия, разрешаем всем
        if (!isset($this->accessRules[$actionId])) {
            return;
        }

        // Проверка авторизации
        if (Yii::$app->user->isGuest) {
            throw new ForbiddenHttpException('Требуется авторизация');
        }

        // Проверка ролей
        $requiredRoles = $this->accessRules[$actionId];
        if (!empty($requiredRoles)) {
            $hasAccess = false;
            foreach ($requiredRoles as $role) {
                if (Yii::$app->user->can($role)) {
                    $hasAccess = true;
                    break;
                }
            }

            if (!$hasAccess) {
                $rolesText = implode(' или ', $requiredRoles);
                throw new ForbiddenHttpException("Требуются права: {$rolesText}");
            }
        }
    }
}
