<?php
namespace app\controllers;

use app\core\components\JwtAuthBehavior;
use app\core\entities\User;
use app\core\helpers\SerializeHelper;
use app\core\repositories\user\UserRepository;
use app\core\services\user\PasswordResetService;
use app\core\services\user\UserService;
use DomainException;
use Firebase\JWT\JWT;
use Yii;
use yii\base\InvalidConfigException;
use yii\db\Exception;
use yii\filters\auth\HttpBearerAuth;
use yii\helpers\Json;
use yii\rest\Controller;
use yii\validators\EmailValidator;

class AuthController extends Controller
{
    private UserRepository $users;
    private UserService $user_service;
    private PasswordResetService $password_reset_service;

    public function __construct(
        $id,
        $module,
        UserRepository $users,
        PasswordResetService $password_reset_service,
        UserService $user_service,
        $config = []
    )
    {
        parent::__construct($id, $module, $config);
        $this->users = $users;
        $this->password_reset_service = $password_reset_service;
        $this->user_service = $user_service;
    }

    public function actionGetCode(): array
    {
        $email = Yii::$app->request->post('email');
        $v = new EmailValidator();
        $v->enableIDN = true;
        if (!$v->validate($email)) {
            return [
                'result' => 'error',
                'message' => 'Ошибка в e-mail адресе'
            ];
        }

        $user = User::find()->andWhere(['email' => $email])->one();
        try {
            $this->password_reset_service->resetRequest($user);
        } catch (\DomainException $e) {
            return [
                'result' => 'error',
                'message' => $e->getMessage()
            ];
        }

        return [
            "result" => "success"
        ];
    }

    /**
     * @throws \yii\base\Exception
     */
    public function actionResetPassword(): array
    {
        $email = Yii::$app->request->post('email');
        $code = Yii::$app->request->post('code');
        $v = new EmailValidator();
        $v->enableIDN = true;
        if (!$v->validate($email)) {
            return [
                'result' => 'error',
                'message' => 'Ошибка в e-mail адресе'
            ];
        }
        $user = User::find()->andWhere(['email' => $email])->one();
        if (!$user->isPasswordResetCodeValid($code)) {
            return [
                'result' => 'error',
                'message' => 'Код не найден или устарел. Повторите процедуру восстановления пароля'
            ];
        }

        $this->password_reset_service->smartReset($user);

        return [
            "result" => "success"
        ];
    }

    /**
     * @throws Exception
     * @throws \yii\base\Exception
     */
    public function actionCheck(): array
    {
        $email = Yii::$app->request->post('username');
        $v = new EmailValidator();
        $v->enableIDN = true;
        if (!$v->validate($email)) {
            return [
                'result' => 'error',
                'message' => 'Ошибка в e-mail адресе'
            ];
        }

        $email = mb_strtolower($email);
        $user = $this->users->findByEmail($email);
        if (!$user) {
            // регистрация
            $user = $this->signup_service->smartSignup($email);
            return [
                'result' => 'registered'
            ];
        }

        return [
            'result' => 'success'
        ];
    }

    private function generateJWT(User $user): array
    {
        // Получаем секретный ключ
        $secretKey = Yii::$app->params['jwtSecretKey'];

        // Если ключ в формате "base64:..." - декодируем
        if (str_starts_with($secretKey, 'base64:')) {
            $secretKey = base64_decode(substr($secretKey, 7));
        }

        // Время
        $issuedAt = time();
        $expire = $issuedAt + (Yii::$app->params['jwtExpired'] ?? 86400 * 30); // 30 дней по умолчанию

        // Полезная нагрузка (payload)
        $payload = [
            'uid' => $user->id,
            'email' => $user->email,
            'username' => $user->username,
            'iss' => Yii::$app->params['jwtIssuer'] ?? Yii::$app->request->hostName,
            'aud' => Yii::$app->params['jwtAudience'] ?? Yii::$app->request->hostName,
            'iat' => $issuedAt,
            'nbf' => $issuedAt,
            'exp' => $expire,
        ];

        // Генерация JWT токена
        $token = JWT::encode($payload, $secretKey, 'HS512');

        return [
            'success' => true,
            'data' => [
                'token' => $token,
                'token_type' => 'Bearer',
                'expires_in' => Yii::$app->params['jwtExpired'],
                'user' => [
                    'id' => $user->id,
                    'email' => $user->email,
                    'username' => $user->username,
                ]
            ],
        ];
    }

    /**
     * JWT login
     * @return string[]
     */
    public function actionLogin(): array
    {
        $email = Yii::$app->request->post('email');
        $password = Yii::$app->request->post('password');

        // check email
        $v = new EmailValidator();
        $v->enableIDN = true;
        if (!$v->validate($email)) {
            return [
                'success' => false,
                'error' => 'Ошибка в e-mail адресе'
            ];
        }

        // check user exists
        $email = mb_strtolower($email);
        $user = $this->users->findByEmail($email);
        if (!$user || !$user->isActive() || !$user->validatePassword($password)) {
            return [
                'success' => false,
                'error' => 'E-mail адрес или пароль указаны неверно'
            ];
        }

        return $this->generateJWT($user);
    }

    public function actionLoginToken($token): array
    {
        try {
            $user = $this->user_service->authByToken($token);
            return $this->generateJWT($user);
        } catch (DomainException $e) {
            return [
                'result' => 'error',
                'message' => $e->getMessage()
            ];
        }
    }

//    public function actionLoginQr(): array
//    {
//        $token = Yii::$app->request->post('token');
//        $user = User::find()
//            ->andWhere(['login_token' => $token])
//            ->andWhere(['>', 'login_token_at', time()])
//            ->one();
//
//        if ($user) {
//            // reset token
//            $user->login_token = null;
//            $user->login_token_at = 0;
//            $this->users->save($user);
//
//            // Получаем секретный ключ
//            $secretKey = Yii::$app->params['jwtSecretKey'];
//
//            // Если ключ в формате "base64:..." - декодируем
//            if (str_starts_with($secretKey, 'base64:')) {
//                $secretKey = base64_decode(substr($secretKey, 7));
//            }
//            // Время
//            $issuedAt = time();
//            $expire = $issuedAt + (Yii::$app->params['jwtExpired'] ?? 86400 * 30); // 30 дней по умолчанию
//
//            // Полезная нагрузка (payload)
//            $payload = [
//                'uid' => $user->id,
//                'email' => $user->email,
//                'username' => $user->username,
//                'iss' => Yii::$app->params['jwtIssuer'] ?? Yii::$app->request->hostName,
//                'aud' => Yii::$app->params['jwtAudience'] ?? Yii::$app->request->hostName,
//                'iat' => $issuedAt,
//                'nbf' => $issuedAt,
//                'exp' => $expire,
//            ];
//
//            // Генерация JWT токена
//            $token = JWT::encode($payload, $secretKey, 'HS512');
//
//            return [
//                'result' => 'success',
//                'data' => [
//                    'token' => $token,
//                    'token_type' => 'Bearer',
//                    'expires_in' => Yii::$app->params['jwtExpired'],
//                    'user' => [
//                        'id' => $user->id,
//                        'email' => $user->email,
//                        'username' => $user->username,
//                    ]
//                ],
//            ];
//        }
//
//        return [
//            'result' => 'error',
//        ];
//    }

    public function actionCheckTelegramAuth(): array
    {
        $code = Yii::$app->request->post('code');

        if (!$code || !preg_match('/^[a-zA-Z0-9]+$/', $code)) {
            return ['result' => 'error', 'message' => 'Ошибка кода'];
        }

        $storagePath = Yii::getAlias('@runtime/user_auth');
        $filename = $storagePath . '/' . $code . '.json';

        if (!file_exists($filename)) {
            return ['result' => 'error', 'message' => 'Код не нйден'];
        }

        $data = Json::decode(file_get_contents($filename));

        // Проверка времени (например, 10 минут)
        if (time() - $data['timestamp'] > 600) {
            unlink($filename); // Удаляем просроченный файл
            return ['result' => 'error', 'message' => 'Код истёк, попробуйте снова'];
        }

        // Находим пользователя по chat_id
        $user = User::find()->where(['tg_id' => $data['chat_id']])->one();

        if (!$user) {
            return ['result' => 'error', 'message' => 'Пользователь не найден'];
        }

        // Удаляем файл после успешной проверки
        unlink($filename);

        // Генерируем JWT токен
        return $this->generateJWT($user);
    }


    public function behaviors(): array
    {
        $behaviors = parent::behaviors();
        $behaviors['authenticator']['except'] = ['login', 'login-qr', 'check', 'token', 'get-code', 'reset-password', 'check-telegram-auth'];
        return $behaviors;
    }

//    public function behaviors(): array
//    {
//        $behaviors = parent::behaviors();
//
//        // Добавляем JWT behavior для автоматического обновления токена
//        $behaviors['jwtAuth'] = [
//            'class' => JwtAuthBehavior::class,
//        ];
//
//        $behaviors['authenticator'] = [
//            'class' => HttpBearerAuth::class,
//            'except' => ['login', 'login-qr', 'check', 'token', 'get-code', 'reset-password'],
//        ];
//
//        return $behaviors;
//    }
}
