<?php

namespace app\controllers;

use app\core\helpers\FileHelper;
use app\core\services\backup\BackupManager;
use app\core\services\CronMaster;
use Exception;
use RecursiveDirectoryIterator;
use RecursiveIteratorIterator;
use Yii;

class BackupController extends ApiController
{
    private BackupManager $backupManager;
    private CronMaster $cronMaster;

    public function __construct($id, $module, BackupManager $backupManager, CronMaster $cronMaster, $config = [])
    {
        parent::__construct($id, $module, $config);
        $this->backupManager = $backupManager;
        $this->cronMaster = $cronMaster;
    }

    /**
     * Создать новый бэкап
     */
    public function actionCreate(): array
    {
        if (Yii::$app->params['demo']) { // Демо режим
            return [
                'result' => 'error',
                'message' => 'Запрещено в демо режиме'
            ];
        }

        $type = Yii::$app->request->post('type', 'sql');
        $description = Yii::$app->request->post('description');

        try {
            $result = $this->backupManager->createBackup(
                $type,
                $description,
                Yii::$app->user->id
            );

            return [
                'result' => 'success',
                'data' => $result,
                'message' => 'Резервная копия создана успешно'
            ];
        } catch (Exception $e) {
            return [
                'result' => 'error',
                'message' => $e->getMessage()
            ];
        }
    }

    /**
     * Получить список бэкапов
     */
    public function actionList(): array
    {
        $backups = $this->backupManager->listBackups();

        return [
            'result' => 'success',
            'backups' => $backups,
            'total' => count($backups),
            'totalSize' => $this->backupManager->sizeBackups(),
            'lastBackupDate' => $this->backupManager->lastBackupsDate()
        ];
    }

    /**
     * Удалить бэкап
     */
    public function actionDelete(): array
    {
        if (Yii::$app->params['demo']) { // Демо режим
            return [
                'result' => 'error',
                'message' => 'Запрещено в демо режиме'
            ];
        }

        $id = Yii::$app->request->post('id');
        try {
            $success = $this->backupManager->deleteBackup($id);

            if ($success) {
                return [
                    'result' => 'success',
                    'message' => 'Резервная копия удалена'
                ];
            } else {
                return [
                    'result' => 'error',
                    'message' => 'Бэкап не найден'
                ];
            }
        } catch (Exception $e) {
            return [
                'result' => 'error',
                'message' => $e->getMessage()
            ];
        }
    }

    /**
     * Восстановить из бэкапа
     */
    public function actionRestore(): array
    {
        if (Yii::$app->params['demo']) { // Демо режим
            return [
                'result' => 'error',
                'message' => 'Запрещено в демо режиме'
            ];
        }

        // ВНИМАНИЕ: Это опасная операция!
        // Нужно подтверждение и дополнительные проверки
        $id = Yii::$app->request->post('id');
        $confirm = Yii::$app->request->post('confirm', false);

        if (!$confirm) {
            return [
                'result' => 'error',
                'message' => 'Требуется подтверждение операции восстановления'
            ];
        }

        try {
            $result = $this->backupManager->restoreBackup($id);

            return [
                'result' => 'success',
                'data' => $result,
                'message' => 'Восстановление выполнено успешно'
            ];
        } catch (Exception $e) {
            return [
                'result' => 'error',
                'message' => $e->getMessage()
            ];
        }
    }

    /**
     * Получить информацию о системе для бэкапов
     */
    public function actionSystemInfo(): array
    {
        return [
            'result' => 'success',
            'info' => [
                'backup_path' => Yii::getAlias($this->backupManager->backupPath),
                'max_backups' => $this->backupManager->maxBackups,
                'disk_free_space' => disk_free_space(Yii::getAlias('@app')),
                'disk_total_space' => disk_total_space(Yii::getAlias('@app')),
                'uploads_size' => $this->calculateUploadsSize(),
            ]
        ];
    }

    /**
     * Посчитать размер папки uploads
     */
    private function calculateUploadsSize(): int
    {
        $uploadsPath = Yii::getAlias('@webroot/uploads');
        if (!file_exists($uploadsPath)) {
            return 0;
        }

        $size = 0;
        $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($uploadsPath, RecursiveDirectoryIterator::SKIP_DOTS)
        );

        foreach ($iterator as $file) {
            if ($file->isFile()) {
                $size += $file->getSize();
            }
        }

        return $size;
    }

    /**
     * Скачать бэкап
     */
    public function actionDownload(string $id)
    {
        if (Yii::$app->params['demo']) { // Демо режим
            return [
                'result' => 'error',
                'message' => 'Запрещено в демо режиме'
            ];
        }

        try {
            return $this->backupManager->sendDownloadResponse($id);
        } catch (Exception $e) {
            Yii::$app->response->statusCode = 404;
            return [
                'result' => 'error',
                'message' => $e->getMessage()
            ];
        }
    }

    /**
     * Получить ссылку для скачивания
     */
    public function actionGetDownloadUrl(): array
    {
        if (Yii::$app->params['demo']) { // Демо режим
            return [
                'result' => 'error',
                'message' => 'Запрещено в демо режиме'
            ];
        }

        $id = Yii::$app->request->post('id');

        try {
            $url = $this->backupManager->getDownloadUrl($id);

            return [
                'result' => 'success',
                'url' => $url
            ];
        } catch (Exception $e) {
            return [
                'result' => 'error',
                'message' => $e->getMessage()
            ];
        }
    }

    public function actionSetTask(): array
    {
        if (Yii::$app->params['demo']) { // Демо режим
            return [
                'result' => 'error',
                'message' => 'Запрещено в демо режиме'
            ];
        }

        $taskId = Yii::$app->request->post('taskId');
        $params = Yii::$app->request->post('params');

        if (!in_array($taskId, ['backup-full', 'backup-sql', 'backup-restore'])) {
            return [
                'result' => 'error',
                'message' => 'Задача не найдена',
            ];
        }
        $description = Yii::$app->request->post('description');
        if (!$this->cronMaster->hasImmediateTask($taskId) && !$this->cronMaster->isTaskRunning($taskId)) {
            //$this->cronMaster->addImmediateTask($taskId, $json_string);
            $this->cronMaster->addImmediateTask($taskId, $params);
            return [
                'result' => 'success',
            ];
        }

        return [
            'result' => 'error',
            'message' => 'Задача уже запущена',
        ];
    }

    public function actionIsExecuted(): array
    {
        $taskId = Yii::$app->request->post('taskId');
        return [
            'result' => 'success',
            'has_task' => $this->cronMaster->hasImmediateTask($taskId) || $this->cronMaster->isTaskRunning($taskId)
        ];
    }

    /**
     * Загрузить бэкап с компьютера
     */
    public function actionUpload(): array
    {
        if (Yii::$app->params['demo']) { // Демо режим
            return [
                'result' => 'error',
                'message' => 'Запрещено в демо режиме'
            ];
        }

        if (empty($_FILES['backup'])) {
            return [
                'result' => 'error',
                'message' => 'Файл не загружен'
            ];
        }

        $uploadedFile = $_FILES['backup'];

        // Проверка размера
        $maxSize = 500 * 1024 * 1024; // 500MB
        if ($uploadedFile['size'] > $maxSize) {
            return [
                'result' => 'error',
                'message' => 'Файл слишком большой. Максимальный размер: 500MB'
            ];
        }

        // Проверка типа
//        $allowedTypes = ['application/zip', 'application/x-zip-compressed', 'application/x-gzip'];
//        $fileType = mime_content_type($uploadedFile['tmp_name']);
//
//        if (!in_array($fileType, $allowedTypes) &&
//            !in_array($uploadedFile['type'], $allowedTypes)) {
//            return [
//                'result' => 'error',
//                'message' => 'Неподдерживаемый формат файла. Разрешены только ZIP архивы'
//            ];
//        }

        // Проверка расширения
        $extension = strtolower(pathinfo($uploadedFile['name'], PATHINFO_EXTENSION));
        if (!in_array($extension, ['zip', 'tar', 'gz', 'tgz'])) {
            return [
                'result' => 'error',
                'message' => 'Неподдерживаемое расширение файла'
            ];
        }

        try {
            // Создаем временную папку
            $tempDir = Yii::getAlias('@runtime/temp_upload_' . uniqid());
            if (!file_exists($tempDir)) {
                mkdir($tempDir, 0755, true);
            }

            // Перемещаем файл
            $tempFile = $tempDir . '/' . $uploadedFile['name'];
            if (!move_uploaded_file($uploadedFile['tmp_name'], $tempFile)) {
                throw new Exception('Не удалось сохранить загруженный файл');
            }

            // Проверяем структуру архива
            $isValid = $this->validateBackupArchive($tempFile);
            if (!$isValid) {
                unlink($tempFile);
                FileHelper::removeDirectory($tempDir);
                return [
                    'result' => 'error',
                    'message' => 'Неверный формат бэкапа. Архив должен содержать manifest.json'
                ];
            }

            // Создаем постоянную папку бэкапа
            $backupId = 'backup_uploaded_' . date('Y-m-d_H-i-s') . '_' . uniqid();
            $backupDir = Yii::getAlias($this->backupManager->backupPath) . '/' . $backupId;

            if (!file_exists($backupDir)) {
                mkdir($backupDir, 0755, true);
            }

            // Распаковываем архив
            $this->extractBackupArchive($tempFile, $backupDir);

            // Проверяем манифест
            $manifestFile = $backupDir . '/manifest.json';
            if (!file_exists($manifestFile)) {
                throw new Exception('Не найден файл manifest.json в архиве');
            }

            $manifest = json_decode(file_get_contents($manifestFile), true);
            if (!$manifest) {
                throw new Exception('Неверный формат manifest.json');
            }

            // Обновляем манифест с информацией о загрузке
            $manifest['uploaded_at'] = date('c');
            $manifest['uploaded_by'] = Yii::$app->user->id;
            $manifest['original_filename'] = $uploadedFile['name'];
            $manifest['file_size'] = $uploadedFile['size'];

            file_put_contents($manifestFile,
                json_encode($manifest, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));

            // Очищаем временные файлы
            unlink($tempFile);
            FileHelper::removeDirectory($tempDir);
            //file_put_contents(Yii::getAlias('@runtime/1111111.txt'), '111');

            // Добавляем в список бэкапов
            $backups = $this->backupManager->listBackups();
            $size = $this->backupManager->calculateDirectorySize($backupDir);

            return [
                'result' => 'success',
                'message' => 'Бэкап успешно загружен',
                'data' => [
                    'backup_id' => $backupId,
                    'size' => $size,
                    'size_formatted' => $this->backupManager->formatBytes($size),
                    'manifest' => $manifest,
                    'total_backups' => count($backups)
                ]
            ];

        } catch (Exception $e) {
            // Очистка в случае ошибки
            if (isset($tempFile) && file_exists($tempFile)) {
                unlink($tempFile);
            }
            if (isset($tempDir) && file_exists($tempDir)) {
                FileHelper::removeDirectory($tempDir);
            }

            return [
                'result' => 'error',
                'message' => 'Ошибка загрузки: ' . $e->getMessage()
            ];
        }
    }

    /**
     * Проверить валидность архива бэкапа
     */
    private function validateBackupArchive(string $archivePath): bool
    {
        $zip = new \ZipArchive();

        if ($zip->open($archivePath) !== true) {
            return false;
        }

        // Ищем manifest.json
        for ($i = 0; $i < $zip->numFiles; $i++) {
            $filename = $zip->getNameIndex($i);
            if (basename($filename) === 'manifest.json') {
                $zip->close();
                return true;
            }
        }

        $zip->close();
        return false;
    }

    /**
     * Распаковать архив бэкапа
     */
    private function extractBackupArchive(string $archivePath, string $destination): void
    {
        $zip = new \ZipArchive();

        if ($zip->open($archivePath) !== true) {
            throw new Exception('Не удалось открыть архив');
        }

        // Извлекаем все файлы
        if (!$zip->extractTo($destination)) {
            $zip->close();
            throw new Exception('Не удалось распаковать архив');
        }

        $zip->close();
    }

    /**
     * Получить информацию о загруженном файле перед загрузкой
     */
    public function actionUploadInfo(): array
    {
        $backupPath = Yii::getAlias($this->backupManager->backupPath);
        $freeSpace = disk_free_space(dirname($backupPath));
        $totalSpace = disk_total_space(dirname($backupPath));

        return [
            'result' => 'success',
            'data' => [
                'max_size_mb' => 500,
                'allowed_formats' => ['zip', 'tar.gz', 'tgz'],
                'free_space' => $freeSpace,
                'free_space_formatted' => $this->backupManager->formatBytes($freeSpace),
                'total_space_formatted' => $this->backupManager->formatBytes($totalSpace),
                'used_percent' => round(($totalSpace - $freeSpace) / $totalSpace * 100, 2)
            ]
        ];
    }

    public array $accessRules = [
        'list' => ['admin'],
        'create' => ['admin'],
        'delete' => ['admin'],
        'restore' => ['admin'],
        'system-info' => ['admin'],
        'set-task' => ['admin'],
        'is-executed' => ['admin'],
        'upload' => ['admin'], // Добавить
        'upload-info' => ['admin'], // Добавить
    ];
}
