<?php

namespace app\controllers;

use app\core\entities\Event;
use app\core\entities\Member;
use app\core\forms\event\EventCreateForm;
use app\core\forms\event\EventForm;
use app\core\forms\event\MemberCreateForm;
use app\core\forms\event\MemberForm;
use app\core\forms\event\NotificationSettingsForm;
use app\core\forms\settings\TemplatesForm;
use app\core\helpers\EventHelper;
use app\core\helpers\SerializeHelper;
use app\core\helpers\StringHelper;
use app\core\providers\MapDataProvider;
use app\core\repositories\event\EventRepository;
use app\core\repositories\user\UserRepository;
use app\core\services\event\EventService;
use app\core\services\event\MemberService;
use DomainException;
use Yii;
use yii\web\BadRequestHttpException;
use yii\web\ForbiddenHttpException;
use yii\web\UploadedFile;

class EventController extends ApiController
{
    private EventRepository $eventRepository;
    private EventService $eventService;
    private MemberService $memberService;
    private UserRepository $userRepository;

    public function __construct(
        $id,
        $module,
        EventRepository $eventRepository,
        EventService $eventService,
        MemberService $memberService,
        UserRepository $userRepository,
        $config = []
    ) {
        parent::__construct($id, $module, $config);
        $this->eventRepository = $eventRepository;
        $this->eventService = $eventService;
        $this->memberService = $memberService;
        $this->userRepository = $userRepository;
    }

    public function actionList(): array
    {
        $page = Yii::$app->request->get('page', 1);
        $perPage = Yii::$app->request->get('perPage', 60);
        $filters = Yii::$app->request->get('filter', []);

        $data = $this->eventRepository->getAll($filters, $page, $perPage);

        return [
            'data' => new MapDataProvider($data['provider'], [SerializeHelper::class, 'serializeListEvents']),
            'total' => $data['count'],
            'page' => $page,
            'perPage' => $perPage,
        ];
    }

    public function actionMonth(): array
    {
        $month = Yii::$app->request->get('month', date('m'));
        $year = Yii::$app->request->get('year', date('Y'));

        // Определяем предыдущий и следующий месяцы
        $prevMonth = $month - 1;
        $prevYear = $year;

        $nextMonth = $month + 1;
        $nextYear = $year;

        // Корректируем год если месяц выходит за границы
        if ($prevMonth < 1) {
            $prevMonth = 12;
            $prevYear = $year - 1;
        }

        if ($nextMonth > 12) {
            $nextMonth = 1;
            $nextYear = $year + 1;
        }

        // Получаем мероприятия за три месяца
        $currentMonthEvents = $this->eventRepository->getByMonth($month, $year);
        $prevMonthEvents = $this->eventRepository->getByMonth($prevMonth, $prevYear);
        $nextMonthEvents = $this->eventRepository->getByMonth($nextMonth, $nextYear);

        // Объединяем все мероприятия
        $allEvents = array_merge($currentMonthEvents, $prevMonthEvents, $nextMonthEvents);

        // Убираем дубликаты (на случай если событие попало в несколько месяцев)
        $uniqueEvents = [];
        $eventIds = [];

        foreach ($allEvents as $event) {
            if (!in_array($event->id, $eventIds)) {
                $eventIds[] = $event->id;
                $uniqueEvents[] = $event;
            }
        }

        return [
            'result' => 'success',
            'events' => array_map(function (Event $event) {
                return SerializeHelper::serializeEvent($event);
            }, $uniqueEvents),
        ];
    }

    public function actionView(string $id): array
    {
        $event = $this->eventRepository->get($id);

        return [
            'result' => 'success',
            'event' => SerializeHelper::serializeEvent($event),
        ];
    }

    public function actionViewBySlug(): array
    {
        $slug = Yii::$app->request->get('slug');
        $event = $this->eventRepository->getBySlug($slug);

        return [
            'result' => 'success',
            'event' => SerializeHelper::serializeEvent($event),
        ];
    }

    public function actionCreate(): array
    {
        if (Yii::$app->params['demo']) { // Демо режим
            return [
                'result' => 'error',
                'message' => 'Запрещено в демо режиме'
            ];
        }

        $form = new EventCreateForm();

        if ($form->load(Yii::$app->request->post(), '') && $form->validate()) {
            $event = $this->eventService->create($form);

            return [
                'result' => 'success',
                'event' => SerializeHelper::serializeEvent($event),
                'message' => 'Мероприятие успешно создано',
            ];
        }

        return [
            'result' => 'error',
            'errors' => $form->errors,
        ];
    }

    public function actionUpdate(): array
    {
        if (Yii::$app->params['demo']) { // Демо режим
            return [
                'result' => 'error',
                'message' => 'Запрещено в демо режиме'
            ];
        }

        $id = Yii::$app->request->post('id');
        $event = $this->eventRepository->get($id);
        $form = new EventForm($event);

        if ($form->load(Yii::$app->request->post(), '') && $form->validate()) {
            $this->eventService->edit($id, $form);

            return [
                'result' => 'success',
                'message' => 'Мероприятие успешно обновлено',
            ];
        }

        return [
            'result' => 'error',
            'errors' => $form->errors,
        ];
    }

    public function actionPublish(): array
    {
        if (Yii::$app->params['demo']) { // Демо режим
            return [
                'result' => 'error',
                'message' => 'Запрещено в демо режиме'
            ];
        }

        $id = Yii::$app->request->post('id');
        $this->eventService->publish($id);

        return [
            'result' => 'success',
            'message' => 'Мероприятие опубликовано',
        ];
    }

    public function actionDraft(): array
    {
        if (Yii::$app->params['demo']) { // Демо режим
            return [
                'result' => 'error',
                'message' => 'Запрещено в демо режиме'
            ];
        }

        $id = Yii::$app->request->post('id');
        $this->eventService->draft($id);

        return [
            'result' => 'success',
            'message' => 'Публикация отменена',
        ];
    }

    public function actionCancel(): array
    {
        if (Yii::$app->params['demo']) { // Демо режим
            return [
                'result' => 'error',
                'message' => 'Запрещено в демо режиме'
            ];
        }

        $id = Yii::$app->request->post('id');
        $this->eventService->cancel($id);

        return [
            'result' => 'success',
            'message' => 'Мероприятие отменено',
        ];
    }

    public function actionArchive(): array
    {
        if (Yii::$app->params['demo']) { // Демо режим
            return [
                'result' => 'error',
                'message' => 'Запрещено в демо режиме'
            ];
        }

        $id = Yii::$app->request->post('id');
        $this->eventService->archive($id);

        return [
            'result' => 'success',
            'message' => 'Мероприятие перемещено в архив',
        ];
    }

    public function actionDelete(): array
    {
        if (Yii::$app->params['demo']) { // Демо режим
            return [
                'result' => 'error',
                'message' => 'Запрещено в демо режиме'
            ];
        }

        $id = Yii::$app->request->post('id');
        $this->eventService->remove($id);

        return [
            'result' => 'success',
            'message' => 'Мероприятие удалено',
        ];
    }

    public function actionUploadImage(): array
    {
        if (Yii::$app->params['demo']) { // Демо режим
            return [
                'result' => 'error',
                'message' => 'Запрещено в демо режиме'
            ];
        }

        $id = Yii::$app->request->post('id');
        $event = $this->eventRepository->get($id);
        $image = UploadedFile::getInstanceByName('image');

        if (!$image) {
            throw new BadRequestHttpException('Файл изображения не загружен');
        }

        // Проверка типа файла
        $allowedTypes = ['image/jpeg', 'image/png', 'image/gif', 'image/webp'];
        if (!in_array($image->type, $allowedTypes)) {
            return [
                'result' => 'error',
                'message' => 'Допустимые форматы: JPEG, PNG, GIF, WebP',
            ];
        }

        // Проверка размера (макс 10MB)
        if ($image->size > 10 * 1024 * 1024) {
            return [
                'result' => 'error',
                'message' => 'Размер файла не должен превышать 10 МБ',
            ];
        }

        try {
            $this->eventService->setImage($event, $image);

            return [
                'result' => 'success',
                'image' => $event->image,
                'message' => 'Изображение успешно загружено',
            ];
        } catch (\Exception $e) {
            return [
                'result' => 'error',
                'message' => $e->getMessage(),
            ];
        }
    }

    public function actionRemoveImage(): array
    {
        if (Yii::$app->params['demo']) { // Демо режим
            return [
                'result' => 'error',
                'message' => 'Запрещено в демо режиме'
            ];
        }

        $id = Yii::$app->request->post('id');
        $event = $this->eventRepository->get($id);
        $this->eventService->removeImage($event);

        return [
            'result' => 'success',
            'message' => 'Изображение удалено',
        ];
    }

    public function actionUploadFile(): array
    {
        if (Yii::$app->params['demo']) { // Демо режим
            return [
                'result' => 'error',
                'message' => 'Запрещено в демо режиме'
            ];
        }

        $id = Yii::$app->request->post('id');
        $event = $this->eventRepository->get($id);
        $file = UploadedFile::getInstanceByName('file');

        if (!$file) {
            throw new BadRequestHttpException('Файл не загружен');
        }

        // Проверка типа файла
        $allowedTypes = ['text/csv', 'application/csv'];
        if (!in_array($file->type, $allowedTypes)) {
            return [
                'result' => 'error',
                'message' => 'Допустимые форматы: CSV, XML',
            ];
        }

        // Проверка размера (макс 10MB)
        if ($file->size > 10 * 1024 * 1024) {
            return [
                'result' => 'error',
                'message' => 'Размер файла не должен превышать 10 МБ',
            ];
        }

        try {
            list($path, $headers) = $this->eventService->saveImportFile($event, $file);

            return [
                'result' => 'success',
                //'image' => $event->image,
                'headers' => $headers,
                'file_id' => basename($path),
                'message' => 'Файл успешно загружен',
            ];
        } catch (\Exception $e) {
            return [
                'result' => 'error',
                'message' => $e->getMessage(),
            ];
        }
    }

    public function actionUpcoming(): array
    {
        $limit = Yii::$app->request->get('limit', 10);
        $events = $this->eventRepository->getUpcomingEvents($limit);

        return [
            'result' => 'success',
            'events' => array_map([SerializeHelper::class, 'serializeEvent'], $events),
        ];
    }

    public function actionPast(): array
    {
        $limit = Yii::$app->request->get('limit', 10);
        $events = $this->eventRepository->getPastEvents($limit);

        return [
            'result' => 'success',
            'events' => array_map([SerializeHelper::class, 'serializeEvent'], $events),
        ];
    }

    public function actionStats(): array
    {
        $stats = [
            'total' => Event::find()->count(),
            'draft' => $this->eventRepository->countByStatus(Event::STATUS_DRAFT),
            'pending' => $this->eventRepository->countByStatus(Event::STATUS_PENDING),
            'published' => $this->eventRepository->countByStatus(Event::STATUS_PUBLISHED),
            'cancelled' => $this->eventRepository->countByStatus(Event::STATUS_CANCELLED),
            'archived' => $this->eventRepository->countByStatus(Event::STATUS_ARCHIVED),
            'active' => $this->eventRepository->countActive(),
            'activeTrend' => $this->eventRepository->activeTrend(),
            'upcoming' => $this->eventRepository->upcoming(),
        ];

        return [
            'result' => 'success',
            'stats' => $stats,
        ];
    }

    public function actionDeleteAll(): array
    {
        if (Yii::$app->params['demo']) { // Демо режим
            return [
                'result' => 'error',
                'message' => 'Запрещено в демо режиме'
            ];
        }

        $ids = Yii::$app->request->post('ids');
        $deleted = 0;
        foreach ($ids as $id) {
            $this->eventService->remove($id);
            $deleted++;
        }
        return [
            'result' => 'success',
            'deleted' => $deleted,
        ];
    }

    /**
     * Получение полей формы мероприятия
     */
    public function actionFields(): array
    {
        $id = Yii::$app->request->get('id');
        $event = $this->eventRepository->get($id);

        $default_fields = [
            ['key' => 'first_name', 'label' => 'Имя', 'required' => true],
            ['key' => 'last_name', 'label' => 'Фамилия', 'required' => true],
            ['key' => 'middle_name', 'label' => 'Отчество', 'required' => false],
            ['key' => 'email', 'label' => 'Email', 'required' => false],
            ['key' => 'phone', 'label' => 'Телефон', 'required' => false],
            ['key' => 'company', 'label' => 'Компания', 'required' => false],
            ['key' => 'position', 'label' => 'Должность', 'required' => false],
        ];

        return [
            'result' => 'success',
            'fields' => array_merge(
                $default_fields,
                array_map([SerializeHelper::class, 'serializeEventField'], $event->fields)
            ),
        ];
    }

    public function actionTemplates(): array
    {
        if (Yii::$app->params['demo']) { // Демо режим
            return [
                'result' => 'error',
                'message' => 'Запрещено в демо режиме'
            ];
        }

        $id = Yii::$app->request->post('id');
        $event = $this->eventRepository->get($id);

        $form = new TemplatesForm();

        if ($form->load(Yii::$app->request->post(), '') && $form->validate()) {

            $this->eventService->setSetting($event->id, 'tpl.after', $form->after);
            $this->eventService->setSetting($event->id, 'tpl.before', $form->before);
            $this->eventService->setSetting($event->id, 'tpl.result', $form->result);
            $this->eventService->setSetting($event->id, 'tpl.mail.notify', $form->notify);
        }

        return [
            'result' => 'success'
        ];
    }

    /**
     * Получение настроек уведомлений
     */
    public function actionNotificationSettings(string $id): array
    {
        $event = $this->eventRepository->get($id);
        //$this->checkEventAccess($event);

        $form = new NotificationSettingsForm();
        $form->notify_emails = $event->getNotifyEmailsArray();
        $form->auto_confirm = (bool)$event->auto_confirm;
        $form->success_message = $event->success_message;

        return [
            'result' => 'success',
            'settings' => $form->attributes,
        ];
    }

    public function actionImport(): array
    {
        if (Yii::$app->params['demo']) { // Демо режим
            return [
                'result' => 'error',
                'message' => 'Запрещено в демо режиме'
            ];
        }

        $id = Yii::$app->request->post('id');
        $event = $this->eventRepository->get($id);

        $request = Yii::$app->request;
        $mapping = $request->post('mapping', []);
        $fileId = $request->post('file_id');
        $notify = Yii::$app->request->post('notify');
        $notify = filter_var($notify, FILTER_VALIDATE_BOOLEAN, FILTER_NULL_ON_FAILURE) ?? false;

        if (empty($mapping)) {
            return [
                'result' => 'error',
                'message' => 'Не указано сопоставление полей',
            ];
        }

        // ПРОВЕРКА БЕЗОПАСНОСТИ: валидируем имя файла
        if (empty($fileId) || !$this->validateFileName($fileId)) {
            return [
                'result' => 'error',
                'message' => 'Неверный идентификатор файла',
            ];
        }

        $tempPath = Yii::getAlias('@webroot/events/' . $fileId);

        $runtimePath = realpath(Yii::getAlias('@webroot'));
        $realFilePath = realpath($tempPath);

        if (!$realFilePath || !str_starts_with($realFilePath, $runtimePath)) {
            return [
                'result' => 'error',
                'message' => 'Доступ к файлу запрещен',
            ];
        }

        if (!file_exists($tempPath)) {
            return [
                'result' => 'error',
                'message' => 'Файл не найден',
            ];
        }

        $extension = pathinfo($tempPath, PATHINFO_EXTENSION);
        $data = $this->extractData($tempPath, $extension);

        // Импортируем данные
        //print_r($data); die;
        $result = $this->importData($event, $data, $mapping, $notify);

        // Удаляем временный файл
        unlink($tempPath);

        return [
            'result' => 'success',
            'stats' => $result
        ];
    }

    private function importData(Event $event, $data, $mapping, $notify = false): array
    {
        $imported = 0;
        $duplicates = 0;
        $errors_fields = 0;
        $errors = 0;
        $messages = [];

        foreach ($data as $row) {
            // Преобразуем строку по маппингу
            $attributes = [];
            foreach ($mapping as $field => $column) {
                if ($column && isset($row[$column])) {
                    $attributes[$field] = trim($row[$column]);
                }
            }

            // Пропускаем строки без обязательных полей
            //if (empty($attributes['first_name']) || empty($attributes['last_name'])) {
            //    $errors++;
            //    continue;
            //}

            // Проверка на дубликаты
            $duplicateCondition = ['or'];

            // 1. Тот же email
            if (isset($attributes['email'])) {
                $duplicateCondition[] = ['email' => $attributes['email']];
            }

            // 2. Тот же телефон (если есть)
            if (isset($attributes['phone'])) {
                if (!empty($attributes['phone'])) {
                    $duplicateCondition[] = ['phone' => $attributes['phone']];
                }
            }

            // 3. Совпадение имени, компании и должности
            $duplicateCondition[] = [
                'and',
                ['first_name' => $attributes['first_name']],
                ['last_name' => $attributes['last_name']],
                ['company' => $attributes['company'] ?? ''],
                ['position' => $attributes['position'] ?? '']
            ];

            $exists = Member::find()
                ->where($duplicateCondition)
                ->andWhere(['event_id' => $event->id])
                ->exists();

            if ($exists) {
                $duplicates++;
                continue;
            }

            $form = new MemberCreateForm();
            $form->attributes = $attributes;
            $form->event_id = $event->id;
            if ($form->validate()) {
                try {
                    $formData = $form->getFormData();
                    $this->memberService->register(
                        $event->id,
                        $form->first_name,
                        $form->last_name,
                        $form->email,
                        $form->phone,
                        $form->middle_name,
                        $form->company,
                        $form->position,
                        $form->city,
                        $form->country,
                        $formData,
                        $notify
                    );
                    $imported++;
                } catch (DomainException $e) {
                    $errors++;
                    $messages[] = $e->getMessage();
                }
            } else {
                $errors_fields++;
                $messages = [... $messages, $form->errors];
            }

            // Сохраняем нового участника
            //$participant = new Member();
            //$participant->attributes = $attributes;
            //$participant->created_at = time();
//            if ($participant->save()) {
//                $imported++;
//            } else {
//                $errors++;
//                Yii::error('Ошибка импорта: ' . print_r($participant->errors, true));
//            }
        }

        return [
            'imported' => $imported,
            'duplicates' => $duplicates,
            'errors' => $errors,
            'errors_fields' => $errors_fields,
            'total' => count($data),
            'messages' => $messages,
        ];
    }

    private function extractData($filePath, $extension): array
    {
        $data = [];

        if ($extension === 'csv') {
            $delimiter = StringHelper::detectCsvDelimiter($filePath);

            if (($handle = fopen($filePath, "r")) !== FALSE) {
                // Пропускаем заголовки
                $headers = fgetcsv($handle, 1000, $delimiter);

                while (($row = fgetcsv($handle, 1000, $delimiter)) !== FALSE) {
                    $data[] = array_combine($headers, $row);
                    //$data[] = $row;
                }
                fclose($handle);
            }
        }
//        elseif ($extension === 'xml') {
//            $xml = simplexml_load_file($filePath);
//            foreach ($xml->participant ?? $xml->row ?? [] as $item) {
//                $row = [];
//                foreach ($item->children() as $child) {
//                    $row[$child->getName()] = (string)$child;
//                }
//                $data[] = $row;
//            }
//        }

        return $data;
    }

    private function validateFileName($fileName): bool
    {
        return !str_contains($fileName, '..') &&
            !str_contains($fileName, '/') &&
            !str_contains($fileName, '\\') &&
            !str_contains($fileName, "\0"); // null-byte injection
    }

    /**
     * Обновление настроек уведомлений
     */
    public function actionUpdateNotificationSettings(string $id): array
    {
        $event = $this->eventRepository->get($id);
        //$this->checkEventAccess($event);

        $form = new NotificationSettingsForm();

        if ($form->load(Yii::$app->request->post(), '') && $form->validate()) {
            $this->eventService->updateNotificationSettings($id, $form->attributes);

            return [
                'result' => 'success',
                'message' => 'Настройки сохранены',
            ];
        }

        return [
            'result' => 'error',
            'errors' => $form->errors,
        ];
    }

    /**
     * Проверка доступа к мероприятию
     */
    private function checkEventAccess(Event $event): void
    {
        // Для публичных действий не проверяем
        $publicActions = ['view', 'view-by-slug', 'upcoming', 'past', 'fields', 'registration-info'];
        if (in_array(Yii::$app->requestedAction->id, $publicActions)) {
            return;
        }

        // Для остальных действий нужны права админа/менеджера
        if (!Yii::$app->user->can('admin') && !Yii::$app->user->can('manager')) {
            throw new ForbiddenHttpException('Недостаточно прав');
        }
    }

    public function actionSubscribe(string $id, string $uid): array
    {
        if ($this->eventService->subscribe($id, $uid)) {
            return ['result' => 'success'];
        }
        return ['result' => 'error'];
    }

    public function actionUnSubscribe(string $id, string $uid): array
    {
        if ($this->eventService->unSubscribe($id, $uid)) {
            return ['result' => 'success'];
        }
        return ['result' => 'error'];
    }

    public array $accessRules = [
        'list' => [], // все авторизованные
        'month' => [],
        'view' => [],
        'view-by-slug' => [],
        'upcoming' => [],
        'past' => [],
        'stats' => [],

        'create' => ['admin', 'manager'],
        'update' => ['admin', 'manager'],
        'publish' => ['admin', 'manager'],
        'draft' => ['admin', 'manager'],
        'cancel' => ['admin', 'manager'],
        'archive' => ['admin', 'manager'],
        'delete' => ['admin', 'manager'],
        'delete-all' => ['admin', 'manager'],
        'upload-image' => ['admin', 'manager'],
        'remove-image' => ['admin', 'manager'],
        'upload-file' => ['admin', 'manager'],
        'import' => ['admin', 'manager'],
        'templates' => ['admin', 'manager'],

        'fields' => [],
        'subscribe' => ['admin', 'manager'],
        'un-subscribe' => ['admin', 'manager'],
        'notification-settings' => ['admin', 'manager'],
        'update-notification-settings' => ['admin', 'manager'],
    ];
}
