<?php

namespace app\controllers;

use app\core\entities\Event;
use app\core\forms\event\EventFieldForm;
use app\core\helpers\SerializeHelper;
use app\core\repositories\event\EventFieldRepository;
use app\core\repositories\event\EventRepository;
use app\core\services\event\EventFieldService;
use Yii;
use yii\web\ForbiddenHttpException;

class EventFieldController extends ApiController
{
    private EventRepository $eventRepository;
    private EventFieldRepository $fieldRepository;
    private EventFieldService $fieldService;

    public function __construct(
        $id,
        $module,
        EventRepository $eventRepository,
        EventFieldRepository $fieldRepository,
        EventFieldService $fieldService,
        $config = []
    ) {
        parent::__construct($id, $module, $config);
        $this->eventRepository = $eventRepository;
        $this->fieldRepository = $fieldRepository;
        $this->fieldService = $fieldService;
    }

    /**
     * Список полей мероприятия
     */
    public function actionList(string $eventId): array
    {
        $event = $this->eventRepository->get($eventId);
        $this->checkEventAccess($event);

        $fields = $this->fieldRepository->getByEvent($eventId);

        return [
            'result' => 'success',
            'fields' => array_map([SerializeHelper::class, 'serializeEventField'], $fields),
        ];
    }

    /**
     * Просмотр поля
     */
    public function actionView(string $id): array
    {
        $field = $this->fieldRepository->get($id);
        $this->checkEventAccess($field->event);

        return [
            'result' => 'success',
            'field' => SerializeHelper::serializeEventField($field),
        ];
    }

    /**
     * Создание поля
     */
    public function actionCreate(): array
    {
        if (Yii::$app->params['demo']) { // Демо режим
            return [
                'result' => 'error',
                'message' => 'Запрещено в демо режиме'
            ];
        }

        $id = Yii::$app->request->post('id');
        $event = $this->eventRepository->get($id);
        $this->checkEventAccess($event);

        $form = new EventFieldForm();

        if ($form->load(Yii::$app->request->post(), '') && $form->validate()) {
            try {
                $field = $this->fieldService->create($id, $form);

                return [
                    'result' => 'success',
                    'field' => SerializeHelper::serializeEventField($field),
                    'message' => 'Поле создано',
                ];
            } catch (\DomainException $e) {
                return [
                    'result' => 'error',
                    'message' => $e->getMessage(),
                ];
            }
        }

        return [
            'result' => 'error',
            'errors' => $form->errors,
        ];
    }

    /**
     * Редактирование поля
     */
    public function actionUpdate(): array
    {
        if (Yii::$app->params['demo']) { // Демо режим
            return [
                'result' => 'error',
                'message' => 'Запрещено в демо режиме'
            ];
        }

        $id = Yii::$app->request->post('id');
        $field = $this->fieldRepository->get($id);
        $this->checkEventAccess($field->event);

        $form = new EventFieldForm($field);

        if ($form->load(Yii::$app->request->post(), '') && $form->validate()) {
            try {
                $this->fieldService->edit($id, $form);

                return [
                    'result' => 'success',
                    'message' => 'Поле обновлено',
                ];
            } catch (\DomainException $e) {
                return [
                    'result' => 'error',
                    'message' => $e->getMessage(),
                ];
            }
        }

        return [
            'result' => 'error',
            'errors' => $form->errors,
        ];
    }

    /**
     * Удаление поля
     */
    public function actionDelete(): array
    {
        if (Yii::$app->params['demo']) { // Демо режим
            return [
                'result' => 'error',
                'message' => 'Запрещено в демо режиме'
            ];
        }

        $id = Yii::$app->request->post('id');
        $field = $this->fieldRepository->get($id);
        $this->checkEventAccess($field->event);

        $this->fieldService->remove($id);

        return [
            'result' => 'success',
            'message' => 'Поле удалено',
        ];
    }

    /**
     * Изменение порядка полей
     */
    public function actionReorder(string $eventId): array
    {
        if (Yii::$app->params['demo']) { // Демо режим
            return [
                'result' => 'error',
                'message' => 'Запрещено в демо режиме'
            ];
        }

        $event = $this->eventRepository->get($eventId);
        $this->checkEventAccess($event);

        $fieldIds = Yii::$app->request->post('fieldIds', []);

        if (empty($fieldIds)) {
            return [
                'result' => 'error',
                'message' => 'Не указаны ID полей',
            ];
        }

        $this->fieldService->reorder($eventId, $fieldIds);

        return [
            'result' => 'success',
            'message' => 'Порядок сохранен',
        ];
    }

    /**
     * Создание стандартных полей
     */
    public function actionCreateDefault(string $eventId): array
    {
        if (Yii::$app->params['demo']) { // Демо режим
            return [
                'result' => 'error',
                'message' => 'Запрещено в демо режиме'
            ];
        }

        $event = $this->eventRepository->get($eventId);
        $this->checkEventAccess($event);

        $this->fieldService->createDefaultFields($eventId);

        return [
            'result' => 'success',
            'message' => 'Стандартные поля созданы',
        ];
    }

    /**
     * Копирование полей из другого мероприятия
     */
    public function actionCopy(string $eventId): array
    {
        if (Yii::$app->params['demo']) { // Демо режим
            return [
                'result' => 'error',
                'message' => 'Запрещено в демо режиме'
            ];
        }

        $event = $this->eventRepository->get($eventId);
        $this->checkEventAccess($event);

        $sourceEventId = Yii::$app->request->post('sourceEventId');

        if (!$sourceEventId) {
            return [
                'result' => 'error',
                'message' => 'Не указан ID исходного мероприятия',
            ];
        }

        try {
            $this->fieldService->copyFields($sourceEventId, $eventId);

            return [
                'result' => 'success',
                'message' => 'Поля скопированы',
            ];
        } catch (\DomainException $e) {
            return [
                'result' => 'error',
                'message' => $e->getMessage(),
            ];
        }
    }

    /**
     * Типы полей
     */
    public function actionTypes(): array
    {
        return [
            'result' => 'success',
            'types' => \app\core\entities\EventField::getTypes(),
        ];
    }

    /**
     * Проверка доступа к мероприятию
     */
    private function checkEventAccess(Event $event): void
    {
        if (!Yii::$app->user->can('admin') && !Yii::$app->user->can('manager')) {
            throw new ForbiddenHttpException('Недостаточно прав');
        }
    }

    public array $accessRules = [
        'list' => ['admin', 'manager'],
        'view' => ['admin', 'manager'],
        'create' => ['admin', 'manager'],
        'update' => ['admin', 'manager'],
        'delete' => ['admin', 'manager'],
        'reorder' => ['admin', 'manager'],
        'create-default' => ['admin', 'manager'],
        'copy' => ['admin', 'manager'],
        'types' => ['admin', 'manager'],
    ];
}