<?php

namespace app\controllers;

use app\core\entities\Member;
use app\core\forms\event\MemberCreateForm;
use app\core\forms\event\MemberForm;
use app\core\forms\event\MemberSearchForm;
use app\core\helpers\MailHelper;
use app\core\helpers\SerializeHelper;
use app\core\helpers\SettingsHelper;
use app\core\providers\MapDataProvider;
use app\core\repositories\event\EventRepository;
use app\core\repositories\event\MemberRepository;
use app\core\services\event\EventService;
use app\core\services\event\MemberService;
use app\core\services\MailTransportFactory;
use DomainException;
use Exception;
use Yii;
use yii\mail\MailerInterface;
use yii\web\UploadedFile;

class MemberController extends ApiController
{
    private MemberRepository $memberRepository;
    private EventRepository $eventRepository;
    private MemberService $memberService;
    private EventService $eventService;
    private MailerInterface $mailer;

    public function __construct(
        $id,
        $module,
        MemberRepository $memberRepository,
        EventRepository $eventRepository,
        MemberService $memberService,
        EventService $eventService,
        MailerInterface $mailer,
        $config = []
    ) {
        parent::__construct($id, $module, $config);
        $this->memberRepository = $memberRepository;
        $this->eventRepository = $eventRepository;
        $this->memberService = $memberService;
        $this->eventService = $eventService;
        $this->mailer = $mailer;
    }

    /**
     * Список всех участников (с фильтрами)
     */
    public function actionList(): array
    {
        $page = Yii::$app->request->get('page', 1);
        $perPage = Yii::$app->request->get('perPage', 60);

        $searchForm = new MemberSearchForm();
        $searchForm->load(Yii::$app->request->get(), 'filter');

        $filters = $searchForm->getFilters();
        $data = $this->memberRepository->getAll($filters, $page, $perPage);

        return [
            'data' => new MapDataProvider($data['provider'], [SerializeHelper::class, 'serializeMember']),
            'total' => $data['count'],
            'page' => $page,
            'perPage' => $perPage,
            'filters' => $searchForm->attributes,
        ];
    }

    /**
     * Список участников конкретного мероприятия
     */
    public function actionEventList(): array
    {
        $eventId = Yii::$app->request->get('eventId');
        $event = $this->eventRepository->get($eventId);
        //$this->checkEventAccess($event);

        $page = Yii::$app->request->get('page', 1);
        $perPage = Yii::$app->request->get('perPage', 60);

        $searchForm = new MemberSearchForm();
        $searchForm->load(Yii::$app->request->get(), '');
        $searchForm->event_id = $eventId;

        $filters = $searchForm->getFilters();
        $data = $this->memberRepository->getByEvent($eventId, $filters, $page, $perPage);

        return [
            'data' => new MapDataProvider($data['provider'], [SerializeHelper::class, 'serializeMember']),
            'total' => $data['count'],
            'page' => $page,
            'perPage' => $perPage,
            'stats' => $this->memberService->getStats($eventId),
        ];
    }

    /**
     * Просмотр участника
     */
    public function actionView(string $id): array
    {
        $member = $this->memberRepository->get($id);
        //$this->checkEventAccess($member->event);

        return [
            'result' => 'success',
            'member' => SerializeHelper::serializeMember($member),
        ];
    }

    /**
     * Регистрация участника (публичная)
     */
    public function actionRegister(): array
    {
        if (Yii::$app->params['demo']) { // Демо режим
            return [
                'result' => 'error',
                'message' => 'Запрещено в демо режиме'
            ];
        }

        $form = new MemberCreateForm();

        if ($form->load(Yii::$app->request->post(), '') && $form->validate()) {
            try {
                $formData = $form->getFormData();

                $event = $this->eventRepository->get($form->event_id);

                $member = $this->memberService->register(
                    $form->event_id,
                    $form->first_name,
                    $form->last_name,
                    $form->email,
                    $form->phone,
                    $form->middle_name,
                    $form->company,
                    $form->position,
                    $form->city,
                    $form->country,
                    $formData,
                    true
                );

                return [
                    'result' => 'success',
                    'member' => SerializeHelper::serializeMember($member),
                    'message' => $member->isPending()
                        ? 'Заявка отправлена на модерацию'
                        : 'Регистрация подтверждена',
                ];
            } catch (DomainException $e) {
                return [
                    'result' => 'error',
                    'message' => $e->getMessage(),
                ];
            }
        }

        return [
            'result' => 'error',
            'errors' => $form->errors,
        ];
    }

    /**
     * Регистрация участника администратором
     */
    public function actionCreate(): array
    {
        if (Yii::$app->params['demo']) { // Демо режим
            return [
                'result' => 'error',
                'message' => 'Запрещено в демо режиме'
            ];
        }

        $eventId = Yii::$app->request->post('event_id');
        $event = $this->eventRepository->get($eventId);
        //$this->checkEventAccess($event);

        $form = new MemberCreateForm($event);

        if ($form->load(Yii::$app->request->post(), '') && $form->validate()) {
            try {
                $member = $this->memberService->registerByAdmin(
                    $eventId,
                    $form->first_name,
                    $form->last_name,
                    $form->email,
                    $form->phone,
                    $form->middle_name,
                    $form->company,
                    $form->position,
                    $form->city,
                    $form->country,
                    Yii::$app->request->post('notes')
                );

                return [
                    'result' => 'success',
                    'member' => SerializeHelper::serializeMember($member),
                    'message' => 'Участник зарегистрирован',
                ];
            } catch (DomainException $e) {
                return [
                    'result' => 'error',
                    'message' => $e->getMessage(),
                ];
            }
        }

        return [
            'result' => 'error',
            'errors' => $form->errors,
        ];
    }

    /**
     * Редактирование участника
     */
    public function actionUpdate(): array
    {
        if (Yii::$app->params['demo']) { // Демо режим
            return [
                'result' => 'error',
                'message' => 'Запрещено в демо режиме'
            ];
        }

        $id = Yii::$app->request->post('id');
        $member = $this->memberRepository->get($id);

        $form = new MemberForm($member);

        if ($form->load(Yii::$app->request->post(), '') && $form->validate()) {
            $this->memberService->edit($id, $form);

            return [
                'result' => 'success',
                'message' => 'Участник обновлен',
            ];
        }

        return [
            'result' => 'error',
            'errors' => $form->errors,
        ];
    }

    /**
     * Удаление участника
     */
    public function actionDelete(): array
    {
        if (Yii::$app->params['demo']) { // Демо режим
            return [
                'result' => 'error',
                'message' => 'Запрещено в демо режиме'
            ];
        }

        $id = Yii::$app->request->post('id');

        $this->memberService->remove($id);

        return [
            'result' => 'success',
            'message' => 'Участник удален',
        ];
    }

    /**
     * Подтверждение регистрации
     */
    public function actionConfirm(): array
    {
        if (Yii::$app->params['demo']) { // Демо режим
            return [
                'result' => 'error',
                'message' => 'Запрещено в демо режиме'
            ];
        }

        $id = Yii::$app->request->post('id');

        $this->memberService->confirm($id);

        return [
            'result' => 'success',
            'message' => 'Регистрация подтверждена',
        ];
    }

    /**
     * Отмена регистрации
     */
    public function actionCancel(): array
    {
        if (Yii::$app->params['demo']) { // Демо режим
            return [
                'result' => 'error',
                'message' => 'Запрещено в демо режиме'
            ];
        }

        $id = Yii::$app->request->post('id');

        $this->memberService->cancel($id);

        return [
            'result' => 'success',
            'message' => 'Регистрация отменена',
        ];
    }

    /**
     * Отметка присутствия
     */
    public function actionAttend(): array
    {
        if (Yii::$app->params['demo']) { // Демо режим
            return [
                'result' => 'error',
                'message' => 'Запрещено в демо режиме'
            ];
        }

        $id = Yii::$app->request->post('id');

        $this->memberService->markAttended($id);

        return [
            'result' => 'success',
            'message' => 'Участник отмечен как присутствовавший',
        ];
    }

    public function actionUnAttend(): array
    {
        if (Yii::$app->params['demo']) { // Демо режим
            return [
                'result' => 'error',
                'message' => 'Запрещено в демо режиме'
            ];
        }

        $id = Yii::$app->request->post('id');

        $this->memberService->unmarkAttended($id);

        return [
            'result' => 'success',
            'message' => 'Отметка посещения снята',
        ];
    }

    /**
     * Check-in по коду
     */
    public function actionCheckin(): array
    {
        if (Yii::$app->params['demo']) { // Демо режим
            return [
                'result' => 'error',
                'message' => 'Запрещено в демо режиме'
            ];
        }

        $id = Yii::$app->request->post('id');
        try {
            $this->memberService->checkin($id);

            return [
                'result' => 'success',
                'message' => 'Участник отмечен как присутствовавший',
            ];
        } catch (DomainException $e) {
            return [
                'result' => 'error',
                'message' => $e->getMessage(),
            ];
        }
    }

    /**
     * Загрузка фото
     */
    public function actionUploadPhoto(): array
    {
        if (Yii::$app->params['demo']) { // Демо режим
            return [
                'result' => 'error',
                'message' => 'Запрещено в демо режиме'
            ];
        }

        $id = Yii::$app->request->post('id');

        $file = UploadedFile::getInstanceByName('file');

        if (!$file) {
            return [
                'result' => 'error',
                'message' => 'Файл не загружен',
            ];
        }

        // Проверка типа файла
        $allowedTypes = ['image/jpeg', 'image/png', 'image/gif', 'image/webp'];
        if (!in_array($file->type, $allowedTypes)) {
            return [
                'result' => 'error',
                'message' => 'Допустимые форматы: JPEG, PNG, GIF, WebP',
            ];
        }

        // Проверка размера (макс 5MB)
        if ($file->size > 5 * 1024 * 1024) {
            return [
                'result' => 'error',
                'message' => 'Размер файла не должен превышать 5 МБ',
            ];
        }

        try {
            $this->memberService->setPhoto($id, $file);
            $member = $this->memberRepository->get($id);

            return [
                'result' => 'success',
                'photo' => $member->photo,
                'message' => 'Фото загружено',
            ];
        } catch (Exception $e) {
            return [
                'result' => 'error',
                'message' => $e->getMessage(),
            ];
        }
    }

    /**
     * Удаление фото
     */
    public function actionRemovePhoto(): array
    {
        if (Yii::$app->params['demo']) { // Демо режим
            return [
                'result' => 'error',
                'message' => 'Запрещено в демо режиме'
            ];
        }

        $id = Yii::$app->request->post('id');

        $this->memberService->removePhoto($id);

        return [
            'result' => 'success',
            'message' => 'Фото удалено',
        ];
    }

    /**
     * Статистика по мероприятию
     */
    public function actionStats(string $eventId): array
    {
        $event = $this->eventRepository->get($eventId);
        //$this->checkEventAccess($event);

        $stats = $this->memberService->getStats($eventId);

        return [
            'result' => 'success',
            'stats' => $stats,
            'max_participants' => $event->max_participants,
        ];
    }

    /**
     * Экспорт участников
     */
    public function actionExport(): array
    {
        if (Yii::$app->params['demo']) { // Демо режим
            return [
                'result' => 'error',
                'message' => 'Запрещено в демо режиме'
            ];
        }

        $id     = Yii::$app->request->get('id');
        $format = Yii::$app->request->get('format', 'csv');

        try {
            $filePath = $this->memberService->export($id, $format);

            return [
                'result' => 'success',
                'file' => basename($filePath),
                'url' => '/exports/' . basename($filePath),
                'message' => 'Экспорт завершен',
            ];
        } catch (Exception $e) {
            return [
                'result' => 'error',
                'message' => $e->getMessage(),
            ];
        }
    }

    private function ruDay($day): string
    {
        $dayMap = [
            'Mon' => 'Пн',
            'Tue' => 'Вт',
            'Wed' => 'Ср',
            'Thu' => 'Чт',
            'Fri' => 'Пт',
            'Sat' => 'Сб',
            'Sun' => 'Вс',
        ];
        return $dayMap[$day] ?? $day;
    }

    public function actionCharts($period = 'week'): array
    {
        $now = time();
        $data = [];

        switch ($period) {
            case 'today':
                $start = strtotime('today');
                $end = $now;
                $interval = 3600; // 1 час
                $format = 'H:00';
                break;

            case 'month':
                $start = strtotime('-30 days', $now);
                $end = $now;
                $interval = 86400; // 1 день
                $format = 'd.m';
                break;

            case 'week':
            default:
                $start = strtotime('-7 days', $now);
                $end = $now;
                $interval = 86400; // 1 день
                $format = 'D'; // Mon, Tue, etc.
                break;
        }

        // Создаем массив точек для графика
        for ($time = $start; $time <= $end; $time += $interval) {
            $nextTime = $time + $interval;

            $count = Member::find()
                ->where(['>=', 'registered_at', $time])
                ->andWhere(['<', 'registered_at', $nextTime])
                ->count();

            $data[] = [
                'date' => $this->ruDay(date($format, $time)),
                'count' => (int)$count,
                'timestamp' => $time,
            ];
        }

        $todayStart = strtotime('today');
        $todayEnd = strtotime('tomorrow') - 1;

        $todayRegistrations = Member::find()
            ->where(['>=', 'registered_at', $todayStart])
            ->andWhere(['<=', 'registered_at', $todayEnd])
            ->count();

        $todayRegistrationsYesterday = Member::find()
            ->where(['>=', 'registered_at', strtotime('-1 day', $todayStart)])
            ->andWhere(['<=', 'registered_at', strtotime('-1 day', $todayEnd)])
            ->count();

        $registrationsTrend = $todayRegistrationsYesterday > 0
            ? round((($todayRegistrations - $todayRegistrationsYesterday) / $todayRegistrationsYesterday) * 100)
            : ($todayRegistrations > 0 ? 100 : 0);

        return [
            'result' => 'success',
            "data" => $data,
            'registrationsTrend' => $registrationsTrend,
            'todayRegistrations' => $todayRegistrations
        ];
    }

//    /**
//     * Проверка доступа к мероприятию
//     */
//    private function checkEventAccess(Event $event): void
//    {
//        // Для публичных действий (регистрация) не проверяем
//        $publicActions = ['register'];
//        if (in_array(Yii::$app->requestedAction->id, $publicActions)) {
//            return;
//        }
//
//        // Для остальных действий нужны права админа/менеджера
//        if (!Yii::$app->user->can('admin') && !Yii::$app->user->can('manager')) {
//            throw new ForbiddenHttpException('Недостаточно прав');
//        }
//    }

    public array $accessRules = [
        'stats' => [],
        'charts' => [],
        'event-list' => [],
        'list' => [],
        'view' => [],
        'register' => [], // публичный доступ, можно и авторизованным
        'create' => ['admin', 'manager'],
        'update' => ['admin', 'manager'],
        'delete' => ['admin', 'manager'],
        'confirm' => ['admin', 'manager'],
        'cancel' => ['admin', 'manager'],
        'attend' => ['admin', 'manager'],
        'un-attend' => ['admin', 'manager'],
        'checkin' => ['admin', 'manager'],
        'upload-photo' => ['admin', 'manager'],
        'remove-photo' => ['admin', 'manager'],
        'export' => ['admin', 'manager'],
    ];
}