<?php

namespace app\controllers;

use app\core\entities\Settings;
use app\core\forms\settings\SettingsGeneralForm;
use app\core\forms\settings\SettingsMailForm;
use app\core\forms\settings\TemplateLettersForm;
use app\core\forms\settings\TemplateNotesForm;
use app\core\helpers\SettingsHelper;
use app\core\helpers\StringHelper;
use Yii;
use yii\db\Exception;
use yii\web\BadRequestHttpException;
use yii\web\UploadedFile;

class SettingsController extends ApiController
{
    public function actionIndex(): array
    {
        try {
            $mods_response = Yii::$app->apiClient->request('GET', 'mod/purchased');
            if (is_array($mods_response)) {
                SettingsHelper::setSetting('shop.mods', json_encode($mods_response, JSON_UNESCAPED_UNICODE));
            }
        } catch (\Exception $e) {

        }

        $mods = [
            'mods' => json_decode(SettingsHelper::getSetting('shop.mods', '[]'), true), //'mods' => ['mod_wl'],
            'company.logo' => SettingsHelper::getSetting('company.logo'),
        ];

        $settings = Settings::find()->all();

        if (!Yii::$app->user->isGuest) {
            //return array_merge(array_column($settings, 'value', 'key'), $mods);
            $settings_array = array_merge(array_column($settings, 'value', 'key'), $mods);
            $settings_array['from_email'] = $settings_array['from_email'] ?? Yii::$app->params['senderEmail'];
            $settings_array['from_name'] = $settings_array['from_name'] ?? Yii::$app->params['senderName'];
            return $settings_array;
        }

        return $mods;
    }

    public function actionHasBot(): bool
    {
        $token = SettingsHelper::getSetting('telegram.token');
        return !!$token;
    }

    public function actionMods(): array
    {
        return [
            //'mods' => ['mod_wl']
            'mods' => []
        ];
    }

    public function actionSaveGeneral(): array
    {
        if (Yii::$app->params['demo']) { // Демо режим
            return [
                'result' => 'error',
                'message' => 'Запрещено в демо режиме'
            ];
        }

        $form = new SettingsGeneralForm();

        if ($form->load(Yii::$app->request->post(), '') && $form->validate()) {
            SettingsHelper::setSetting('from_name', $form->name);
            SettingsHelper::setSetting('from_email', $form->email);

            if ($form->logo) {
                SettingsHelper::setSetting('company.logo', $form->logo);
            }

            return [
                'result' => 'success',
                'message' => 'Настройки обновлены',
            ];
        }

        return [
            'result' => 'error',
            'errors' => $form->errors,
        ];
    }

    /**
     * @throws Exception
     */
    public function actionSaveMail(): array
    {
        if (Yii::$app->params['demo']) { // Демо режим
            return [
                'result' => 'error',
                'message' => 'Запрещено в демо режиме'
            ];
        }

        $form = new SettingsMailForm();

        if ($form->load(Yii::$app->request->post(), '') && $form->validate()) {
            SettingsHelper::setSetting('mail.method', $form->method);
            SettingsHelper::setSetting('mail.encryption', $form->encryption);
            SettingsHelper::setSetting('mail.smtp.host', $form->host);
            SettingsHelper::setSetting('mail.smtp.port', $form->port);
            SettingsHelper::setSetting('mail.smtp.username', $form->username);
            SettingsHelper::setSetting('mail.smtp.password', $form->password);

            return [
                'result' => 'success'
            ];
        }

        return [
            'result' => 'error',
            'errors' => $form->errors,
        ];
    }

    public function actionSaveTelegram(): array
    {
        if (Yii::$app->params['demo']) { // Демо режим
            return [
                'result' => 'error',
                'message' => 'Запрещено в демо режиме'
            ];
        }

        $token = Yii::$app->request->post('token');
        if (!$token || StringHelper::isValidTelegramToken($token)) {
            SettingsHelper::setSetting('telegram.token', $token);
            return [
                'result' => 'success',
            ];
        }
        return [
            'result' => 'error',
        ];
    }

    public function actionTemplateNotes(): array
    {
        if (Yii::$app->params['demo']) { // Демо режим
            return [
                'result' => 'error',
                'message' => 'Запрещено в демо режиме'
            ];
        }

        $form = new TemplateNotesForm();

        if ($form->load(Yii::$app->request->post(), '') && $form->validate()) {
            SettingsHelper::setSetting('tpl.before', $form->before);
            SettingsHelper::setSetting('tpl.after', $form->after);
            SettingsHelper::setSetting('tpl.result', $form->result);

            return [
                'result' => 'success',
                'message' => 'Настройки обновлены',
            ];
        }

        return [
            'result' => 'error',
            'errors' => $form->errors,
        ];
    }

    public function actionTemplateLetters(): array
    {
        if (Yii::$app->params['demo']) { // Демо режим
            return [
                'result' => 'error',
                'message' => 'Запрещено в демо режиме'
            ];
        }

        $form = new TemplateLettersForm();

        if ($form->load(Yii::$app->request->post(), '') && $form->validate()) {
            SettingsHelper::setSetting('tpl.mail.notify', $form->notify);

            return [
                'result' => 'success',
                'message' => 'Настройки обновлены',
            ];
        }

        return [
            'result' => 'error',
            'errors' => $form->errors,
        ];
    }

    public function actionSaveSetting(): array
    {
        if (Yii::$app->params['demo']) { // Демо режим
            return [
                'result' => 'error',
                'message' => 'Запрещено в демо режиме'
            ];
        }

        $key = Yii::$app->request->post('key');
        $value = Yii::$app->request->post('value');
        SettingsHelper::setSetting($key, $value);
        return [
            'result' => 'success',
        ];
    }

    /**
     * Загрузка логотипа компании
     */
    public function actionUploadLogo(): array
    {
        if (Yii::$app->params['demo']) { // Демо режим
            return [
                'result' => 'error',
                'message' => 'Запрещено в демо режиме'
            ];
        }

        $logo = UploadedFile::getInstanceByName('logo');

        if (!$logo) {
            throw new BadRequestHttpException('Файл логотипа не загружен');
        }

        // Проверка типа файла
        $allowedTypes = ['image/jpeg', 'image/png', 'image/gif', 'image/webp', 'image/svg+xml'];
        if (!in_array($logo->type, $allowedTypes)) {
            return [
                'result' => 'error',
                'message' => 'Допустимые форматы: JPEG, PNG, GIF, WebP, SVG',
            ];
        }

        // Проверка размера (макс 5MB)
        if ($logo->size > 5 * 1024 * 1024) {
            return [
                'result' => 'error',
                'message' => 'Размер файла не должен превышать 5 МБ',
            ];
        }

        try {
            // Генерируем уникальное имя файла
            $fileName = 'logo-' . md5('company-logo-' . time()) . '.' . $logo->extension;
            $path = Yii::getAlias('@webroot/uploads/logo/' . $fileName);

            // Создаем директорию если не существует
            $dir = dirname($path);
            if (!is_dir($dir)) {
                mkdir($dir, 0755, true);
            }

            // Удаляем старый логотип если есть
            $oldLogo = SettingsHelper::getSetting('company.logo');
            if ($oldLogo && file_exists(Yii::getAlias('@webroot' . $oldLogo))) {
                unlink(Yii::getAlias('@webroot' . $oldLogo));
            }

            // Сохраняем файл
            if ($logo->saveAs($path)) {
                // Сохраняем путь в настройках
                $logoPath = '/uploads/logo/' . $fileName;
                SettingsHelper::setSetting('company.logo', $logoPath);

                return [
                    'result' => 'success',
                    'logo' => $logoPath,
                    'message' => 'Логотип успешно загружен',
                ];
            } else {
                throw new \Exception('Не удалось сохранить файл');
            }
        } catch (\Exception $e) {
            return [
                'result' => 'error',
                'message' => $e->getMessage(),
            ];
        }
    }

    /**
     * Удаление логотипа компании
     */
    public function actionRemoveLogo(): array
    {
        if (Yii::$app->params['demo']) { // Демо режим
            return [
                'result' => 'error',
                'message' => 'Запрещено в демо режиме'
            ];
        }

        try {
            $logoPath = SettingsHelper::getSetting('company.logo');

            if ($logoPath && file_exists(Yii::getAlias('@webroot' . $logoPath))) {
                unlink(Yii::getAlias('@webroot' . $logoPath));
            }

            SettingsHelper::setSetting('company.logo', null);

            return [
                'result' => 'success',
                'message' => 'Логотип удален',
            ];
        } catch (\Exception $e) {
            return [
                'result' => 'error',
                'message' => $e->getMessage(),
            ];
        }
    }

    public array $accessRules = [
        'save-general' => ["admin"],
        'save-mail' => ["admin"],
        'save-telegram' => ["admin"],
        'save-setting' => ["admin"],
        'upload-logo' => ["admin"],
        'remove-logo' => ["admin"],
    ];

    public static function publicActions(): array
    {
        return [
          'index', 'has-bot'
        ];
    }
}