<?php

namespace app\controllers;

use app\core\entities\User;
use app\core\helpers\SettingsHelper;
use Yii;
use yii\helpers\Json;
use yii\web\BadRequestHttpException;
use yii\web\TooManyRequestsHttpException;

class TelegramController extends ApiController
{
    /**
     * Обработка входящих сообщений от Telegram
     */
    public function actionWebhook(): array
    {
        $input = Yii::$app->request->getRawBody();

        // Проверка размера
        if (strlen($input) > 1024 * 1024) {
            throw new BadRequestHttpException('Payload too large', 413);
        }

        $update = Json::decode($input);
        if ($update === null) {
            throw new BadRequestHttpException('Invalid JSON data');
        }

        $chatId = $update['message']['chat']['id'] ?? null;

        // Rate limiting по chat_id
        if ($chatId && !$this->checkRateLimit($chatId)) {
            throw new TooManyRequestsHttpException('Rate limit exceeded');
        }

        // Обработка команды /start
        if (isset($update['message']['text']) && strpos($update['message']['text'], '/start') === 0) {
            $this->processStartCommand($update);
            return ['status' => 'ok'];
        }

        // Обработка обычных сообщений
        if (isset($update['message']['text'])) {
            $this->processMessage($update);
            return ['status' => 'ok'];
        }

        return ['status' => 'ok'];
    }

    /**
     * Установка веб хука
     */
    public function actionSetWebhook(): array
    {
        if (Yii::$app->params['demo']) { // Демо режим
            return [
                'result' => 'error',
                'message' => 'Запрещено в демо режиме'
            ];
        }

        $botToken = SettingsHelper::getSetting('telegram.token');
        if (!$botToken) {
            throw new BadRequestHttpException('Ошибка конфигурации бота');
        }

        // Ваша функция установки веб хука
        $data = $this->setWebHook();

        return [
            'result' => 'success',
            'data' => $data
        ];
    }

    /**
     * Удаление веб хука
     */
    public function actionDeleteWebhook(): array
    {
        if (Yii::$app->params['demo']) { // Демо режим
            return [
                'result' => 'error',
                'message' => 'Запрещено в демо режиме'
            ];
        }

        $botToken = SettingsHelper::getSetting('telegram.token');
        if (!$botToken) {
            throw new BadRequestHttpException('Telegram bot token not configured');
        }

        $apiUrl = "https://api.telegram.org/bot{$botToken}/deleteWebhook";
        $result = file_get_contents($apiUrl);
        $response = Json::decode($result);

        return [
            'result' => 'success',
            'data' => $response
        ];
    }

    /**
     * Получение информации о вебхуке
     */
    public function actionGetWebhookInfo()
    {
        $botToken = SettingsHelper::getSetting('telegram.token');
        if (!$botToken) {
            throw new BadRequestHttpException('Не указан токен Telegram бота');
        }

        $apiUrl = "https://api.telegram.org/bot{$botToken}/getWebhookInfo";
        $result = file_get_contents($apiUrl);
        $response = Json::decode($result);

        $response['auto_detected_url'] = $this->getWebhookUrl();

        return [
            'result' => 'success',
            'data' => $response
        ];
    }

    /**
     * Получение информации о боте
     */
    public function actionGetBotInfo(): array
    {
        $botToken = SettingsHelper::getSetting('telegram.token');
        if (!$botToken) {
            throw new BadRequestHttpException('Telegram бот не настроен');
        }

        $apiUrl = "https://api.telegram.org/bot{$botToken}/getMe";
        $result = @file_get_contents($apiUrl);
        $response = Json::decode($result);

        if (!$response || !isset($response['ok']) || !$response['ok']) {
            throw new BadRequestHttpException('Не удалось получить информацию о боте');
        }

        return [
            'result' => 'success',
            'data' => $response
        ];
    }

    /**
     * Обработка команды /start
     */
    private function processStartCommand(array $update): void
    {
        $chatId = $update['message']['chat']['id'];
        $text = $update['message']['text'];
        $firstName = $update['message']['chat']['first_name'] ?? 'Пользователь';

        // Обработка кода 123456_0 (привязка) _1 (логин)
        $parts = explode(' ', $text);
        $code_line = $parts[1] ?? '';

        if ($code_line) {
            $parts2 = explode('_', $code_line);
            $code = $parts2[0] ?? '';

            if (($parts2[1] ?? null) == '0') { // link
                $user = User::find()
                    ->where(['tg_code' => $code])
                    ->andWhere(['>', 'tg_expired', time()])
                    ->one();

                if ($user) {
                    $user->tg_id = $chatId;
                    $user->tg_code = null;
                    $user->tg_expired = null;
                    $user->save();
                    $this->sendTelegramMessage($chatId, "Аккаунт привязан.\nТеперь можно авторизоваться с помощью Telegram");
                } else {
                    $this->sendTelegramMessage($chatId, "Пользователь не найден или ключ устарел.\nПопробуйте на сайте обновить страницу и снова привязать Telegram");
                }
            } elseif (($parts2[1] ?? null) == '1') { // login
                $user = User::find()->where(['tg_id' => $chatId])->one();
                if ($user) {
                    $this->setUserAuth($chatId, $code);
                    //$this->loginWithTelegram($chatId, $code, $firstName);
                    $this->sendTelegramMessage($chatId, "Авторизация пройдена.\nВернитесь в браузер и обновите страницу");
                    $user->tg_access = 1;
                    $user->tg_code = null;
                    $user->tg_expired = null;
                    $user->save();
                } else {
                    $this->sendTelegramMessage($chatId, "Пользователь не найден или ключ устарел.\nПопробуйте на сайте обновить страницу и снова авторизоваться с помощью Telegram");
                }
            }
        } else {
            $this->processStartMenu($chatId, $text, $firstName);
        }
    }

    /**
     * Обработка стартового меню
     */
    private function processStartMenu($chatId, $text, $firstName): void
    {
        $parts = explode(' ', $text);
        if (count($parts) < 2) {
            $this->showMainMenu($chatId);
            return;
        }

        $params = $parts[1];
        parse_str($params, $parsedParams);

        $type = $parsedParams['type'] ?? '';
        $code = $parsedParams['code'] ?? '';

        if (empty($type) || empty($code)) {
            $this->showMainMenu($chatId);
            return;
        }

        switch ($type) {
            case 'link':
                $this->linkTelegramAccount($chatId, $code, $firstName);
                break;
            case 'login':
                $this->loginWithTelegram($chatId, $code, $firstName);
                break;
            default:
                $this->sendTelegramMessage($chatId, "❌ Неизвестный тип операции");
        }
    }

    /**
     * Обработка обычных сообщений
     */
    private function processMessage(array $update): void
    {
        $chatId = $update['message']['chat']['id'];
        $text = $update['message']['text'];
        $firstName = $update['message']['chat']['first_name'] ?? 'Пользователь';

        $this->showHelp($chatId);
    }

    /**
     * Привязка аккаунта Telegram
     */
    private function linkTelegramAccount($chatId, $code, $firstName): void
    {
        $user = User::find()->where(['tg_code' => $code])->one();

        if (!$user) {
            $this->sendTelegramMessage($chatId, "❌ Код привязки не найден или устарел. Получите новый код на сайте.");
            return;
        }

        if ($user->tg_expired && $user->tg_expired < time()) {
            $this->sendTelegramMessage($chatId, "❌ Срок действия кода истек. Получите новый код на сайте.");
            return;
        }

        if (Yii::$app->params['demo']) { // Демо режим
            $this->sendTelegramMessage($chatId, "❌ Запрещено в демо режиме.");
            return;
        }

        $user->updateAttributes([
            'tg_id' => $chatId,
            'tg_code' => null,
            'tg_expired' => null
        ]);

        $message = "✅ Телеграм успешно привязан!\n\n";
        $message .= "👤 Аккаунт: {$user->email}\n";
        $message .= "🆔 Ваш Telegram ID: {$chatId}\n\n";
        $message .= "Теперь вы можете использовать авторизацию через Telegram.";

        $this->sendTelegramMessage($chatId, $message);
        $this->showMainMenu($chatId);
    }

    /**
     * Авторизация через Telegram
     */
    private function loginWithTelegram($chatId, $code, $firstName): void
    {
        $user = User::find()->where(['tg_code' => $code])->one();

        if (!$user) {
            $this->sendTelegramMessage($chatId, "❌ Код авторизации не найден или устарел.");
            return;
        }

        if ($user->tg_expired && $user->tg_expired < time()) {
            $this->sendTelegramMessage($chatId, "❌ Срок действия кода истек. Попробуйте авторизоваться снова.");
            return;
        }

        if ($user->tg_id != $chatId) {
            $this->sendTelegramMessage($chatId, "❌ Этот Telegram аккаунт не привязан к данному пользователю.");
            return;
        }

        if (Yii::$app->params['demo']) { // Демо режим
            $this->sendTelegramMessage($chatId, "❌ Запрещено в демо режиме.");
            return;
        }

        $user->updateAttributes([
            'tg_access' => 1,
            'tg_code' => null,
            'tg_expired' => null
        ]);

        $message = "✅ Авторизация прошла успешно!\n\n";
        $message .= "👋 Добро пожаловать, {$firstName}!\n";
        $message .= "👤 Аккаунт: {$user->email}\n\n";
        $message .= "Вернитесь в браузер и продолжайте работу.";

        $this->sendTelegramMessage($chatId, $message);
    }

    /**
     * Отправка сообщения в Telegram
     */
    private function sendTelegramMessage($chatId, $text, $showMenu = false): void
    {
        $botToken = SettingsHelper::getSetting('telegram.token');

        if (!$botToken) {
            Yii::error("Telegram bot token not configured. Message: {$text}");
            return;
        }

        $data = [
            'chat_id' => $chatId,
            'text' => $text,
            'parse_mode' => 'HTML'
        ];

        $apiUrl = "https://api.telegram.org/bot{$botToken}/sendMessage";

        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $apiUrl);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $data);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_exec($ch);
        curl_close($ch);
    }

    /**
     * Rate limiting
     */
    private function checkRateLimit($chatId): bool
    {
        $cache = Yii::$app->cache;
        $cacheKey = "telegram_rate_limit_{$chatId}";

        $requests = $cache->get($cacheKey) ?: [];
        $now = time();

        // Очистка старых записей
        $requests = array_filter($requests, function($timestamp) use ($now) {
            return $now - $timestamp < 60; // 1 минута
        });

        // Проверка лимита (не более 10 запросов в минуту)
        if (count($requests) >= 10) {
            return false;
        }

        $requests[] = $now;
        $cache->set($cacheKey, $requests, 60);

        return true;
    }

    /**
     * Сохранение данных авторизации
     */
    private function setUserAuth($chatId, $code, $tempData = null): bool
    {
        if (!preg_match('/^[a-zA-Z0-9]+$/', $code)) {
            return false;
        }

        $storagePath = Yii::getAlias('@runtime/user_auth');
        if (!is_dir($storagePath)) {
            mkdir($storagePath, 0755, true);
        }

        $filename = $storagePath . '/' . $code . '.json';

        $data = [
            'code' => $code,
            'chat_id' => $chatId,
            'temp_data' => $tempData,
            'timestamp' => time()
        ];

        return file_put_contents($filename, Json::encode($data)) !== false;
    }

    /**
     * Показать главное меню
     */
    private function showMainMenu($chatId): void
    {
        $message = "👋 Добро пожаловать!\n\n";
        $message .= "Выберите действие:\n\n";
        $message .= "🆘 Помощь - справка по боту";

        $this->sendTelegramMessage($chatId, $message, true);
    }

    /**
     * Показать справку
     */
    private function showHelp($chatId): void
    {
        $message = "🆘 Помощь по боту\n\n";
        $message .= "• Бот предназначен только для быстрой авторизации администраторов\n";

        $this->sendTelegramMessage($chatId, $message, true);
    }

    /**
     * Установить веб хук (заглушка - реализуйте свою логику)
     */
    private function setWebHook(): array
    {
        if (Yii::$app->params['demo']) { // Демо режим
            return [
                'result' => 'error',
                'message' => 'Запрещено в демо режиме'
            ];
        }

        $botToken = SettingsHelper::getSetting('telegram.token');

        if (!$botToken) {
            return [
                'result' => 'error'
            ];
        }

        $webhookUrl = $this->getWebhookUrl();

        $apiUrl = "https://api.telegram.org/bot{$botToken}/setWebhook?url={$webhookUrl}";

        $result = @file_get_contents($apiUrl);
        $response = json_decode($result, true);

        return [
            'webhook_url' => $webhookUrl,
            'result' => $response
        ];
    }

    private function getWebhookUrl(): string
    {
        //return Yii::$app->params['generalDomain'] . '/telegram/webhook';

        //print_r(Yii::$app->params['generalDomain'] . '/api.php?r=/telegram/webhook'); die;
        return Yii::$app->params['generalDomain'] . '/api.php?r=/telegram/webhook';

        ///api.php?r=/telegram/webhook

        //return $this->full_url('/telegram/webhook');
    }

//    private function full_url($path = '/'): string
//    {
//        $relativeUrl = url($path);
//        $protocol = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
//        $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
//
//        // Безопасная проверка порта
//        $port = $_SERVER['SERVER_PORT'] ?? null;
//        if ($port && !in_array($port, [80, 443])) {
//            $host = $host . ':' . $port;
//        }
//
//        return "{$protocol}://{$host}{$relativeUrl}";
//    }

    public static function publicActions(): array
    {
        return ['get-bot-info', 'webhook'];
    }
}
