<?php

namespace app\controllers;

use Exception;
use Yii;

class UpdateController extends ApiController
{
    public array $accessRules = [
        'check' => [],
        'available' => [],
        'status' => [],
        'history' => [],
        'request' => ['admin'],
        'cancel' => ['admin',],
        'force-check' => ['admin'],
    ];

    /**
     * Проверка наличия обновлений
     * GET /update/check
     */
    public function actionCheck(): array
    {
        try {
            $currentVersion = $this->getCurrentVersion();
            $lastCheck = $this->getLastCheckTime();

            return [
                'success' => true,
                'data' => [
                    'current_version' => $currentVersion,
                    'last_check' => $lastCheck,
                    'has_updates' => $this->hasPendingUpdates(),
                    'update_in_progress' => $this->isUpdateInProgress(),
                ]
            ];

        } catch (Exception $e) {
            Yii::error("Update check failed: " . $e->getMessage());
            return $this->errorResponse($e->getMessage());
        }
    }

    /**
     * Получение доступных обновлений
     * GET /update/available
     */
    public function actionAvailable(): array
    {
        try {
            $currentVersion = $this->getCurrentVersion();

            $availableUpdates = $this->fetchAvailableUpdates($currentVersion);
            $updateChain = $this->getUpdateChain($currentVersion);
            $subs = $this->fetchUpdateSubscription();

            return [
                'result' => 'success',
                'data' => [
                    'current_version' => $currentVersion,
                    'available_updates' => $availableUpdates,
                    'update_chain' => $updateChain,
                    'latest_version' => !empty($availableUpdates) ? end($availableUpdates) : $currentVersion,
                    'can_update' => !empty($availableUpdates) && !$this->isUpdateInProgress(),
                ],
                'subs' => $subs
            ];

        } catch (Exception $e) {
            Yii::error("Fetch available updates failed: " . $e->getMessage());
            return $this->errorResponse($e->getMessage());
        }
    }

    /**
     * Запрос на обновление
     * POST /update/request
     */
    public function actionRequest(): array
    {
        if (Yii::$app->params['demo']) { // Демо режим
            return [
                'result' => 'error',
                'message' => 'Запрещено в демо режиме'
            ];
        }

        try {
            $request = Yii::$app->request;
            $targetVersion = $request->post('target_version');

            if ($this->isUpdateInProgress()) {
                return $this->errorResponse('Обновление уже выполняется');
            }

            $currentVersion = $this->getCurrentVersion();

            if ($targetVersion) {
                if (version_compare($targetVersion, $currentVersion) <= 0) {
                    return $this->errorResponse('Целевая версия должна быть больше текущей');
                }

                $this->createUpdateRequest($targetVersion);

                return [
                    'success' => true,
                    'message' => "Запрос на обновление до версии {$targetVersion} отправлен",
                    'data' => [
                        'target_version' => $targetVersion,
                        'current_version' => $currentVersion,
                    ]
                ];
            } else {
                $availableUpdates = $this->fetchAvailableUpdates($currentVersion);

                if (empty($availableUpdates)) {
                    return $this->errorResponse('Нет доступных обновлений');
                }

                $latestVersion = end($availableUpdates);
                $this->createUpdateRequest($latestVersion);

                return [
                    'success' => true,
                    'message' => "Запрос на обновление до версии {$latestVersion} отправлен",
                    'data' => [
                        'target_version' => $latestVersion,
                        'current_version' => $currentVersion,
                        'update_chain' => $availableUpdates,
                    ]
                ];
            }

        } catch (Exception $e) {
            Yii::error("Update request failed: " . $e->getMessage());
            return $this->errorResponse($e->getMessage());
        }
    }

    /**
     * Статус обновления
     * GET /update/status
     */
    public function actionStatus(): array
    {
        try {
            $data = [
                'current_version' => $this->getCurrentVersion(),
                'has_pending_request' => $this->hasPendingUpdates(),
                'update_in_progress' => $this->isUpdateInProgress(),
                'last_check' => $this->getLastCheckTime(),
            ];

            if ($this->isUpdateInProgress()) {
                $progressData = $this->getUpdateProgress();
                $data['progress'] = $progressData;

                if (isset($progressData['status']) && $progressData['status'] === 'failed') {
                    $data['failed'] = true;
                    $data['error'] = $progressData['error'] ?? 'Неизвестная ошибка';
                }
            }

            return [
                'success' => true,
                'data' => $data
            ];

        } catch (Exception $e) {
            Yii::error("Update status failed: " . $e->getMessage());
            return $this->errorResponse($e->getMessage());
        }
    }

    /**
     * История обновлений
     * GET /update/history
     */
    public function actionHistory(): array
    {
        try {
            $history = $this->getUpdateHistory();

            return [
                'success' => true,
                'data' => [
                    'history' => $history,
                    'total' => count($history),
                ]
            ];

        } catch (Exception $e) {
            Yii::error("Update history failed: " . $e->getMessage());
            return $this->errorResponse($e->getMessage());
        }
    }

    /**
     * Отмена обновления
     * POST /update/cancel
     */
    public function actionCancel(): array
    {
        if (Yii::$app->params['demo']) { // Демо режим
            return [
                'result' => 'error',
                'message' => 'Запрещено в демо режиме'
            ];
        }

        try {
            if (!$this->hasPendingUpdates()) {
                return $this->errorResponse('Нет активных запросов на обновление');
            }

            $this->cancelUpdateRequest();

            return [
                'success' => true,
                'message' => 'Запрос на обновление отменен'
            ];

        } catch (Exception $e) {
            Yii::error("Update cancel failed: " . $e->getMessage());
            return $this->errorResponse($e->getMessage());
        }
    }

    /**
     * Принудительная проверка обновлений
     * POST /update/force-check
     */
    public function actionForceCheck(): array
    {
        try {
            $currentVersion = $this->getCurrentVersion();
            $availableUpdates = $this->fetchAvailableUpdates($currentVersion);

            $this->updateLastCheckTime();

            return [
                'success' => true,
                'data' => [
                    'current_version' => $currentVersion,
                    'available_updates' => $availableUpdates,
                    'has_updates' => !empty($availableUpdates),
                    'checked_at' => date('Y-m-d H:i:s'),
                ]
            ];

        } catch (Exception $e) {
            Yii::error("Force check failed: " . $e->getMessage());
            return $this->errorResponse($e->getMessage());
        }
    }

    // ========== ВСПОМОГАТЕЛЬНЫЕ МЕТОДЫ ==========

    private function getCurrentVersion(): string
    {
        return Yii::$app->params['version'];
    }

    private function getLastCheckTime(): ?string
    {
        $checkFile = Yii::getAlias('@runtime/last_update_check.txt');

        if (file_exists($checkFile)) {
            return date('Y-m-d H:i:s', filemtime($checkFile));
        }

        return null;
    }

    private function updateLastCheckTime(): void
    {
        $checkFile = Yii::getAlias('@runtime/last_update_check.txt');
        touch($checkFile);
    }

    private function hasPendingUpdates(): bool
    {
        $needFile = Yii::getAlias('@runtime/update_need.txt');
        return file_exists($needFile);
    }

    private function isUpdateInProgress(): bool
    {
        $progressFile = Yii::getAlias('@runtime/update_process.txt');
        return file_exists($progressFile);
    }

    private function fetchUpdateSubscription(): array
    {
        try {
            $response = Yii::$app->apiClient->request('GET', 'subscription/my-subscription');
            if (is_array($response)) {
                //print_r($response); die;
                return $response;
            }
        } catch (\Exception $e) {

        }

        return ['result' => 'error'];

//        if (!$updates || !json_validate($updates)) {
//            return [];
//        } else {
//            return json_decode($updates, true);
//        }
    }

    private function fetchAvailableUpdates(string $currentVersion): array
    {
        // Запрос к серверу обновлений
        $updates = @file_get_contents('https://update.id-pass.ru/update.php?current_version=' . Yii::$app->params['version'] . '&host=' . Yii::$app->params['generalDomain']);

        //print_r($updates); die;

        if (!$updates || !json_validate($updates)) {
            return [];
        } else {
            return json_decode($updates, true);
        }

        // Отключено пока
        // Симулируем
        $updatesFile = Yii::getAlias('@runtime/available_updates.json');

        if (file_exists($updatesFile)) {
            $data = json_decode(file_get_contents($updatesFile), true);
            return $data['updates'] ?? [];
        }

        // Генерация тестовых обновлений
        preg_match('/^(\d+)-(\d+)-(\d+)-(\d+)$/', $currentVersion, $matches);

        if (empty($matches)) {
            return [];
        }

        list(, $year, $major, $minor, $patch) = $matches;
        $updates = [];

        for ($i = 1; $i <= 3; $i++) {
            $newPatch = $patch + $i;
            $newMinor = $minor;
            $newMajor = $major;

            if ($newPatch > 9) {
                $newPatch = 0;
                $newMinor++;

                if ($newMinor > 9) {
                    $newMinor = 0;
                    $newMajor++;
                }
            }

            $updates[] = "{$year}-{$newMajor}-{$newMinor}-{$newPatch}";
        }

        return $updates;
    }

    private function getUpdateChain(string $currentVersion): array
    {
        if (!$this->hasPendingUpdates()) {
            return [];
        }

        $needFile = Yii::getAlias('@runtime/update_need.txt');
        $data = json_decode(file_get_contents($needFile), true);

        if (isset($data['updates'])) {
            return $data['updates'];
        }

        if (isset($data['target_version'])) {
            $targetVersion = $data['target_version'];
            $chain = [];
            $available = $this->fetchAvailableUpdates($currentVersion);

            foreach ($available as $version) {
                if (version_compare($version, $targetVersion) <= 0) {
                    $chain[] = $version;
                }
            }

            return $chain;
        }

        return [];
    }

    private function createUpdateRequest(string $targetVersion): void
    {
        $needFile = Yii::getAlias('@runtime/update_need.txt');

        $currentVersion = $this->getCurrentVersion();
        $chain = $this->getUpdateChainFromServer($currentVersion, $targetVersion);

        file_put_contents($needFile, json_encode([
            'requested_at' => date('Y-m-d H:i:s'),
            'requested_by' => Yii::$app->user->identity->username ?? 'api',
            'target_version' => $targetVersion,
            'updates' => $chain,
            'user_id' => Yii::$app->user->id,
            'auto' => false
        ], JSON_PRETTY_PRINT));
    }

    private function getUpdateChainFromServer(string $fromVersion, string $toVersion): array
    {
        $allUpdates = $this->fetchAvailableUpdates($fromVersion);
        $chain = [];

        foreach ($allUpdates as $version) {
            if (version_compare($version, $toVersion) <= 0) {
                $chain[] = $version;
            }
        }

        return $chain;
    }

    private function getUpdateProgress(): array
    {
        $progressFile = Yii::getAlias('@runtime/update_process.txt');

        if (!file_exists($progressFile)) {
            return ['status' => 'idle'];
        }

        $data = json_decode(file_get_contents($progressFile), true);

        if (!$data) {
            return ['status' => 'unknown'];
        }

        if (isset($data['total_steps']) && $data['total_steps'] > 0) {
            $currentStep = $data['current_step'] ?? 0;
            $data['progress_percent'] = round(($currentStep / $data['total_steps']) * 100);
        }

        if (isset($data['started_at'])) {
            $startTime = strtotime($data['started_at']);
            $elapsed = time() - $startTime;
            $data['elapsed_seconds'] = $elapsed;

            if ($elapsed > 3600) {
                $data['stuck'] = true;
            }
        }

        return $data;
    }

    private function getUpdateHistory(): array
    {
        $history = [];
        $appliedDir = Yii::getAlias('@runtime/updates/applied');

        if (!is_dir($appliedDir)) {
            return $history;
        }

        $files = glob($appliedDir . '/*.zip');

        foreach ($files as $file) {
            $filename = basename($file);
            preg_match('/\d{4}-\d{2}-\d{2}_\d{6}_(.+)-from-(.+).zip/', $filename, $matches);

            if ($matches) {
                $history[] = [
                    'file' => $filename,
                    'applied_at' => date('Y-m-d H:i:s', filemtime($file)),
                    'from_version' => $matches[2],
                    'to_version' => $matches[1],
                    'size' => filesize($file),
                ];
            }
        }

        usort($history, function($a, $b) {
            return strtotime($b['applied_at']) - strtotime($a['applied_at']);
        });

        return $history;
    }

    private function cancelUpdateRequest(): void
    {
        $needFile = Yii::getAlias('@runtime/update_need.txt');

        if (file_exists($needFile)) {
            $data = json_decode(file_get_contents($needFile), true);
            $data['cancelled_at'] = date('Y-m-d H:i:s');
            $data['cancelled_by'] = Yii::$app->user->identity->username ?? 'api';

            $cancelledFile = str_replace('update_need.txt', 'update_cancelled_' . date('Ymd_His') . '.txt', $needFile);
            rename($needFile, $cancelledFile);
        }
    }

    private function errorResponse(string $message): array
    {
        return [
            'success' => false,
            'error' => $message
        ];
    }
}
