<?php

namespace app\controllers;

use app\core\entities\User;
use app\core\forms\user\UserCreateForm;
use app\core\forms\user\UserForm;
use app\core\helpers\SerializeHelper;
use app\core\providers\MapDataProvider;
use app\core\repositories\user\UserRepository;
use app\core\services\user\UserService;
use Endroid\QrCode\Color\Color;
use Endroid\QrCode\Encoding\Encoding;
use Endroid\QrCode\ErrorCorrectionLevel;
use Endroid\QrCode\QrCode;
use Endroid\QrCode\RoundBlockSizeMode;
use Endroid\QrCode\Writer\SvgWriter;
use Yii;
use yii\web\ForbiddenHttpException;
use yii\web\Response;
use yii\web\UploadedFile;

class UserController extends ApiController
{
    private UserRepository $user_repository;
    private UserService $user_service;

    public function __construct($id, $module, UserRepository $user_repository, UserService $user_service, $config = [])
    {
        parent::__construct($id, $module, $config);
        $this->user_repository = $user_repository;
        $this->user_service = $user_service;
    }

    public function actionData(): array
    {
        if (Yii::$app->user->isGuest) {
            // Логика для возврата гостевых данных (например, пустой массив, сообщение об ошибке и т.д.)
            return [];
        }

        $user = Yii::$app->user->identity->user;
        return SerializeHelper::serializeListUsers($user);
    }

    public function actionList(): array
    {
        $dataProvider = $this->user_repository->getAll();

        return [
            'data' => new MapDataProvider($dataProvider['provider'], [SerializeHelper::class, 'serializeListUsers']),//$dataProvider['provider'],
            'total' => $dataProvider['count']
        ];
    }

    public function actionView($id): array
    {
        if (!Yii::$app->user->can('admin') && Yii::$app->user->id != $id) {
            throw new ForbiddenHttpException("Недостаточно прав");
        }

        $user = User::find()->andWhere(['id' => $id])->one();

        if ($user->id == Yii::$app->user->id) {
            $code = substr(str_shuffle('0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ'), 0, 8);
            $user->tg_code = $code;
            $user->tg_expired = time() + 10*60; // date('Y-m-d H:i:s', strtotime('+10 minutes'));
            $this->user_repository->save($user);
        }

        if ($user) {
            return [
                'result' => 'success',
                'user' => SerializeHelper::serializeListUsers($user),
            ];
        }
        return ['result' => 'error'];
    }

    public function actionUpdate(): array
    {
        if (Yii::$app->params['demo']) { // Демо режим
            return [
                'result' => 'error',
                'message' => 'Запрещено в демо режиме'
            ];
        }

        $id = Yii::$app->request->post('id');

        if (!Yii::$app->user->can('admin') && Yii::$app->user->id != $id) {
            throw new ForbiddenHttpException("Недостаточно прав");
        }

        $user = User::find()->andWhere(['id' => $id])->one();

        if (!$user) {
            return [
                'result' => 'error',
                'message' => 'Пользователь не найден',
            ];
        }

        $form = new UserForm($user);
        if ($form->load(Yii::$app->request->post(), '') && $form->validate()) {
            $this->user_service->edit($id, $form);
            return [
                'result' => 'success'
            ];
        }

        return [
            'result' => 'error',
            'message' => $form->errors,
        ];
    }

    public function actionCreate(): array
    {
        if (Yii::$app->params['demo']) { // Демо режим
            return [
                'result' => 'error',
                'message' => 'Запрещено в демо режиме'
            ];
        }

        $email = Yii::$app->request->post('email');
        $username = Yii::$app->request->post('username');
        $password = Yii::$app->request->post('password');

        $form           = new UserCreateForm();
        $form->username = $username;
        $form->email    = $email;
        $form->password = $password;

        if ($form->validate()) {
            $user = $this->user_service->create($form);
            return [
                'result' => 'success',
                'user' => SerializeHelper::serializeListUsers($user),
            ];
        }

        return [
            'result' => 'error',
            'message' => $form->errors,
        ];
    }

    public function actionRepair(): array
    {
        if (Yii::$app->params['demo']) { // Демо режим
            return [
                'result' => 'error',
                'message' => 'Запрещено в демо режиме'
            ];
        }

        $id = Yii::$app->request->post('id');
        $user = User::find()->andWhere(['id' => $id])->one();

        if ($user) {
            if ($user->status == User::STATUS_DELETED) {
                $this->user_service->repair($user->id);
                return [
                    'result' => 'success',
                    'message' => 'Пользователь восстановлен',
                ];
            }
        }

        return [
            'result' => 'error'
        ];
    }

    public function actionDeleteAll(): array
    {
        if (Yii::$app->params['demo']) { // Демо режим
            return [
                'result' => 'error',
                'message' => 'Запрещено в демо режиме'
            ];
        }

        $ids = Yii::$app->request->post('ids');
        $deleted = 0;
        $marked = 0;
        foreach ($ids as $id) {
            $user = User::find()->andWhere(['id' => $id])->one();
            if ($user->status == User::STATUS_ACTIVE) {
                $this->user_service->markDeleted($id);
                $marked++;
            } elseif ($user->status == User::STATUS_DELETED) {
                $this->user_service->remove($user->id);
                $deleted++;
            }
        }
        return [
            'result' => 'success',
            'deleted' => $deleted,
            'marked' => $marked,
        ];
    }

    public function actionDelete(): array
    {
        if (Yii::$app->params['demo']) { // Демо режим
            return [
                'result' => 'error',
                'message' => 'Запрещено в демо режиме'
            ];
        }

        $id = Yii::$app->request->post('id');
        $user = User::find()->andWhere(['id' => $id])->one();

        if ($user) {
            if ($user->status == User::STATUS_ACTIVE) {
                $this->user_service->markDeleted($id);
                return [
                    'result' => 'success',
                    'message' => 'Пользователь отмечен как удалённый',
                ];
            } else {
                $this->user_service->remove($user->id);
                return [
                    'result' => 'success',
                    'message' => 'Пользователь удалён',

                ];
            }
        }

        return [
            'result' => 'error'
        ];
    }

    public function actionSetAvatar(): array
    {
        if (Yii::$app->params['demo']) { // Демо режим
            return [
                'result' => 'error',
                'message' => 'Запрещено в демо режиме'
            ];
        }

        $id = Yii::$app->request->post('id');

        if (!Yii::$app->user->can('admin') && Yii::$app->user->id != $id) {
            throw new ForbiddenHttpException("Недостаточно прав");
        }

        $user = User::find()->andWhere(['id' => $id])->one();
        $user_pic = UploadedFile::getInstanceByName('file');

        /* @var $user_pic UploadedFile */
        if ($user_pic) {
            $fileName = md5('avatar-' . $user_pic->baseName . time()) . '.' . $user_pic->extension;
            $path = Yii::getAlias('@runtime/' . $fileName);

            if ($user_pic->size > 15 * 1024 * 1024) { // 15Mb
                return [
                    'result' => 'error',
                    'message' => 'Размер файла не должен превышать 15 Мб'
                ];
            }

            $user_pic->saveAs($path);

            $user->user_pic = basename(Yii::$app->avatar->update($user->username . '_' . $user->id, null, $path));

            if (file_exists($path)) {
                unlink($path);
            }
            $this->user_repository->save($user);
        }

        return [
            'result' => 'success'
        ];
    }

    public function actionLoginQr(): string
    {
        $user = Yii::$app->user->identity->user;
        $this->user_service->loginToken($user, 60);
        $authLink = Yii::$app->params['generalDomain'] . '?auth/login-token&token=' . $user->login_token;
        $qrCode = new QrCode(
            $authLink,
            new Encoding('UTF-8'),
            ErrorCorrectionLevel::Low,
            600,
            1,
            RoundBlockSizeMode::Margin,
            new Color(0,0,0, 0),
            new Color(255,255,255, 0),
        );

        $writer = new SvgWriter();
        $result = $writer->write($qrCode);
        Yii::$app->response->headers->set('Content-type', $result->getMimeType());
        Yii::$app->response->format = Response::FORMAT_RAW;
        return base64_encode($result->getString());
    }

    public array $accessRules = [
        'data' => [],  // любой авторизованный
        'view' => [],  // любой авторизованный - все кроме админа только себя
        'update' => [], // любой авторизованный - все кроме админа только себя
        'set-avatar' => [], // любой авторизованный - все кроме админа только себя
        'generate-telegram-code' => [],

        'list' => ['admin'],
        'create' => ['admin'],
        'repair' => ['admin'],
        'delete' => ['admin'],
        'delete-all' => ['admin'],  // только админ

        //'list' => ['admin', 'manager'],
    ];
}
