<?php

namespace app\core\auth;

use app\core\entities\User;
use app\core\repositories\user\UserRepository;
use Exception;
use Firebase\JWT\JWT;
use Firebase\JWT\Key;
use Yii;
use yii\base\InvalidConfigException;
use yii\di\NotInstantiableException;
use yii\filters\RateLimitInterface;
use yii\web\IdentityInterface;

class Identity implements IdentityInterface, RateLimitInterface
{
    public User $user;

    public int $rateLimit = 800;
//    public $allowance;
//    public $allowance_updated_at;

    public function __construct(User $user)
    {
        $this->user = $user;
    }

    /**
     * @throws NotInstantiableException
     * @throws InvalidConfigException
     */
    public static function findIdentity($id): ?Identity
    {
        $user = self::getRepository()->findActiveById($id);
        return $user ? new self($user): null;
    }

    public static function findIdentityByAccessToken($token, $type = null): User|IdentityInterface|null
    {
        try {
            // Убираем 'Bearer ' если есть
            if (str_starts_with($token, 'Bearer ')) {
                $token = substr($token, 7);
            }

            // Получаем ключ
            $secretKey = Yii::$app->params['jwtSecretKey'];
            if (str_starts_with($secretKey, 'base64:')) {
                $secretKey = base64_decode(substr($secretKey, 7));
            }

            // Декодируем JWT
            $decoded = JWT::decode($token, new Key($secretKey, 'HS512'));

            // Преобразуем объект в массив
            $data = (array) $decoded;

            if (isset($data['uid'])) {
                $user = static::findIdentity($data['uid']); //self::getRepository()->findActiveById($data['uid']);
                return $user ? $user : null;
            }

        } catch (Exception $e) {
            // Логируем ошибку, но не показываем пользователю
            Yii::error("JWT decode error: " . $e->getMessage());
        }

        return null;
    }

    /**
     * @throws NotInstantiableException
     * @throws InvalidConfigException
     */
//    public static function findIdentityByAccessTokenOld($token, $type = null): User|IdentityInterface|null
//    {
        //print_r($token); die;
        //Yii::debug("=== findIdentityByAccessToken CALLED ===");
        //Yii::debug("Token: " . substr($token, 0, 20) . '...');

        //return self::getRepository()->findByAccessToken($token);
        //$data = self::getOauth()->getServer()->getResourceController()->getToken();
        //return !empty($data['user_id']) ? static::findIdentity($data['user_id']) : null;

//        $jwt = new Jwt();
//        $decoded = $jwt->decode($token);
//
//        if ($decoded !== null && isset($decoded['body']['uid'])) {
//            return static::findIdentity($decoded['body']['uid']);
//        }
//
//        return null;
//    }

    public function getId(): string
    {
        return $this->user->id;
    }

    public function getAuthKey(): string
    {
        return $this->user->auth_key;
    }

    public function validateAuthKey($authKey): bool
    {
        return $this->getAuthKey() === $authKey;
    }

    /**
     * @throws NotInstantiableException
     * @throws InvalidConfigException
     */
    public function checkUserCredentials($username, $password): bool
    {
        $factor2 = filter_var(Yii::$app->request->post('factor2'), FILTER_VALIDATE_INT);

        if (!$user = self::getRepository()->findByEmail($username)) {
            return false;
        }

//        if ($user->usedFactor2()) {
//            if (!$factor2) {
//                return false;
//            }
//
//            $ga = new GoogleAuthenticator();
//            $checkResult = $ga->verifyCode($user->factor2_code, $factor2, 2);
//            if (!$checkResult) {
//                return false;
//            }
//        }

        return $user->validatePassword($password);
    }

    /**
     * @param $username
     * @return array
     * @throws InvalidConfigException
     * @throws NotInstantiableException
     */
    public function getUserDetails($username): array
    {
        $user = self::getRepository()->findByEmail($username);
        return ['user_id' => $user->id];
    }

    /**
     * @throws NotInstantiableException
     * @throws InvalidConfigException
     */
    private static function getRepository(): UserRepository
    {
        return Yii::$container->get(UserRepository::class);
    }

    public function getRateLimit($request, $action): array
    {
        return [$this->rateLimit, 600];
    }

    public function loadAllowance($request, $action): array
    {
        return [$this->user->allowance, $this->user->allowance_updated_at];
    }

    public function saveAllowance($request, $action, $allowance, $timestamp): void
    {
        $this->user->allowance = $allowance;
        $this->user->allowance_updated_at = $timestamp;
        $this->user->save();
    }
}
