<?php

namespace app\core\components;

use Exception;
use Yii;

class ApiClient
{
    public ?string $token = null;
    public ?int $token_expired = null;

    public function setToken(string $token, int $expires_in): void
    {
        $this->token = $token;
        $this->token_expired = time() + $expires_in;
    }

    public function getToken(): ?string
    {
        if ($this->token_expired < time()) {
            $this->fetchToken();
        }

        return $this->token;
    }

    public function clearToken(): void
    {
        $this->token = null;
        $this->token_expired = null;
    }

    public function fetchToken(): array
    {
        $this->clearToken();

        $domain = Yii::$app->params['generalDomain'];
        $url = Yii::$app->params['apiFatherUrl'] . '?r=/site/check';

        $ch = curl_init();

        // Данные для отправки
        $postData = ['domain' => $domain];

        curl_setopt_array($ch, [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => http_build_query($postData), // Формат: domain=example.com
            CURLOPT_TIMEOUT => 30,
            CURLOPT_HTTPHEADER => [
                'Content-Type: application/x-www-form-urlencoded',
            ],
        ]);

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);

        if (curl_errno($ch)) {
            $error = curl_error($ch);
            curl_close($ch);
            //throw new \Exception('CURL error: ' . $error);
            return [
                'result' => 'error',
                'error' => 'API недоступен: ' . $error,
                'http_code' => null
            ];
        }

        curl_close($ch);

        // Проверка HTTP кода
        if ($httpCode !== 200) {
            return [
                'result' => 'error',
                'error' => "Статус API: {$httpCode}",
                'http_code' => $httpCode,
                'raw_response' => $response
            ];
        }

        // Парсим JSON
        if (!json_validate($response)) {
            return [
                'result' => 'error',
                'error' => 'Некорректный ответ от API',
                'http_code' => $httpCode
            ];
        }

        $response = json_decode($response, true);

        if (isset($response['data']['token']) && isset($response['data']['expires_in'])) {
            $this->setToken($response['data']['token'], $response['data']['expires_in']);
            return [
                'result' => 'success',
            ];
        }
        return [
            'result' => 'error',
        ];
    }

    public function request($method, $endpoint, $data = [])
    {
        if (!$this->token) {
            $this->fetchToken();
        }

        $url = Yii::$app->params['apiFatherUrl'] . '?r=' . '/' . ltrim($endpoint, '/');

        // Для GET запросов добавляем параметры в URL
        if ($method === 'GET' && !empty($data)) {
            $url .= '&' . http_build_query($data);
        }

        $headers = [
            'Accept: application/json',
            'Content-Type: application/json',
        ];

        if ($this->token) {
            $headers[] = 'Authorization: Bearer ' . $this->token;
        }

        $ch = curl_init();

        curl_setopt_array($ch, [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_CUSTOMREQUEST => $method,
            CURLOPT_HTTPHEADER => $headers,
            CURLOPT_TIMEOUT => 30,
        ]);

        if (in_array($method, ['POST', 'PUT', 'PATCH'])) {
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        }

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);

        curl_close($ch);

        if ($error) {
            throw new Exception('API request failed: ' . $error);
        }

        $data = json_decode($response, true);
        if ($httpCode >= 400) {
            throw new Exception('API error ' . $httpCode . ': ' . ($data['message'] ?? 'Unknown error'));
        }

        return $data;
    }
}
