<?php

namespace app\core\entities;

use app\core\events\EventTrait;
use app\core\AggregateRoot;
use DomainException;
use Exception;
use Ramsey\Uuid\Uuid;
use yii\behaviors\SluggableBehavior;
use yii\behaviors\TimestampBehavior;
use yii\db\ActiveQuery;
use yii\db\ActiveRecord;
use yii\helpers\HtmlPurifier;

/**
 * Event model
 *
 * @property string $id
 * @property string $title
 * @property string $slug
 * @property string $description
 * @property string $short_description
 * @property string $content [jsonb]
 * @property integer $status
 * @property string $type
 * @property string $location_type
 * @property string $location
 * @property string $image
 * @property integer $start_date
 * @property integer $end_date
 * @property integer $registration_start
 * @property integer $registration_end
 * @property integer $max_participants
 * @property float $price
 * @property string $currency
 * @property string $user_id
 * @property string $settings [jsonb]
 * @property string $meta_title
 * @property string $meta_description
 * @property string $meta_keywords
 * @property integer $created_at
 * @property integer $updated_at
 * @property bool $auto_confirm
 * @property string $notify_emails [jsonb]
 * @property string $success_message
 * @property integer $color
 *
 * @property User $user
 */
class Event extends ActiveRecord implements AggregateRoot
{
    use EventTrait;

    const int STATUS_DRAFT = 0;
    const int STATUS_DELETED = 1;
    const int STATUS_PENDING = 5;
    const int STATUS_PUBLISHED = 10;
    const int STATUS_CANCELLED = 15;
    const int STATUS_ARCHIVED = 20;

    const string TYPE_EVENT = 'event';
    const string TYPE_WEBINAR = 'webinar';
    const string TYPE_WORKSHOP = 'workshop';
    const string TYPE_CONFERENCE = 'conference';
    const string TYPE_MEETUP = 'meetup';
    const string TYPE_SEMINAR = 'seminar';
    const string TYPE_FORUM = 'forum';
    const string TYPE_SUMMIT = 'summit';



    const string LOCATION_ONLINE = 'online';
    const string LOCATION_OFFLINE = 'offline';
    const string LOCATION_HYBRID = 'hybrid';

    /**
     * @throws Exception
     */
    public static function create(
        string $title,
        string $slug,
        string $description,
        ?int $startDate,
        ?int $endDate,
        string $userId,
        ?string $shortDescription = null,
        ?array $content = null,
        string $type = self::TYPE_EVENT,
        string $locationType = self::LOCATION_ONLINE,
        ?string $location = null,
        ?string $image = null,
        ?int $registrationStart = null,
        ?int $registrationEnd = null,
        ?int $maxParticipants = null,
        float $price = 0.0,
        string $currency = 'RUB',
        ?array $settings = null,
        ?string $metaTitle = null,
        ?string $metaDescription = null,
        ?string $metaKeywords = null,
        int $color = null,
    ): self {
        $event = new self();
        $event->id = Uuid::uuid4()->toString();
        $event->title = $title;
        $event->slug = $slug;
        $event->description = HtmlPurifier::process($description);
        $event->short_description = $shortDescription ? HtmlPurifier::process($shortDescription) : null;
        $event->content = $content ? json_encode($content, JSON_UNESCAPED_UNICODE) : null;
        $event->status = self::STATUS_PUBLISHED;
        $event->type = $type;
        $event->location_type = $locationType;
        $event->location = $location;
        $event->image = $image;
        $event->start_date = $startDate;
        $event->end_date = $endDate;
        $event->registration_start = $registrationStart;
        $event->registration_end = $registrationEnd;
        $event->max_participants = $maxParticipants;
        $event->price = $price;
        $event->currency = $currency;
        $event->user_id = $userId;
        $event->settings = $settings ? json_encode($settings, JSON_UNESCAPED_UNICODE) : null;
        $event->meta_title = $metaTitle;
        $event->meta_description = $metaDescription;
        $event->meta_keywords = $metaKeywords;
        $event->created_at = time();
        $event->updated_at = time();
        $event->color = $color;

        return $event;
    }

    public function edit(
        string $title,
        string $slug,
        string $description,
        ?int $startDate,
        ?int $endDate,
        ?string $shortDescription = null,
        ?array $content = null,
        string $type = self::TYPE_EVENT,
        string $locationType = self::LOCATION_ONLINE,
        ?string $location = null,
        ?string $image = null,
        ?int $registrationStart = null,
        ?int $registrationEnd = null,
        ?int $maxParticipants = null,
        float $price = 0.0,
        string $currency = 'RUB',
        ?array $settings = null,
        ?string $metaTitle = null,
        ?string $metaDescription = null,
        ?string $metaKeywords = null,
        int $color = null,
    ): void {
        $this->title = $title;
        $this->slug = $slug;
        $this->description = HtmlPurifier::process($description);
        $this->short_description = $shortDescription ? HtmlPurifier::process($shortDescription) : null;
        $this->content = $content ? json_encode($content, JSON_UNESCAPED_UNICODE) : null;
        $this->type = $type;
        $this->location_type = $locationType;
        $this->location = $location;
        $this->image = $image;
        $this->start_date = $startDate;
        $this->end_date = $endDate;
        $this->registration_start = $registrationStart;
        $this->registration_end = $registrationEnd;
        $this->max_participants = $maxParticipants;
        $this->price = $price;
        $this->currency = $currency;
        $this->settings = $settings ? json_encode($settings, JSON_UNESCAPED_UNICODE) : null;
        $this->meta_title = $metaTitle;
        $this->meta_description = $metaDescription;
        $this->meta_keywords = $metaKeywords;
        $this->updated_at = time();
        $this->color = $color;
    }

    public function publish(): void
    {
        if ($this->status === self::STATUS_PUBLISHED) {
            throw new DomainException('Event is already published');
        }

        $this->status = self::STATUS_PUBLISHED;
        $this->updated_at = time();
    }

    public function draft(): void
    {
        $this->status = self::STATUS_DRAFT;
        $this->updated_at = time();
    }

    public function cancel(): void
    {
        $this->status = self::STATUS_CANCELLED;
        $this->updated_at = time();
    }

    public function archive(): void
    {
        $this->status = self::STATUS_ARCHIVED;
        $this->updated_at = time();
    }

    public function moveToPending(): void
    {
        $this->status = self::STATUS_PENDING;
        $this->updated_at = time();
    }

    public function isDraft(): bool
    {
        return $this->status === self::STATUS_DRAFT;
    }

    public function isPublished(): bool
    {
        return $this->status === self::STATUS_PUBLISHED;
    }

    public function isCancelled(): bool
    {
        return $this->status === self::STATUS_CANCELLED;
    }

    public function isDeleted(): bool
    {
        return $this->status === self::STATUS_DELETED;
    }

    public function isArchived(): bool
    {
        return $this->status === self::STATUS_ARCHIVED;
    }

    public function isPending(): bool
    {
        return $this->status === self::STATUS_PENDING;
    }

    public function isRegistrationOpen(): bool
    {
        $now = time();

        if ($this->registration_start && $now < $this->registration_start) {
            return false;
        }

        if ($this->registration_end && $now > $this->registration_end) {
            return false;
        }

        return $this->isPublished() && !$this->isCancelled();
    }

    /**
     * @inheritdoc
     */
    public static function tableName(): string
    {
        return '{{%events}}';
    }

    /**
     * @inheritdoc
     */
    public function behaviors(): array
    {
        return [
            TimestampBehavior::class,
            [
                'class' => SluggableBehavior::class,
                'attribute' => 'title',
                'slugAttribute' => 'slug',
            ]
        ];
    }

    public function transactions(): array
    {
        return [
            self::SCENARIO_DEFAULT => self::OP_ALL,
        ];
    }

    /**
     * @return ActiveQuery
     */
    public function getUser(): ActiveQuery
    {
        return $this->hasOne(User::class, ['id' => 'user_id']);
    }

    public static function getStatuses(): array
    {
        return [
            self::STATUS_DRAFT => 'Черновик',
            self::STATUS_PENDING => 'На модерации',
            self::STATUS_PUBLISHED => 'Опубликовано',
            self::STATUS_CANCELLED => 'Отменено',
            self::STATUS_ARCHIVED => 'В архиве',
        ];
    }

    public static function getTypes(): array
    {
        return [
            self::TYPE_EVENT => 'Мероприятие',
            self::TYPE_WEBINAR => 'Вебинар',
            self::TYPE_WORKSHOP => 'Воркшоп',
            self::TYPE_CONFERENCE => 'Конференция',
            self::TYPE_MEETUP => 'Митап',
            self::TYPE_SEMINAR => 'Семинар',
            self::TYPE_FORUM => 'Форум',
            self::TYPE_SUMMIT => 'Саммит',
        ];
    }

    public static function getLocationTypes(): array
    {
        return [
            self::LOCATION_ONLINE => 'Онлайн',
            self::LOCATION_OFFLINE => 'Оффлайн',
            self::LOCATION_HYBRID => 'Гибрид',
        ];
    }

    /**
     * Получение массива emails для уведомлений
     */
    public function getNotifyEmailsArray(): array
    {
        return $this->notify_emails ? json_decode($this->notify_emails, true) : [];
    }

    public function setNotifyEmailsArray($array): void
    {
        $this->notify_emails = json_encode($array, JSON_UNESCAPED_UNICODE);
    }

    /**
     * Участники мероприятия
     */
    public function getMembers(): ActiveQuery
    {
        return $this->hasMany(Member::class, ['event_id' => 'id']);
    }

    /**
     * Кастомные поля мероприятия
     */
    public function getFields(): ActiveQuery
    {
        return $this->hasMany(EventField::class, ['event_id' => 'id'])
            ->orderBy(['sort_order' => SORT_ASC]);
    }

    /**
     * Подсчет участников по статусам
     */
    public function getMembersCountByStatus(): array
    {
        return [
            'total' => $this->getMembers()->count(),
            'pending' => $this->getMembers()->andWhere(['status' => Member::STATUS_PENDING])->count(),
            'confirmed' => $this->getMembers()->andWhere(['status' => Member::STATUS_CONFIRMED])->count(),
            'attended' => $this->getMembers()->andWhere(['status' => Member::STATUS_ATTENDED])->count(),
        ];
    }

    // В классе Event добавляем:
    /**
     * Подтвержденные регистрации
     */
    public function getConfirmedRegistrations(): ActiveQuery
    {
        return $this->getMembers()
            ->andWhere(['status' => Member::STATUS_CONFIRMED]);
    }

    /**
     * Все регистрации (участники)
     */
    public function getRegistrations(): ActiveQuery
    {
        return $this->getMembers();
    }

    /**
     * Получить количество свободных мест
     */
    public function getAvailableSeats(): ?int
    {
        if (!$this->max_participants) {
            return null;
        }

        $confirmedCount = $this->getConfirmedRegistrations()->count();
        $attendedCount = $this->getMembers()
            ->where(['status' => Member::STATUS_ATTENDED])
            ->count();

        return $this->max_participants - $confirmedCount - $attendedCount;
    }

    /**
     * Проверить, есть ли свободные места
     */
    public function hasAvailableSeats(): bool
    {
        $availableSeats = $this->getAvailableSeats();
        return $availableSeats === null || $availableSeats > 0;
    }
}
