<?php

namespace app\core\entities;

use app\core\events\EventTrait;
use app\core\AggregateRoot;
use Ramsey\Uuid\Uuid;
use yii\behaviors\TimestampBehavior;
use yii\db\ActiveQuery;
use yii\db\ActiveRecord;

/**
 * EventField model (кастомное поле формы)
 *
 * @property string $id
 * @property string $event_id
 * @property string $name
 * @property string $label
 * @property string $type
 * @property string $options [jsonb]
 * @property boolean $required
 * @property integer $sort_order
 * @property integer $created_at
 * @property integer $updated_at
 *
 * @property Event $event
 */
class EventField extends ActiveRecord implements AggregateRoot
{
    use EventTrait;

    const string TYPE_TEXT = 'text';
    const string TYPE_EMAIL = 'email';
    const string TYPE_PHONE = 'phone';
    const string TYPE_SELECT = 'select';
    const string TYPE_CHECKBOX = 'checkbox';
    const string TYPE_TEXTAREA = 'textarea';
    const string TYPE_DATE = 'date';
    const string TYPE_FILE = 'file';

    /**
     * Создание кастомного поля
     */
    public static function create(
        string $eventId,
        string $name,
        string $label,
        string $type = self::TYPE_TEXT,
        bool $required = false,
        int $sortOrder = 0,
        ?array $options = null
    ): self {
        $field = new self();
        $field->id = Uuid::uuid4()->toString();
        $field->event_id = $eventId;
        $field->name = $name;
        $field->label = $label;
        $field->type = $type;
        $field->required = $required;
        $field->sort_order = $sortOrder;
        $field->options = $options ? json_encode($options, JSON_UNESCAPED_UNICODE) : null;
        $field->created_at = time();
        $field->updated_at = time();

        return $field;
    }

    /**
     * Редактирование поля
     */
    public function edit(
        string $label,
        string $type,
        bool $required,
        int $sortOrder,
        ?array $options = null
    ): void {
        $this->label = $label;
        $this->type = $type;
        $this->required = $required;
        $this->sort_order = $sortOrder;
        $this->options = $options ? json_encode($options, JSON_UNESCAPED_UNICODE) : null;
        $this->updated_at = time();
    }

    /**
     * Типы полей
     */
    public static function getTypes(): array
    {
        return [
            self::TYPE_TEXT => 'Текстовое поле',
            self::TYPE_EMAIL => 'Email',
            self::TYPE_PHONE => 'Телефон',
            self::TYPE_SELECT => 'Выпадающий список',
            self::TYPE_CHECKBOX => 'Чекбокс',
            self::TYPE_TEXTAREA => 'Текстовая область',
            self::TYPE_DATE => 'Дата',
            self::TYPE_FILE => 'Файл',
        ];
    }

    /**
     * Получение массива options
     */
    public function getOptionsArray(): ?array
    {
        return $this->options ? json_decode($this->options, true) : null;
    }

    /**
     * Проверки типов
     */
    public function isFileType(): bool { return $this->type === self::TYPE_FILE; }
    public function isSelectType(): bool { return $this->type === self::TYPE_SELECT; }

    /**
     * Связи
     */
    public static function tableName(): string { return '{{%event_fields}}'; }

    public function behaviors(): array
    {
        return [TimestampBehavior::class];
    }

    public function getEvent(): ActiveQuery
    {
        return $this->hasOne(Event::class, ['id' => 'event_id']);
    }
}
