<?php

namespace app\core\entities;

use app\core\events\EventTrait;
use app\core\AggregateRoot;
use DomainException;
use Ramsey\Uuid\Uuid;
use Yii;
use yii\behaviors\TimestampBehavior;
use yii\db\ActiveQuery;
use yii\db\ActiveRecord;

/**
 * Member model (участник = регистрация)
 *
 * @property string $id
 * @property string $event_id
 * @property string $first_name
 * @property string $last_name
 * @property string $middle_name
 * @property string $email
 * @property string $phone
 * @property string $company
 * @property string $position
 * @property string $city
 * @property string $country
 * @property string $bio
 * @property string $documents [jsonb]
 * @property string $photo
 * @property integer $status
 * @property string $form_data [jsonb]
 * @property string $notes
 * @property string $registration_ip
 * @property string $user_agent
 * @property integer $registered_at
 * @property integer $confirmed_at
 * @property integer $cancelled_at
 * @property integer $attended_at
 * @property string $settings [jsonb]
 * @property integer $created_at
 * @property integer $updated_at
 *
 * @property Event $event
 */
class Member extends ActiveRecord implements AggregateRoot
{
    use EventTrait;

    const int STATUS_PENDING = 0;
    const int STATUS_CONFIRMED = 10;
    const int STATUS_CANCELLED = 20;
    const int STATUS_ATTENDED = 30;

    /**
     * Создание участника при регистрации
     */
    public static function create(
        string $eventId,
        string $firstName,
        string $lastName,
        ?string $email = null,
        ?string $phone = null,
        ?string $middleName = null,
        ?string $company = null,
        ?string $position = null,
        ?string $city = null,
        ?string $country = null,
        ?array $formData = null,
        ?string $ip = null,
        ?string $userAgent = null,
        bool $isAdminRegistration = false
    ): self {
        $member = new self();
        $member->id = Uuid::uuid4()->toString();
        $member->event_id = $eventId;
        $member->first_name = $firstName;
        $member->last_name = $lastName;
        $member->middle_name = $middleName;
        $member->email = $email;
        $member->phone = $phone;
        $member->company = $company;
        $member->position = $position;
        $member->city = $city;
        $member->country = $country;
        $member->status = self::STATUS_PENDING;
        $member->form_data = $formData ? json_encode($formData, JSON_UNESCAPED_UNICODE) : null;
        $member->registration_ip = $ip;
        $member->user_agent = $userAgent;
        $member->registered_at = time();
        $member->created_at = time();
        $member->updated_at = time();

        if ($isAdminRegistration) {
            $member->confirm(); // Админская регистрация сразу подтверждается
        }

        return $member;
    }

    /**
     * Редактирование данных участника
     */
    public function edit(
        string $firstName,
        string $lastName,
        ?string $email = null,
        ?string $phone = null,
        ?string $middleName = null,
        ?string $company = null,
        ?string $position = null,
        ?string $city = null,
        ?string $country = null,
        ?string $bio = null,
        ?array $documents = null
    ): void {
        $this->first_name = $firstName;
        $this->last_name = $lastName;
        $this->middle_name = $middleName;
        $this->email = $email;
        $this->phone = $phone;
        $this->company = $company;
        $this->position = $position;
        $this->city = $city;
        $this->country = $country;
        $this->bio = $bio;

        if ($documents !== null) {
            $this->documents = json_encode($documents, JSON_UNESCAPED_UNICODE);
        }

        $this->updated_at = time();
    }

    /**
     * Подтверждение регистрации
     */
    public function confirm(): void
    {
        if ($this->status !== self::STATUS_PENDING) {
            throw new DomainException('Можно подтвердить только ожидающую регистрацию');
        }

        $this->status = self::STATUS_CONFIRMED;
        $this->confirmed_at = time();
        $this->updated_at = time();
    }

    /**
     * Отмена регистрации
     */
    public function cancel(): void
    {
        $this->status = self::STATUS_CANCELLED;
        $this->cancelled_at = time();
        $this->updated_at = time();
    }

    /**
     * Отметка присутствия
     */
    public function markAttended(): void
    {
        if ($this->status !== self::STATUS_CONFIRMED) {
            throw new DomainException('Можно отметить только подтвержденную регистрацию');
        }

        $this->status = self::STATUS_ATTENDED;
        $this->attended_at = time();
        $this->updated_at = time();
    }

    public function UnmarkAttended(): void
    {
        if ($this->status !== self::STATUS_ATTENDED) {
            throw new DomainException('Можно отменить только подтвержденную присутствие');
        }

        $this->status = self::STATUS_CONFIRMED;
        $this->attended_at = time();
        $this->updated_at = time();
    }

    /**
     * Установка фото
     */
    public function setPhoto(?string $photo): void
    {
        $this->photo = $photo;
        $this->updated_at = time();
    }

    /**
     * Добавление документа
     */
    public function addDocument(array $document): void
    {
        $documents = $this->getDocumentsArray();
        $documents[] = $document;
        $this->documents = json_encode($documents, JSON_UNESCAPED_UNICODE);
        $this->updated_at = time();
    }

    /**
     * Получение массива документов
     */
    public function getDocumentsArray(): array
    {
        return $this->documents ? json_decode($this->documents, true) : [];
    }

    /**
     * Получение массива form_data
     */
    public function getFormDataArray(): array
    {
        return $this->form_data ? json_decode($this->form_data, true) : [];
    }

    /**
     * Полное имя участника
     */
    public function getFullName(): string
    {
        $parts = array_filter([$this->last_name, $this->first_name, $this->middle_name]);
        return implode(' ', $parts) ?: 'Участник';
    }

    /**
     * Статусы регистрации
     */
    public static function getStatuses(): array
    {
        return [
            self::STATUS_PENDING => 'Ожидает подтверждения',
            self::STATUS_CONFIRMED => 'Подтверждено',
            self::STATUS_CANCELLED => 'Отменено',
            self::STATUS_ATTENDED => 'Присутствовал',
        ];
    }

    /**
     * Проверки статусов
     */
    public function isPending(): bool { return $this->status === self::STATUS_PENDING; }
    public function isConfirmed(): bool { return $this->status === self::STATUS_CONFIRMED; }
    public function isCancelled(): bool { return $this->status === self::STATUS_CANCELLED; }
    public function isAttended(): bool { return $this->status === self::STATUS_ATTENDED; }

    /**
     * Связи
     */
    public static function tableName(): string { return '{{%members}}'; }

    public function behaviors(): array
    {
        return [TimestampBehavior::class];
    }

    public function getEvent(): ActiveQuery
    {
        return $this->hasOne(Event::class, ['id' => 'event_id']);
    }
}
