<?php
namespace app\core\entities;

use app\core\events\EventTrait;
use app\core\AggregateRoot;
use app\core\events\user\UserSmartSignUpRequested;
use DomainException;
use Ramsey\Uuid\Uuid;
use Random\RandomException;
use Yii;
use yii\base\Exception;
use yii\base\InvalidConfigException;
use yii\behaviors\TimestampBehavior;
use yii\db\ActiveQuery;
use yii\db\ActiveRecord;
use yii\helpers\ArrayHelper;
use yii\web\UploadedFile;

/**
 * User model
 *
 * @property string $id
 * @property string $username
 * @property string $password_hash
 * @property string $password_reset_token
 * @property string $email
 * @property string $email_confirm_token
 * @property string $auth_key
 * @property integer $status
 * @property integer $created_at
 * @property integer $updated_at
 * @property string|UploadedFile $user_pic
 * @property string $allowance [integer]
 * @property string $allowance_updated_at [integer]
 * @property string $settings [jsonb]
 * @property string $login_token
 * @property int $login_token_at
 * @property int $tg_id
 * @property int $tg_access
 * @property int $tg_expired
 * @property string $tg_code
 */

class User extends ActiveRecord implements AggregateRoot
{
    use EventTrait;

    const int STATUS_WAIT = 0;
    const int STATUS_DELETED = 1;
    const int STATUS_ACTIVE = 10;

    /**
     * @throws Exception
     */
    public static function create(string $username, string $email, string $password): self
    {
        $user           = new User();
        $user->id       = Uuid::uuid4()->toString();
        $user->username = $username;
        $user->email    = $email;
        $user->setPassword(!empty($password) ? $password : Yii::$app->security->generateRandomString());
        $user->created_at = time();
        $user->status     = self::STATUS_ACTIVE;
        $user->auth_key   = Yii::$app->security->generateRandomString();

        return $user;
    }

    /**
     * @throws Exception
     */
    public function edit(string $username, string $email, string $password): void
    {
        $this->username   = $username;
        $this->email      = $email;
        $this->updated_at = time();

        if ($password) {
            $this->setPassword(!empty($password) ? $password : Yii::$app->security->generateRandomString());
            $this->generateAuthKey();
        }
    }

    /**
     * @param $password
     *
     * @throws Exception
     */
    public function setPassword($password): void
    {
        $this->password_hash = Yii::$app->security->generatePasswordHash($password);
    }

    /**
     * @throws Exception
     */
    public function editProfile(
        string $username,
        string $password = null
    ): void {
        //$this->email    = $email;
        $this->username = $username;
        if ($password) {
            $this->setPassword($password);
        }
        $this->updated_at = time();
    }

    public function requestPasswordResetCode(): void
    {
        if (!empty($this->password_reset_token) && self::isPasswordResetTokenValid($this->password_reset_token)) {
            $timestamp = (int)substr($this->password_reset_token, strrpos($this->password_reset_token, '_') + 1);
            if ($timestamp + 5 * 60 > time()) {
                throw new DomainException('Вы уже недавно запрашивали код восстановления пароля. Повторить запрос можно через ' . (5 * 60 - (time() - $timestamp)) . ' сек.', 10201);
            }
        }

        try {
            $this->password_reset_token = random_int(100000, 999999) . '_' . time();
        } catch (RandomException $e) {
            $this->password_reset_token = '143706_' . time();
        }
    }

    public function resetPassword($password): void
    {
        if (empty($this->password_reset_token)) {
            throw new DomainException('Запрос на сброс пароля не поступал');
        }
        try {
            $this->setPassword($password);
            $this->password_reset_token = null;
        } catch (Exception $e) {
            throw new DomainException('Ошибка сброса пароль');
        }
    }

    public function isWait(): bool
    {
        return $this->status === self::STATUS_WAIT;
    }

    public function isActive(): bool
    {
        return $this->status === self::STATUS_ACTIVE;
    }

    public function isDeleted(): bool
    {
        return $this->status === self::STATUS_DELETED;
    }

    /**
     * @inheritdoc
     */
    public static function tableName(): string
    {
        return '{{%users}}';
    }

    /**
     * @inheritdoc
     */
    public function behaviors(): array
    {
        return [
            [
                'class' => TimestampBehavior::class,
            ]
        ];
    }

    public function transactions(): array
    {
        return [
            self::SCENARIO_DEFAULT => self::OP_ALL,
        ];
    }

    /**
     * Finds user by password reset token
     *
     * @param string $token password reset token
     *
     * @return User|null
     */
    public static function findByPasswordResetToken(string $token): ?static
    {
        if (!static::isPasswordResetTokenValid($token)) {
            return null;
        }

        return static::findOne([
            'password_reset_token' => $token,
        ]);
    }

    /**
     * Finds out if password reset token is valid
     *
     * @param string $token password reset token
     *
     * @return bool
     */
    public static function isPasswordResetTokenValid(string $token): bool
    {
        if (empty($token)) {
            return false;
        }

        $timestamp = (int)substr($token, strrpos($token, '_') + 1);
        $expire    = Yii::$app->params['user.passwordResetTokenExpire'];

        return $timestamp + $expire >= time();
    }

    /**
     * Validates password
     *
     * @param string $password password to validate
     *
     * @return bool if password provided is valid for current user
     */
    public function validatePassword(string $password): bool
    {
        return Yii::$app->security->validatePassword($password, $this->password_hash);
    }

    /**
     * Generates "remember me" authentication key
     * @throws Exception
     */
    public function generateAuthKey(): void
    {
        $this->auth_key = Yii::$app->security->generateRandomString();
    }

    public function isPasswordResetCodeValid(string $code): bool
    {
        if (empty($code)) {
            return false;
        }

        $parts = explode('_', $this->password_reset_token);
        $expire = Yii::$app->params['user.passwordResetTokenExpire'];

        return (int)$parts[1] + $expire >= time() && $parts[0] == $code;
    }

    public function generateLoginToken($time = 0): void
    {
        $this->login_token = Yii::$app->security->generateRandomString(16);
        $this->login_token_at = time() + $time;
    }

    public function clearLoginToken(): void
    {
        $this->login_token = null;
        $this->login_token_at = 0;
    }
}
