<?php

namespace app\core\forms\event;

use yii\base\Model;

class EventCreateForm extends Model
{
    public $title;
    public $slug;
    public $description;
    public $short_description;
    public $content;
    public $type;
    public $location_type;
    public $location;
    public $start_date;
    public $end_date;
    public $registration_start;
    public $registration_end;
    public $max_participants;
    public $price;
    public $currency;
    public $settings;
    public $meta_title;
    public $meta_description;
    public $meta_keywords;
    public $color;

    public function rules(): array
    {
        return [
            [['title', 'type', 'location_type'], 'required'],
            [['title', 'slug', 'location', 'meta_title'], 'string', 'max' => 255],
            ['slug', 'match', 'pattern' => '/^[a-z0-9\-]+$/'],
            ['slug', 'unique', 'targetClass' => \app\core\entities\Event::class],
            [['description', 'short_description', 'meta_description', 'meta_keywords'], 'string'],
            [['short_description'], 'string', 'max' => 500],
            [['start_date', 'end_date', 'registration_start', 'registration_end', 'color'], 'integer'],
            [['max_participants'], 'integer', 'min' => 1],
            ['price', 'number', 'min' => 0],
            ['currency', 'string', 'length' => 3],
            [['type', 'location_type'], 'string'],
            ['type', 'in', 'range' => array_keys(\app\core\entities\Event::getTypes())],
            ['location_type', 'in', 'range' => array_keys(\app\core\entities\Event::getLocationTypes())],
            [['content', 'settings'], 'safe'],
            ['end_date', 'compare', 'compareAttribute' => 'start_date', 'operator' => '>=', 'type' => 'number'],
            ['registration_end', 'compare', 'compareAttribute' => 'registration_start', 'operator' => '>=', 'type' => 'number', 'skipOnEmpty' => true],
        ];
    }

    public function attributeLabels(): array
    {
        return [
            'title' => 'Название',
            'slug' => 'URL',
            'description' => 'Описание',
            'short_description' => 'Краткое описание',
            'content' => 'Контент',
            'type' => 'Тип мероприятия',
            'location_type' => 'Тип локации',
            'location' => 'Место проведения',
            'start_date' => 'Дата начала',
            'end_date' => 'Дата окончания',
            'registration_start' => 'Начало регистрации',
            'registration_end' => 'Окончание регистрации',
            'max_participants' => 'Максимум участников',
            'price' => 'Стоимость',
            'currency' => 'Валюта',
            'meta_title' => 'Meta заголовок',
            'meta_description' => 'Meta описание',
            'meta_keywords' => 'Meta ключевые слова',
            'color' => 'Цвет',
        ];
    }

    public function beforeValidate(): bool
    {
        if (parent::beforeValidate()) {
            // Преобразуем даты из формата Vue в timestamp
            if ($this->start_date && !is_numeric($this->start_date)) {
                $this->start_date = strtotime($this->start_date);
            }
            if ($this->end_date && !is_numeric($this->end_date)) {
                $this->end_date = strtotime($this->end_date);
            }
            if ($this->registration_start && !is_numeric($this->registration_start)) {
                $this->registration_start = strtotime($this->registration_start);
            }
            if ($this->registration_end && !is_numeric($this->registration_end)) {
                $this->registration_end = strtotime($this->registration_end);
            }
            return true;
        }
        return false;
    }
}
