<?php

namespace app\core\forms\event;

use app\core\entities\Event;
use Yii;
use yii\base\Model;

class EventForm extends Model
{
    public $title;
    public $slug;
    public $description;
    public $short_description;
    public $content;
    public $type;
    public $location_type;
    public $location;
    public $start_date;
    public $end_date;
    public $registration_start;
    public $registration_end;
    public $max_participants;
    public $price;
    public $currency;
    public $settings;
    public $meta_title;
    public $meta_description;
    public $meta_keywords;
    public $status;

    public $registration_requires_approval;
    public $registration_notification_emails;
    public $registration_custom_form;
    public $registration_success_message;

    public $color;

    private ?Event $_event;

    public function __construct(?Event $event = null, $config = [])
    {
        if ($event) {
            $this->_event = $event;
            $this->title = $event->title;
            $this->slug = $event->slug;
            $this->description = $event->description;
            $this->short_description = $event->short_description;
            $this->content = $event->content ? json_decode($event->content, true) : null;
            $this->type = $event->type;
            $this->location_type = $event->location_type;
            $this->location = $event->location;
            $this->start_date = $event->start_date;
            $this->end_date = $event->end_date;
            $this->registration_start = $event->registration_start;
            $this->registration_end = $event->registration_end;
            $this->max_participants = $event->max_participants;
            $this->price = $event->price;
            $this->currency = $event->currency;
            $this->settings = $event->settings ? json_decode($event->settings, true) : null;
            $this->meta_title = $event->meta_title;
            $this->meta_description = $event->meta_description;
            $this->meta_keywords = $event->meta_keywords;
            $this->status = $event->status;
            $this->color = $event->color;
        }
        parent::__construct($config);
    }

    public function rules(): array
    {
        return [
            [['title', 'slug'], 'required'],
            [['title', 'slug', 'location', 'meta_title'], 'string', 'max' => 255],
            ['slug', 'match', 'pattern' => '/^[a-z0-9\-]+$/'],
            ['slug', 'unique', 'targetClass' => Event::class, 'filter' => $this->_event ? ['!=', 'id', $this->_event->id] : null],
            [['description', 'short_description', 'meta_description', 'meta_keywords'], 'string'],
            [['short_description'], 'string', 'max' => 500],
            [['start_date', 'end_date', 'registration_start', 'registration_end', 'color'], 'integer'],
            //[['max_participants'], 'integer', 'min' => 1],
            [['max_participants'], 'integer'],
            ['price', 'number', 'min' => 0],
            ['currency', 'string', 'length' => 3],
            [['type', 'location_type'], 'string'],
            ['type', 'in', 'range' => array_keys(Event::getTypes())],
            ['location_type', 'in', 'range' => array_keys(Event::getLocationTypes())],
            ['status', 'in', 'range' => array_keys(Event::getStatuses())],
            [['content', 'settings'], 'safe'],
            ['end_date', 'compare', 'compareAttribute' => 'start_date', 'operator' => '>=', 'type' => 'number'],
            ['registration_end', 'compare', 'compareAttribute' => 'registration_start', 'operator' => '>=', 'type' => 'number', 'skipOnEmpty' => true],

            [['registration_requires_approval', 'registration_custom_form'], 'boolean'],
            ['registration_notification_emails', 'validateEmails'],
            ['registration_success_message', 'string'],
        ];
    }

    public function validateEmails($attribute, $params): void
    {
        if (is_array($this->$attribute)) {
            foreach ($this->$attribute as $index => $email) {
                if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
                    $this->addError($attribute, "Email #{$index} имеет неверный формат");
                }
            }
        }
    }

    public function attributeLabels(): array
    {
        return [
            'title' => 'Название',
            'slug' => 'URL',
            'description' => 'Описание',
            'short_description' => 'Краткое описание',
            'content' => 'Контент',
            'type' => 'Тип мероприятия',
            'location_type' => 'Тип локации',
            'location' => 'Место проведения',
            'start_date' => 'Дата начала',
            'end_date' => 'Дата окончания',
            'registration_start' => 'Начало регистрации',
            'registration_end' => 'Окончание регистрации',
            'max_participants' => 'Максимум участников',
            'price' => 'Стоимость',
            'currency' => 'Валюта',
            'meta_title' => 'Meta заголовок',
            'meta_description' => 'Meta описание',
            'meta_keywords' => 'Meta ключевые слова',
            'status' => 'Статус',
            'registration_requires_approval' => 'Требуется подтверждение регистрации',
            'registration_notification_emails' => 'Email для уведомлений',
            'registration_custom_form' => 'Настраиваемая форма регистрации',
            'registration_success_message' => 'Сообщение после регистрации',
            'color' => 'Цвет',
        ];
    }

    public function beforeValidate(): bool
    {
        if (parent::beforeValidate()) {
            // Преобразуем даты из формата Vue в timestamp
            if ($this->start_date && !is_numeric($this->start_date)) {
                $this->start_date = strtotime($this->start_date);
            }
            if ($this->end_date && !is_numeric($this->end_date)) {
                $this->end_date = strtotime($this->end_date);
            }
            if ($this->registration_start && !is_numeric($this->registration_start)) {
                $this->registration_start = strtotime($this->registration_start);
            }
            if ($this->registration_end && !is_numeric($this->registration_end)) {
                $this->registration_end = strtotime($this->registration_end);
            }

            // Обработка JSON полей
            if (is_array($this->content)) {
                $this->content = json_encode($this->content, JSON_UNESCAPED_UNICODE);
            }
            if (is_array($this->settings)) {
                $this->settings = json_encode($this->settings, JSON_UNESCAPED_UNICODE);
            }

            // Обработка emails
            if (is_array($this->registration_notification_emails)) {
                $this->registration_notification_emails = array_filter($this->registration_notification_emails);
                $this->registration_notification_emails = array_values($this->registration_notification_emails);

                if (!empty($this->registration_notification_emails)) {
                    $this->registration_notification_emails = json_encode(
                        $this->registration_notification_emails,
                        JSON_UNESCAPED_UNICODE
                    );
                } else {
                    $this->registration_notification_emails = null;
                }
            }

            return true;
        }
        return false;
    }
}
