<?php

namespace app\core\forms\event;

use app\core\entities\EventField;
use yii\base\Model;

class MemberCreateForm extends Model
{
    public $event_id;
    public $first_name;
    public $last_name;
    public $middle_name;
    public $email;
    public $phone;
    public $company;
    public $position;
    public $city;
    public $country;
    public $custom_fields = []; // для кастомных полей

    private $_event;
    private $_fields;

    public function __construct($event = null, $config = [])
    {
        if ($event) {
            $this->_event = $event;
            $this->event_id = $event->id;
            $this->_fields = $event->fields;
        }
        parent::__construct($config);
    }

    public function rules(): array
    {
        $rules = [
            [['event_id', 'first_name', 'last_name'], 'required'],
            ['event_id', 'exist', 'targetClass' => \app\core\entities\Event::class, 'targetAttribute' => 'id'],

            [['first_name', 'last_name', 'middle_name'], 'string', 'max' => 100],
            ['email', 'email'],
            ['email', 'string', 'max' => 255],
            ['phone', 'string', 'max' => 50],
            [['company', 'position', 'city', 'country'], 'string', 'max' => 255],

            [['email', 'phone'], 'default', 'value' => null],
            ['custom_fields', 'safe'],
        ];

        // Добавляем правила из кастомных полей если есть
        if ($this->_fields) {
            foreach ($this->_fields as $field) {
                $fieldRules = $this->getFieldRules($field);
                $rules = array_merge($rules, $fieldRules);
            }
        }

        return $rules;
    }

    private function getFieldRules(EventField $field): array
    {
        $rules = [];
        $attribute = "custom_fields[{$field->name}]";

        if ($field->required) {
            $rules[] = [$attribute, 'required', 'message' => "Поле '{$field->label}' обязательно"];
        }

        // Базовые правила по типу поля
        switch ($field->type) {
            case EventField::TYPE_EMAIL:
                $rules[] = [$attribute, 'email'];
                break;
            case EventField::TYPE_PHONE:
                $rules[] = [$attribute, 'match', 'pattern' => '/^[\d\s\-\+\(\)]+$/'];
                break;
            case EventField::TYPE_TEXT:
            case EventField::TYPE_TEXTAREA:
                $rules[] = [$attribute, 'string'];
                break;
            case EventField::TYPE_DATE:
                $rules[] = [$attribute, 'date', 'format' => 'php:Y-m-d'];
                break;
            case EventField::TYPE_SELECT:
                $options = $field->getOptionsArray();
                if ($options) {
                    $rules[] = [$attribute, 'in', 'range' => array_keys($options)];
                }
                break;
            case EventField::TYPE_FILE:
                // Валидация файлов будет в контроллере
                $rules[] = [$attribute, 'safe'];
                break;
        }

        return $rules;
    }

    public function attributeLabels(): array
    {
        $labels = [
            'event_id' => 'Мероприятие',
            'first_name' => 'Имя',
            'last_name' => 'Фамилия',
            'middle_name' => 'Отчество',
            'email' => 'Email',
            'phone' => 'Телефон',
            'company' => 'Компания',
            'position' => 'Должность',
            'city' => 'Город',
            'country' => 'Страна',
            'custom_fields' => 'Дополнительные поля',
        ];

        // Добавляем labels для кастомных полей
        if ($this->_fields) {
            foreach ($this->_fields as $field) {
                $labels["custom_fields[{$field->name}]"] = $field->label;
            }
        }

        return $labels;
    }

    public function beforeValidate(): bool
    {
        if (parent::beforeValidate()) {
            // Очистка телефонного номера
            if ($this->phone) {
                $this->phone = preg_replace('/[^\d\+]/', '', $this->phone);
            }

            // Обработка кастомных полей
            if (is_array($this->custom_fields)) {
                foreach ($this->custom_fields as $key => $value) {
                    if (is_string($value)) {
                        $this->custom_fields[$key] = trim($value);
                    }
                }
            }

            return true;
        }
        return false;
    }

    /**
     * Получение данных для сохранения
     */
    public function getFormData(): array
    {
        $data = [
            'first_name' => $this->first_name,
            'last_name' => $this->last_name,
            'middle_name' => $this->middle_name,
            'email' => $this->email,
            'phone' => $this->phone,
            'company' => $this->company,
            'position' => $this->position,
            'city' => $this->city,
            'country' => $this->country,
        ];

        // Добавляем кастомные поля
        if ($this->custom_fields) {
            foreach ($this->custom_fields as $key => $value) {
                $data[$key] = $value;
            }
        }

        return $data;
    }
}