<?php

namespace app\core\forms\event;

use app\core\entities\Event;
use app\core\entities\EventRegistrationField;
use yii\base\Model;

class RegistrationForm extends Model
{
    public $event_id;
    public $first_name;
    public $last_name;
    public $middle_name;
    public $company;
    public $position;
    public $email;
    public $phone;
    public $custom_fields;

    private ?Event $_event;

    public function __construct(?Event $event = null, $config = [])
    {
        if ($event) {
            $this->_event = $event;
            $this->event_id = $event->id;
        }
        parent::__construct($config);
    }

    public function rules(): array
    {
        $rules = [
            [['event_id', 'first_name', 'last_name', 'email'], 'required'],
            ['event_id', 'exist', 'targetClass' => Event::class, 'targetAttribute' => 'id'],
            [['first_name', 'last_name', 'middle_name'], 'string', 'max' => 100],
            [['company', 'position'], 'string', 'max' => 255],
            ['email', 'email'],
            ['email', 'string', 'max' => 255],
            ['phone', 'string', 'max' => 50],
            ['phone', 'match', 'pattern' => '/^[\d\s\-\+\(\)]+$/'],
            ['custom_fields', 'safe'],
        ];

        // Добавляем правила из кастомных полей если есть
        if ($this->_event && $this->_event->hasCustomForm()) {
            $fields = $this->_event->registrationFields;
            foreach ($fields as $field) {
                $fieldRules = $this->getFieldRules($field);
                $rules = array_merge($rules, $fieldRules);
            }
        }

        return $rules;
    }

    private function getFieldRules(EventRegistrationField $field): array
    {
        $rules = [];

        $attribute = "custom_fields[{$field->name}]";

        if ($field->required) {
            $rules[] = [$attribute, 'required', 'message' => "Поле '{$field->label}' обязательно"];
        }

        // Базовые правила по типу поля
        switch ($field->type) {
            case EventRegistrationField::TYPE_EMAIL:
                $rules[] = [$attribute, 'email'];
                break;
            case EventRegistrationField::TYPE_PHONE:
                $rules[] = [$attribute, 'match', 'pattern' => '/^[\d\s\-\+\(\)]+$/'];
                break;
            case EventRegistrationField::TYPE_TEXT:
            case EventRegistrationField::TYPE_TEXTAREA:
                $rules[] = [$attribute, 'string'];
                break;
            case EventRegistrationField::TYPE_DATE:
                $rules[] = [$attribute, 'date', 'format' => 'php:Y-m-d'];
                break;
            case EventRegistrationField::TYPE_SELECT:
                $options = $field->getOptionsArray();
                if ($options) {
                    $rules[] = [$attribute, 'in', 'range' => array_keys($options)];
                }
                break;
        }

        // Добавляем кастомные правила валидации
        $validationRules = $field->getValidationRulesArray();
        if ($validationRules) {
            foreach ($validationRules as $ruleName => $ruleParams) {
                $rules[] = array_merge([$attribute, $ruleName], (array)$ruleParams);
            }
        }

        return $rules;
    }

    public function attributeLabels(): array
    {
        $labels = [
            'first_name' => 'Имя',
            'last_name' => 'Фамилия',
            'middle_name' => 'Отчество',
            'company' => 'Компания',
            'position' => 'Должность',
            'email' => 'Email',
            'phone' => 'Телефон',
        ];

        // Добавляем labels для кастомных полей
        if ($this->_event && $this->_event->hasCustomForm()) {
            $fields = $this->_event->registrationFields;
            foreach ($fields as $field) {
                $labels["custom_fields[{$field->name}]"] = $field->label;
            }
        }

        return $labels;
    }

    public function beforeValidate(): bool
    {
        if (parent::beforeValidate()) {
            // Очистка телефонного номера
            if ($this->phone) {
                $this->phone = preg_replace('/[^\d\+]/', '', $this->phone);
            }

            // Обработка кастомных полей
            if (is_array($this->custom_fields)) {
                foreach ($this->custom_fields as $key => $value) {
                    if (is_string($value)) {
                        $this->custom_fields[$key] = trim($value);
                    }
                }
            }

            return true;
        }
        return false;
    }
}
