<?php

namespace app\core\repositories\event;

use app\core\dispatchers\EventDispatcher;
use app\core\entities\EventField;
use app\core\repositories\NotFoundException;
use RuntimeException;

class EventFieldRepository
{
    private EventDispatcher $dispatcher;

    public function __construct(EventDispatcher $dispatcher)
    {
        $this->dispatcher = $dispatcher;
    }

    public function get(string $id): EventField
    {
        return $this->getBy(['id' => $id]);
    }

    public function save(EventField $field): void
    {
        if (!$field->isNewRecord) {
            $field->touch('updated_at');
        }

        if (!$field->save()) {
            throw new RuntimeException('Saving error: ' . implode(', ', $field->getFirstErrors()));
        }
        $this->dispatcher->dispatchAll($field->releaseEvents());
    }

    public function remove(EventField $field): void
    {
        if (!$field->delete()) {
            throw new RuntimeException('Removing error.');
        }
        $this->dispatcher->dispatchAll($field->releaseEvents());
    }

    private function getBy(array $condition): EventField
    {
        if (!$field = EventField::find()->andWhere($condition)->limit(1)->one()) {
            throw new NotFoundException('Event field not found.');
        }
        return $field;
    }

    /**
     * Получение полей мероприятия
     */
    public function getByEvent(string $eventId): array
    {
        return EventField::find()
            ->where(['event_id' => $eventId])
            ->orderBy(['sort_order' => SORT_ASC])
            ->all();
    }

    /**
     * Удаление всех полей мероприятия
     */
    public function removeAllByEvent(string $eventId): void
    {
        EventField::deleteAll(['event_id' => $eventId]);
    }

    /**
     * Проверка существования поля с таким именем
     */
    public function existsByName(string $eventId, string $name, ?string $excludeId = null): bool
    {
        $query = EventField::find()
            ->where(['event_id' => $eventId, 'name' => $name]);

        if ($excludeId) {
            $query->andWhere(['!=', 'id', $excludeId]);
        }

        return $query->exists();
    }

    /**
     * Получение следующего порядка сортировки
     */
    public function getNextSortOrder(string $eventId): int
    {
        $maxOrder = EventField::find()
            ->where(['event_id' => $eventId])
            ->max('sort_order');

        return $maxOrder ? $maxOrder + 1 : 0;
    }
}
