<?php

namespace app\core\repositories\event;

use app\core\entities\Event;
use app\core\entities\Member;
use app\core\repositories\NotFoundException;
use RuntimeException;
use yii\data\ActiveDataProvider;

class EventRepository
{
    public function get($id): Event
    {
        return $this->getBy(['id' => $id]);
    }

    public function getBySlug(string $slug): Event
    {
        return $this->getBy(['slug' => $slug]);
    }

    public function findActiveBySlug(string $slug): ?Event
    {
        return Event::findOne(['slug' => $slug, 'status' => Event::STATUS_PUBLISHED]);
    }

    public function save(Event $event): void
    {
        if (!$event->isNewRecord) {
            $event->touch('updated_at');
        }

        if (!$event->save()) {
            throw new RuntimeException('Saving error: ' . implode(', ', $event->getFirstErrors()));
        }
    }

    public function remove(Event $event): void
    {
        if (!$event->delete()) {
            throw new RuntimeException('Removing error.');
        }
    }

    private function getBy(array $condition): Event
    {
        if (!$event = Event::find()->andWhere($condition)->limit(1)->one()) {
            throw new NotFoundException('Event not found.');
        }
        return $event;
    }

    public function getAll(array $filter = [], int $page = 1, int $pageSize = 60): array
    {
        $query = Event::find()->with(['user']);

        // Применение фильтров
        if (!empty($filter)) {
            foreach ($filter as $key => $value) {
                if (empty($value)) continue;

                switch ($key) {
                    case 'title':
                        $query->andWhere(['or',
                            ['like', 'title', $value],
                            //['like', 'description', $value],
                            //['like', 'short_description', $value],
                        ]);
                        break;
                    case 'status':
                        $query->andWhere(['status' => $value]);
                        break;
                    case 'type':
                        $query->andWhere(['type' => $value]);
                        break;
                    case 'location_type':
                        $query->andWhere(['location_type' => $value]);
                        break;
                    case 'user_id':
                        $query->andWhere(['user_id' => $value]);
                        break;
                    case 'date_from':
                        $query->andWhere(['>=', 'start_date', strtotime($value)]);
                        break;
                    case 'date_to':
                        $query->andWhere(['<=', 'end_date', strtotime($value)]);
                        break;
                    default:
                        if (Event::getTableSchema()->getColumn($key)) {
                            $query->andWhere([$key => $value]);
                        }
                }
            }
        }

        $count = $query->count();

        // Пагинация
        if ($page && $pageSize) {
            $query->limit($pageSize);
            $query->offset(($page - 1) * $pageSize);
        }

        $query->orderBy(['start_date' => SORT_DESC]);

        return [
            'provider' => new ActiveDataProvider([
                'query' => $query,
                'pagination' => false,
            ]),
            'count' => $count
        ];
    }

    public function getUpcomingEvents(int $limit = 10): array
    {
        return Event::find()
            ->where(['status' => Event::STATUS_PUBLISHED])
            ->andWhere(['>=', 'start_date', time()])
            ->orderBy(['start_date' => SORT_ASC])
            ->limit($limit)
            ->all();
    }

    public function getPastEvents(int $limit = 10): array
    {
        return Event::find()
            ->where(['status' => Event::STATUS_PUBLISHED])
            ->andWhere(['<', 'start_date', time()])
            ->orderBy(['start_date' => SORT_DESC])
            ->limit($limit)
            ->all();
    }

    public function countByStatus(int $status): int
    {
        return Event::find()
            ->where(['status' => $status])
            ->count();
    }

    public function countActive(): int
    {
        $now = time();
        return Event::find()
            ->where(['status' => Event::STATUS_PUBLISHED])
            ->andWhere(['<=', 'start_date', $now]) // которые начались
            ->andWhere(['>=', 'end_date', $now]) // не закончились
            ->count();
    }

    public function activeTrend(): float|int
    {
        $now = time();

        $activeEvents = $this->countActive();

        $activeEventsLastWeek = Event::find()
            ->where(['status' => Event::STATUS_PUBLISHED])
            ->andWhere(['>=', 'start_date', $now - 604800]) // -7 дней
            ->andWhere(['<', 'start_date', $now])
            ->count();

        return $activeEventsLastWeek > 0
            ? round((($activeEvents - $activeEventsLastWeek) / $activeEventsLastWeek) * 100)
            : ($activeEvents > 0 ? 100 : 0);
    }

    public function upcoming($limit = 5): array
    {
        $now = time();

        $events = Event::find()
            ->alias('e')
            ->select([
                'e.*',
                'COUNT(m.id) as participants_count',
                'SUM(CASE WHEN m.status = :confirmed THEN 1 ELSE 0 END) as confirmed_count',
                'SUM(CASE WHEN m.status = :attended THEN 1 ELSE 0 END) as attended_count',
            ])
            ->leftJoin(['m' => Member::tableName()], 'e.id = m.event_id')
            ->where(['e.status' => Event::STATUS_PUBLISHED])
            //->andWhere(['>=', 'e.start_date', $now])
            ->andWhere(['>=', 'e.end_date', $now])
            ->params([
                ':confirmed' => Member::STATUS_CONFIRMED,
                ':attended' => Member::STATUS_ATTENDED,
            ])
            ->groupBy('e.id')
            ->orderBy(['e.start_date' => SORT_ASC])
            ->limit($limit)
            ->asArray()
            ->all();

        // Добавляем статус мероприятия
        foreach ($events as &$event) {
            $event['status'] = $this->getEventStatus($event['start_date'], $event['end_date'], $now);
        }

        return $events;
    }

    private function getEventStatus($startDate, $endDate, $now): string
    {
        if ($now < $startDate) {
            return 'upcoming';
        } elseif ($now <= $endDate) {
            return 'ongoing';
        } else {
            return 'completed';
        }
    }

    public function existsBySlug(string $slug, ?string $excludeId = null): bool
    {
        $query = Event::find()->where(['slug' => $slug]);
        if ($excludeId) {
            $query->andWhere(['!=', 'id', $excludeId]);
        }
        return $query->exists();
    }

    /**
     * Получение мероприятия с участниками
     */
    public function getWithMembers(string $id): Event
    {
        return Event::find()
            ->with(['members'])
            ->where(['id' => $id])
            ->one();
    }

    /**
     * Получение мероприятия с полями формы
     */
    public function getWithFields(string $id): Event
    {
        return Event::find()
            ->with(['fields'])
            ->where(['id' => $id])
            ->one();
    }

    /**
     * Проверка доступности мест
     */
    public function hasAvailableSeats(string $eventId): bool
    {
        $event = $this->get($eventId);

        if (!$event->max_participants) {
            return true;
        }

        $confirmedCount = Member::find()
            ->where([
                'event_id' => $eventId,
                'status' => [Member::STATUS_CONFIRMED, Member::STATUS_ATTENDED]
            ])
            ->count();

        return $confirmedCount < $event->max_participants;
    }

    /**
     * Получение мероприятий с открытой регистрацией
     */
    public function getEventsWithOpenRegistration(int $limit = 10): array
    {
        $now = time();

        return Event::find()
            ->where(['status' => Event::STATUS_PUBLISHED])
            ->andWhere(['or',
                ['registration_start' => null],
                ['<=', 'registration_start', $now]
            ])
            ->andWhere(['or',
                ['registration_end' => null],
                ['>=', 'registration_end', $now]
            ])
            ->andWhere(['or',
                ['max_participants' => null],
                ['>', 'max_participants', $this->getConfirmedCount('events.id')]
            ])
            ->orderBy(['start_date' => SORT_ASC])
            ->limit($limit)
            ->all();
    }

    public function getByMonth($month, $year): array
    {
        $start_date = strtotime("$year-$month-01 00:00:00");
        $end_date = strtotime("last day of $year-$month 23:59:59");

        return Event::find()
            ->andWhere(['>=', 'start_date', $start_date])
            ->andWhere(['<=', 'end_date', $end_date])
            ->andWhere(['status' => Event::STATUS_PUBLISHED])
            ->all();
    }

    /**
     * Получение количества подтвержденных участников
     */
    private function getConfirmedCount(string $eventIdExpression): string
    {
        return "(SELECT COUNT(*) FROM members WHERE event_id = {$eventIdExpression} AND status IN (" .
            Member::STATUS_CONFIRMED . "," . Member::STATUS_ATTENDED . "))";
    }
}
