<?php

namespace app\core\repositories\event;

use app\core\dispatchers\EventDispatcher;
use app\core\entities\Member;
use app\core\repositories\NotFoundException;
use RuntimeException;
use yii\data\ActiveDataProvider;
use yii\db\ActiveQuery;

class MemberRepository
{
    private EventDispatcher $dispatcher;

    public function __construct(EventDispatcher $dispatcher)
    {
        $this->dispatcher = $dispatcher;
    }

    public function get(string $id): Member
    {
        return $this->getBy(['id' => $id]);
    }

    public function save(Member $member): void
    {
        if (!$member->isNewRecord) {
            $member->touch('updated_at');
        }

        if (!$member->save()) {
            throw new RuntimeException('Saving error: ' . implode(', ', $member->getFirstErrors()));
        }
        $this->dispatcher->dispatchAll($member->releaseEvents());
    }

    public function remove(Member $member): void
    {
        if (!$member->delete()) {
            throw new RuntimeException('Removing error.');
        }
        $this->dispatcher->dispatchAll($member->releaseEvents());
    }

    private function getBy(array $condition): Member
    {
        if (!$member = Member::find()->andWhere($condition)->limit(1)->one()) {
            throw new NotFoundException('Member not found.');
        }
        return $member;
    }

    /**
     * Получение участников мероприятия с фильтрацией
     */
    public function getByEvent(string $eventId, array $filter = [], int $page = 1, int $pageSize = 60): array
    {
        $query = Member::find()->where(['event_id' => $eventId]);

        // Применение фильтров
        if (!empty($filter)) {
            foreach ($filter as $key => $value) {
                if (empty($value)) continue;

                switch ($key) {
                    case 'status':
                        $query->andWhere(['status' => $value]);
                        break;
                    case 'search':
                        $query->andWhere(['or',
                            ['like', 'first_name', $value],
                            ['like', 'last_name', $value],
                            ['like', 'email', $value],
                            ['like', 'company', $value],
                        ]);
                        break;
                    case 'email':
                        $query->andWhere(['email' => $value]);
                        break;
                    case 'phone':
                        $query->andWhere(['phone' => $value]);
                        break;
                    case 'date_from':
                        $query->andWhere(['>=', 'registered_at', strtotime($value)]);
                        break;
                    case 'date_to':
                        $query->andWhere(['<=', 'registered_at', strtotime($value)]);
                        break;
                }
            }
        }

        $count = $query->count();

        // Пагинация
        if ($page && $pageSize) {
            $query->limit($pageSize);
            $query->offset(($page - 1) * $pageSize);
        }

        $query->orderBy(['registered_at' => SORT_DESC]);

        return [
            'provider' => new ActiveDataProvider([
                'query' => $query,
                'pagination' => false,
            ]),
            'count' => $count
        ];
    }

    /**
     * Получение всех участников с фильтрацией
     */
    public function getAll(array $filter = [], int $page = 1, int $pageSize = 60): array
    {
        $query = Member::find()->with(['event']);

        // Применение фильтров
        if (!empty($filter)) {
            foreach ($filter as $key => $value) {
                if (empty($value)) continue;
                switch ($key) {
                    case 'first_name':
                        $query->andWhere(['or',
                            ['like', 'first_name', $value],
                        ]);
                        break;
                    case 'last_name':
                        $query->andWhere(['or',
                            ['like', 'last_name', $value],
                        ]);
                        break;
                    case 'email':
                        $query->andWhere(['or',
                            ['like', 'email', $value],
                        ]);
                        break;
                    case 'company':
                        $query->andWhere(['or',
                            ['like', 'company', $value],
                        ]);
                        break;
                    case 'phone':
                        $phone = preg_replace('/[^\d]/', '', $value);
                        $query->andWhere(['ilike',
                            "regexp_replace(phone, '[^\d]', '', 'g')",
                            '%' . $phone . '%',
                            false
                        ]);

                        //$query->andWhere(['or',
                        //    ['like', 'phone', $value],
                        //]);
                        break;
                    case 'search':
                        $query->andWhere(['or',
                            ['like', 'first_name', $value],
                            ['like', 'last_name', $value],
                            ['like', 'email', $value],
                            ['like', 'company', $value],
                        ]);
                        break;
                    case 'status':
                        $query->andWhere(['status' => $value]);
                        break;
                    case 'event_id':
                        $query->andWhere(['event_id' => $value]);
                        break;
                }
            }
        }

        $count = $query->count();

        // Пагинация
        if ($page && $pageSize) {
            $query->limit($pageSize);
            $query->offset(($page - 1) * $pageSize);
        }

        $query->orderBy(['created_at' => SORT_DESC]);

        return [
            'provider' => new ActiveDataProvider([
                'query' => $query,
                'pagination' => false,
            ]),
            'count' => $count
        ];
    }

    /**
     * Подсчет участников по статусу
     */
    public function countByEventAndStatus(string $eventId, int $status): int
    {
        return Member::find()
            ->where(['event_id' => $eventId, 'status' => $status])
            ->count();
    }

    /**
     * Проверка существования участника на мероприятии
     */
    public function existsForEventAndEmail(string $eventId, ?string $email): bool
    {
        if (!$email) return false;

        return Member::find()
            ->where(['event_id' => $eventId, 'email' => $email])
            ->exists();
    }

    /**
     * Получение статистики по мероприятию
     */
    public function getStats(string $eventId): array
    {
        return [
            'total' => $this->countByEventAndStatus($eventId, Member::STATUS_PENDING) +
                $this->countByEventAndStatus($eventId, Member::STATUS_CONFIRMED) +
                $this->countByEventAndStatus($eventId, Member::STATUS_ATTENDED),
            'pending' => $this->countByEventAndStatus($eventId, Member::STATUS_PENDING),
            'confirmed' => $this->countByEventAndStatus($eventId, Member::STATUS_CONFIRMED),
            'attended' => $this->countByEventAndStatus($eventId, Member::STATUS_ATTENDED),
        ];
    }
}