<?php

namespace app\core\repositories\user;

use app\core\dispatchers\EventDispatcher;
use app\core\entities\User;
use app\core\repositories\NotFoundException;
use Exception;
use RuntimeException;
use Yii;
use yii\data\ActiveDataProvider;
use yii\data\DataProviderInterface;
use yii\db\ActiveQuery;
use yii\db\Expression;
use yii\db\StaleObjectException;

class UserRepository
{
    private EventDispatcher $dispatcher;

    public function __construct(EventDispatcher $dispatcher)
    {
        $this->dispatcher = $dispatcher;
    }

    public function findByUsernameOrEmail($value): ?User
    {
        return User::find()->andWhere(['or', ['username' => $value], ['email' => $value]])->one();
    }

    public function findByEmail($value): ?User
    {
        return User::find()->andWhere(['email' => $value])->one();
    }

    public function findByLoginToken(string $token): ?User
    {
        return User::find()->andWhere(['login_token' => $token])->one();
    }

    public function findByNetworkIdentity($network, $identity): ?User
    {
        return User::find()->joinWith('networks n')->andWhere(['n.network' => $network, 'n.identity' => $identity])->one();
    }

    public function get($id): User
    {
        return $this->getBy(['id' => $id]);
    }

    public function getByEmailConfirmToken($token): User
    {
        return $this->getBy(['email_confirm_token' => $token]);
    }

    public function getByEmail($email): User
    {
        return $this->getBy(['email' => $email]);
    }

    public function getByPasswordResetToken($token): User
    {
        return $this->getBy(['password_reset_token' => $token]);
    }

    public function getByPasswordResetCode(string $code, string $email): User
    {
        return $this->getBy(['password_reset_code' => $code, 'email' => $email]);
    }

    public function existsByPasswordResetToken(string $token): bool
    {
        return (bool) User::findByPasswordResetToken($token);
    }

    public function save(User $user): void
    {
        if (!$user->isNewRecord) {
            $user->touch('updated_at');
        }

        if (!$user->save()) {
            throw new RuntimeException('Saving error.');
        }
        $this->dispatcher->dispatchAll($user->releaseEvents());
    }

    /**
     * @throws StaleObjectException|\Throwable
     */
    public function remove(User $user): void
    {
        if (!$user->delete()) {
            throw new RuntimeException('Removing error.');
        }
        $this->dispatcher->dispatchAll($user->releaseEvents());
    }

    private function getBy(array $condition): User
    {
        if (!$user = User::find()->andWhere($condition)->limit(1)->one()) {
            throw new NotFoundException('Пользователь не найден.');
        }
        return $user;
    }

    public function getAll(): array
    {
        // Изменяем получение параметров:
        $page = filter_var(Yii::$app->request->get('page'), FILTER_VALIDATE_INT) ?: 1;
        $pageSize = filter_var(Yii::$app->request->get('perPage'), FILTER_VALIDATE_INT) ?: 60; // было pageSize

        // Vue3 присылает 'filters', а не 'filter'
        $filter = Yii::$app->request->get('filters') ?: Yii::$app->request->get('filter', []);

        $query = User::find();

        // применение фильтра
        if (is_array($filter) && count(array_keys($filter)) > 0) {
            foreach (array_keys($filter) as $key) {
                if (!$filter[$key]) continue;
                if (!User::getTableSchema()->getColumn($key)) continue;

                if (in_array($key, ['id', 'username'])) {
                    $query->andWhere(['LIKE', $key, $filter[$key]]);
                }
            }
        }

        $count = $query->count();

        if ($page && $pageSize) {
            $query->limit($pageSize);
            $query->offset(($page-1)*$pageSize);
        }

        $query->orderBy(['created_at' => SORT_DESC]);

        return ['provider' => $this->getProvider($query), 'count' => $count];
    }

    public function getAllOrigin(): array
    {
        $page = filter_var(Yii::$app->request->get('page'), FILTER_VALIDATE_INT);
        $pageSize = filter_var(Yii::$app->request->get('pageSize'),FILTER_VALIDATE_INT);
        $filter = Yii::$app->request->get('filter');

        $query = User::find();

        // применение фильтра
        if (is_array($filter) && count(array_keys($filter)) > 0) {
            foreach (array_keys($filter) as $key) {
                // выход, если нет значения
                if (!$filter[$key]) continue;
                // выход, если поля не существует
                if (!User::getTableSchema()->getColumn($key)) continue;

                if (in_array($key, ['id', 'username'])) {
                    $query->andWhere(['LIKE', $key, $filter[$key]]);
                }
                /*elseif ($key == 'status') {
                    if ($filter[$key] == 'check') $query->andWhere(['IN', 'status', [0,1,3]]); // на проверке
                    elseif ($filter[$key] == 'reject') $query->andWhere(['IN', 'status', [5,6]]); // отклонено
                    elseif ($filter[$key] == 'disabled') $query->andWhere(['status' => 2]); // отключено
                    elseif ($filter[$key] == 'enabled') $query->andWhere(['status' => 4]); // отображается
                    elseif ($filter[$key] == 'banned') $query->andWhere(['status' => 7]); // заблокировано
                }*/
            }
        }

        $count = $query->count();

        if ($page && $pageSize) {
            $query->limit($pageSize);
            $query->offset(($page-1)*$pageSize);
        }

        $query->orderBy(['created_at' => SORT_DESC]);

        return ['provider' => $this->getProvider($query), 'count' => $count];
    }

    private function getProvider(ActiveQuery $query): ActiveDataProvider
    {
        $query->andFilterWhere([
            'type' => Yii::$app->request->get('type'),
        ]);

        $provider =  new ActiveDataProvider([
            'query' => $query,
            'sort' => ['defaultOrder' => ['created_at' => SORT_DESC]],
            'pagination' => [
                'pageSizeParam' => 'limit',
                //'pageSize' => 30,
            ]
        ]);
        Yii::$app->response->headers->add('x-total-users-count', $provider->getTotalCount());

        return $provider;
    }

    public function count(): int
    {
        return User::find()
            ->andWhere(['status' => User::STATUS_ACTIVE])
            ->count();
    }

    public function getAllByRange($offset, $limit): array
    {
        return User::find()
            ->andWhere(['status' => User::STATUS_ACTIVE])
            ->orderBy(['created_at' => SORT_ASC])
            ->limit($limit)
            ->offset($offset)
            ->all();
    }

    public function findActiveById(string $id): ?User
    {
        return User::findOne(['id' => $id, 'status' => User::STATUS_ACTIVE]);
    }
}
