<?php

namespace app\core\services;

use Yii;
use yii\base\Component;

class CronMaster extends Component
{
    private array $tasks = [];
    private array $lastRun = [];
    private array $stats = [];
    private array $config = [];
    private array $runningTasks = [];
    private array $taskPids = [];

    public function __construct(array $config = [])
    {
        parent::__construct($config);
        $this->initConfig();
        $this->loadLastRun();
        $this->loadStats();
        $this->loadRunningTasks();
    }

    private function initConfig(): void
    {
        $this->config = array_merge([
            'log_file' => Yii::getAlias('@runtime/logs/cron.log'),
            'stats_file' => Yii::getAlias('@runtime/cron_stats.json'),
            'last_run_file' => Yii::getAlias('@runtime/cron_last_run.json'),
            'running_tasks_file' => Yii::getAlias('@runtime/cron_running.json'),
            'pid_dir' => Yii::getAlias('@runtime/cron/pids'),
            'log_level' => 'INFO',
            'min_interval' => 60,
            'max_execution_time' => 50,
            'log_max_size' => 10 * 1024 * 1024,
            'log_max_files' => 5,
            'task_timeout' => 3600, // 1 час по умолчанию
            'lock_timeout' => 7200, // 2 часа для автоматического снятия блокировки
        ], $this->config);

        // Создаем директорию для PID файлов
        if (!is_dir($this->config['pid_dir'])) {
            mkdir($this->config['pid_dir'], 0755, true);
        }
    }

    /**
     * Загружаем информацию о выполняющихся задачах
     */
    private function loadRunningTasks(): void
    {
        // Из файла с метаданными
        if (file_exists($this->config['running_tasks_file'])) {
            $content = file_get_contents($this->config['running_tasks_file']);
            $this->runningTasks = json_decode($content, true) ?? [];
        }

        // Из PID файлов
        $pidFiles = glob($this->config['pid_dir'] . '/*.pid');
        foreach ($pidFiles as $pidFile) {
            $taskName = pathinfo($pidFile, PATHINFO_FILENAME);
            $pid = (int)file_get_contents($pidFile);

            if ($this->isProcessRunning($pid)) {
                $this->taskPids[$taskName] = $pid;
            } else {
                // Удаляем устаревший PID файл
                @unlink($pidFile);
            }
        }
    }

    /**
     * Проверяет, запущен ли процесс с указанным PID
     */
    private function isProcessRunning(int $pid): bool
    {
        if ($pid <= 0) {
            return false;
        }

        if (function_exists('posix_kill')) {
            return posix_kill($pid, 0);
        }

        // Для Windows
        if (strtoupper(substr(PHP_OS, 0, 3)) === 'WIN') {
            $output = [];
            exec("tasklist /FI \"PID eq $pid\" 2>NUL", $output);
            return count($output) > 1;
        }

        // Для Unix-подобных систем
        exec("ps -p $pid 2>&1", $output, $returnCode);
        return $returnCode === 0;
    }

    /**
     * Проверяет, выполняется ли задача в данный момент
     * @param string $taskName Имя задачи
     * @return bool
     */
    public function isTaskRunning(string $taskName): bool
    {
        // Проверяем по PID файлу
        if (isset($this->taskPids[$taskName])) {
            $pid = $this->taskPids[$taskName];
            if ($this->isProcessRunning($pid)) {
                return true;
            } else {
                // Удаляем неактуальный PID
                unset($this->taskPids[$taskName]);
                $pidFile = $this->config['pid_dir'] . '/' . $taskName . '.pid';
                @unlink($pidFile);
            }
        }

        // Проверяем по метаданным
        if (isset($this->runningTasks[$taskName])) {
            $startTime = $this->runningTasks[$taskName]['start_time'] ?? 0;
            $timeout = $this->runningTasks[$taskName]['timeout'] ?? $this->config['task_timeout'];

            // Если задача запущена слишком давно, считаем её зависшей
            if (time() - $startTime > $timeout) {
                $this->cleanupStuckTask($taskName);
                return false;
            }

            return true;
        }

        return false;
    }

    /**
     * Очистка зависшей задачи
     */
    private function cleanupStuckTask(string $taskName): void
    {
        $this->log("Очистка зависшей задачи: $taskName", 'WARNING');

        // Удаляем PID файл
        $pidFile = $this->config['pid_dir'] . '/' . $taskName . '.pid';
        if (file_exists($pidFile)) {
            $pid = (int)file_get_contents($pidFile);

            // Пытаемся завершить процесс
            if ($this->isProcessRunning($pid)) {
                $this->killProcess($pid);
            }

            @unlink($pidFile);
        }

        // Удаляем из списка выполняющихся
        unset($this->runningTasks[$taskName]);
        unset($this->taskPids[$taskName]);
        $this->saveRunningTasks();
    }

    /**
     * Завершает процесс
     */
    private function killProcess(int $pid, int $signal = 15): bool
    {
        if (!$this->isProcessRunning($pid)) {
            return true;
        }

        if (function_exists('posix_kill')) {
            return posix_kill($pid, $signal);
        }

        if (strtoupper(substr(PHP_OS, 0, 3)) === 'WIN') {
            exec("taskkill /PID $pid /F 2>NUL", $output, $returnCode);
            return $returnCode === 0;
        }

        exec("kill -$signal $pid 2>&1", $output, $returnCode);
        return $returnCode === 0;
    }

    /**
     * Получает информацию о выполняющейся задаче
     * @param string $taskName Имя задачи
     * @return array|null Информация о задаче или null если не выполняется
     */
    public function getTaskInfo(string $taskName): ?array
    {
        if (!$this->isTaskRunning($taskName)) {
            return null;
        }

        $info = $this->runningTasks[$taskName] ?? [];

        if (isset($this->taskPids[$taskName])) {
            $info['pid'] = $this->taskPids[$taskName];
            $info['running_time'] = time() - ($info['start_time'] ?? time());
            $info['memory_usage'] = $this->getProcessMemory($this->taskPids[$taskName]);
        }

        return $info;
    }

    /**
     * Получает использование памяти процессом
     */
    private function getProcessMemory(int $pid): ?int
    {
        if (strtoupper(substr(PHP_OS, 0, 3)) === 'WIN') {
            return null;
        }

        $statusFile = "/proc/$pid/status";
        if (file_exists($statusFile)) {
            $content = file_get_contents($statusFile);
            if (preg_match('/VmRSS:\s*(\d+)\s*kB/', $content, $matches)) {
                return (int)$matches[1] * 1024; // Возвращаем в байтах
            }
        }

        return null;
    }

    /**
     * Прерывает выполнение задачи
     * @param string $taskName Имя задачи
     * @param bool $force Принудительное завершение
     * @return bool Успешно ли завершена задача
     */
    public function stopTask(string $taskName, bool $force = false): bool
    {
        if (!$this->isTaskRunning($taskName)) {
            $this->log("Задача $taskName не выполняется", 'WARNING');
            return false;
        }

        $pidFile = $this->config['pid_dir'] . '/' . $taskName . '.pid';

        if (file_exists($pidFile)) {
            $pid = (int)file_get_contents($pidFile);

            // Сначала мягкий сигнал завершения
            $signal = $force ? 9 : 15; // SIGKILL или SIGTERM

            if ($this->killProcess($pid, $signal)) {
                $this->log("Задача $taskName остановлена" . ($force ? ' принудительно' : ''), 'INFO');

                // Ждем завершения процесса
                for ($i = 0; $i < 10; $i++) {
                    if (!$this->isProcessRunning($pid)) {
                        break;
                    }
                    sleep(1);
                }

                @unlink($pidFile);
                unset($this->taskPids[$taskName]);
                unset($this->runningTasks[$taskName]);
                $this->saveRunningTasks();

                return true;
            }
        }

        // Если PID файла нет, просто очищаем метаданные
        unset($this->runningTasks[$taskName]);
        $this->saveRunningTasks();

        return true;
    }

    /**
     * Останавливает все выполняющиеся задачи
     * @param bool $force Принудительное завершение
     * @return array Статистика остановленных задач
     */
    public function stopAllTasks(bool $force = false): array
    {
        $result = [
            'total' => 0,
            'stopped' => 0,
            'failed' => 0,
            'tasks' => [],
        ];

        $tasksToStop = array_merge(
            array_keys($this->taskPids),
            array_keys($this->runningTasks)
        );
        $tasksToStop = array_unique($tasksToStop);

        foreach ($tasksToStop as $taskName) {
            $result['total']++;

            if ($this->stopTask($taskName, $force)) {
                $result['stopped']++;
                $result['tasks'][$taskName] = 'stopped';
            } else {
                $result['failed']++;
                $result['tasks'][$taskName] = 'failed';
            }
        }

        $this->log("Остановлено задач: {$result['stopped']} из {$result['total']}", 'INFO');

        return $result;
    }

    private function rotateLogIfNeeded(): void
    {
        $logFile = $this->config['log_file'];

        if (!file_exists($logFile)) {
            return;
        }

        $size = filesize($logFile);
        $maxSize = $this->config['log_max_size'];

        if ($size >= $maxSize) {
            $this->rotateLogFile($logFile);
        }
    }

    private function rotateLogFile(string $logFile): void
    {
        $maxFiles = $this->config['log_max_files'];

        // Удаляем самый старый лог
        $oldestLog = $logFile . '.' . $maxFiles;
        if (file_exists($oldestLog)) {
            @unlink($oldestLog);
        }

        // Сдвигаем остальные логи
        for ($i = $maxFiles - 1; $i >= 1; $i--) {
            $oldLog = $logFile . '.' . $i;
            $newLog = $logFile . '.' . ($i + 1);

            if (file_exists($oldLog)) {
                rename($oldLog, $newLog);
            }
        }

        // Переименовываем текущий лог
        $firstBackup = $logFile . '.1';
        rename($logFile, $firstBackup);
    }

    private function writeToLog(string $message, string $level = 'INFO', bool $rotate = true): void
    {
        if ($rotate) {
            $this->rotateLogIfNeeded();
        }

        $timestamp = date('Y-m-d H:i:s');
        $logLine = sprintf("[%s] [%s] %s\n", $timestamp, $level, $message);

        file_put_contents($this->config['log_file'], $logLine, FILE_APPEND);
    }

    public function log(string $message, string $level = 'INFO'): void
    {
        $timestamp = date('Y-m-d H:i:s');
        $logLine = sprintf("[%s] [%s] %s\n", $timestamp, $level, $message);

        $this->writeToLog($message, $level);
        echo $logLine;
    }

    /**
     * Устанавливает задачи
     */
    public function setTasks(array $tasks): void
    {
        $this->tasks = $tasks;
    }

    /**
     * Добавляет задачу с настройкой таймаута
     */
    public function addTask(string $name, array $controller, string $schedule, array $options = []): void
    {
        $this->tasks[$name] = [
            'controller' => $controller,
            'schedule' => $schedule,
            'options' => array_merge([
                'timeout' => $this->config['task_timeout'],
                'lock_timeout' => $this->config['lock_timeout'],
                'enabled' => true,
                'description' => '',
            ], $options),
        ];
    }

    /**
     * Создает PID файл для задачи
     */
    private function createPidFile(string $taskName): ?int
    {
        $pid = getmypid();
        if ($pid === false) {
            return null;
        }

        $pidFile = $this->config['pid_dir'] . '/' . $taskName . '.pid';
        file_put_contents($pidFile, $pid);

        return $pid;
    }

    /**
     * Удаляет PID файл
     */
    private function removePidFile(string $taskName): void
    {
        $pidFile = $this->config['pid_dir'] . '/' . $taskName . '.pid';
        if (file_exists($pidFile)) {
            @unlink($pidFile);
        }
    }

    /**
     * Сохраняет информацию о выполняющихся задачах
     */
    private function saveRunningTasks(): void
    {
        file_put_contents(
            $this->config['running_tasks_file'],
            json_encode($this->runningTasks, JSON_PRETTY_PRINT)
        );
    }

    /**
     * Запускает задачу с отслеживанием
     */
    private function runTrackedTask(string $taskName, callable $taskCallback, array $options = []): bool
    {
        $pid = $this->createPidFile($taskName);

        if ($pid === null) {
            $this->log("Не удалось получить PID для задачи $taskName", 'ERROR');
            return false;
        }

        $this->taskPids[$taskName] = $pid;

        // Сохраняем метаданные о запуске
        $this->runningTasks[$taskName] = [
            'start_time' => time(),
            'pid' => $pid,
            'timeout' => $options['timeout'] ?? $this->config['task_timeout'],
            'command' => implode(' ', $_SERVER['argv'] ?? []),
        ];
        $this->saveRunningTasks();

        // Регистрируем функцию для очистки при завершении
        register_shutdown_function(function () use ($taskName) {
            $this->removePidFile($taskName);
            unset($this->taskPids[$taskName]);
            unset($this->runningTasks[$taskName]);
            $this->saveRunningTasks();
        });

        try {
            $result = call_user_func($taskCallback);

            $this->removePidFile($taskName);
            unset($this->taskPids[$taskName]);
            unset($this->runningTasks[$taskName]);
            $this->saveRunningTasks();

            return $result !== false;

        } catch (\Throwable $e) {
            $this->removePidFile($taskName);
            unset($this->taskPids[$taskName]);
            unset($this->runningTasks[$taskName]);
            $this->saveRunningTasks();

            throw $e;
        }
    }

    private function loadLastRun(): void
    {
        if (file_exists($this->config['last_run_file'])) {
            $content = file_get_contents($this->config['last_run_file']);
            $this->lastRun = json_decode($content, true) ?? [];
        } else {
            $this->lastRun = [];
        }
    }

    private function saveLastRun(string $task): void
    {
        $this->lastRun[$task] = [
            'timestamp' => time(),
            'date' => date('Y-m-d H:i:s')
        ];

        file_put_contents(
            $this->config['last_run_file'],
            json_encode($this->lastRun, JSON_PRETTY_PRINT)
        );
    }

    private function loadStats(): void
    {
        if (file_exists($this->config['stats_file'])) {
            $content = file_get_contents($this->config['stats_file']);
            $this->stats = json_decode($content, true) ?? [];
        } else {
            $this->stats = [];
        }
    }

    private function saveStats(): void
    {
        file_put_contents(
            $this->config['stats_file'],
            json_encode($this->stats, JSON_PRETTY_PRINT)
        );
    }

    public function shouldRun(string $task, string $schedule): bool
    {
        $lastRun = $this->lastRun[$task]['timestamp'] ?? 0;

        if ((time() - $lastRun) < $this->config['min_interval']) {
            $this->log("Задача $task пропущена: запускалась менее {$this->config['min_interval']} секунд назад", 'DEBUG');
            return false;
        }

        if (preg_match('/^(\S+)\s+(\S+)\s+(\S+)\s+(\S+)\s+(\S+)$/', $schedule)) {
            return $this->checkCronSchedule($schedule, $lastRun);
        }

        if (strpos($schedule, 'every ') === 0) {
            return $this->checkIntervalSchedule($schedule, $lastRun);
        } elseif (strpos($schedule, 'at ') === 0) {
            return $this->checkTimeSchedule($schedule, $lastRun);
        } elseif ($schedule === 'always') {
            return true;
        }

        $this->log("Неизвестный формат расписания для задачи $task: $schedule", 'ERROR');
        return false;
    }

    private function checkCronSchedule(string $cronExpression, int $lastRun): bool
    {
        list($minute, $hour, $day, $month, $weekday) = explode(' ', $cronExpression);

        $currentMinute = (int)date('i');
        $currentHour = (int)date('G');
        $currentDay = (int)date('j');
        $currentMonth = (int)date('n');
        $currentWeekday = (int)date('w');

        return $this->cronFieldMatches($minute, $currentMinute) &&
            $this->cronFieldMatches($hour, $currentHour) &&
            $this->cronFieldMatches($day, $currentDay) &&
            $this->cronFieldMatches($month, $currentMonth) &&
            $this->cronFieldMatches($weekday, $currentWeekday);
    }

    private function cronFieldMatches(string $pattern, int $value): bool
    {
        if ($pattern === '*') return true;
        if (preg_match('/^\*\/(\d+)$/', $pattern, $matches)) return $value % (int)$matches[1] === 0;
        if (preg_match('/^(\d+)-(\d+)$/', $pattern, $matches)) return $value >= (int)$matches[1] && $value <= (int)$matches[2];
        if (strpos($pattern, ',') !== false) {
            $values = array_map('intval', explode(',', $pattern));
            return in_array($value, $values, true);
        }
        if (is_numeric($pattern)) return (int)$pattern === $value;
        return false;
    }

    private function checkIntervalSchedule(string $schedule, int $lastRun): bool
    {
        preg_match('/every (\d+) (minute|hour|day)s?/', $schedule, $matches);
        if (!$matches) return false;

        $interval = (int)$matches[1];
        $unit = rtrim($matches[2], 's');

        $multipliers = ['minute' => 60, 'hour' => 3600, 'day' => 86400];
        $intervalSeconds = $interval * ($multipliers[$unit] ?? 60);

        return (time() - $lastRun) >= $intervalSeconds;
    }

    private function checkTimeSchedule(string $schedule, int $lastRun): bool
    {
        preg_match('/at (\d{2}:\d{2})/', $schedule, $matches);
        if (!$matches) return false;

        $targetTime = $matches[1];
        $lastRunDate = $lastRun ? date('Y-m-d', $lastRun) : '';
        $today = date('Y-m-d');

        return ($lastRunDate !== $today) && (date('H:i') === $targetTime);
    }

    /**
     * Выполняет задачу с отслеживанием
     */
    public function executeTask(string $taskName, callable $taskCallback, array $options = []): bool
    {
        // Проверяем, не выполняется ли уже задача
        if ($this->isTaskRunning($taskName)) {
            $info = $this->getTaskInfo($taskName);
            $runningTime = $info['running_time'] ?? 0;

            $this->log("⚠️ Задача $taskName уже выполняется (PID: {$info['pid']}, время: {$runningTime}сек)", 'WARNING');
            return false;
        }

        $startTime = microtime(true);
        $memoryStart = memory_get_usage(true);

        $this->log("Запуск задачи: $taskName", 'INFO');

        try {
            // Запускаем задачу с отслеживанием
            $success = $this->runTrackedTask($taskName, function () use ($taskCallback, $taskName) {
                return call_user_func($taskCallback);
            }, $options);

            if ($success) {
                $duration = round(microtime(true) - $startTime, 3);
                $memoryUsed = round((memory_get_peak_usage(true) - $memoryStart) / 1024 / 1024, 2);

                $this->saveLastRun($taskName);

                if (!isset($this->stats[$taskName])) {
                    $this->stats[$taskName] = [
                        'total_runs' => 0,
                        'successful_runs' => 0,
                        'failed_runs' => 0,
                        'last_success' => null,
                        'last_error' => null,
                    ];
                }

                $this->stats[$taskName]['total_runs']++;
                $this->stats[$taskName]['successful_runs']++;
                $this->stats[$taskName]['last_success'] = date('Y-m-d H:i:s');
                $this->saveStats();

                $this->log("✅ Задача $taskName завершена за {$duration}сек, память: {$memoryUsed}MB", 'INFO');
                return true;
            } else {
                $this->log("❌ Задача $taskName завершилась с ошибкой", 'ERROR');
                return false;
            }

        } catch (\Throwable $e) {
            $duration = round(microtime(true) - $startTime, 3);

            if (isset($this->stats[$taskName])) {
                $this->stats[$taskName]['total_runs']++;
                $this->stats[$taskName]['failed_runs']++;
                $this->stats[$taskName]['last_error'] = [
                    'message' => $e->getMessage(),
                    'time' => date('Y-m-d H:i:s')
                ];
                $this->saveStats();
            }

            $this->log("❌ Ошибка в задаче $taskName: " . $e->getMessage(), 'ERROR');
            return false;
        }
    }

    public function runScheduledTasks(): array
    {
        // Сначала немедленные задачи
        $this->runImmediateTasks();

        $startTime = time();
        $this->log("[" . date('Y-m-d H:i:s') . "] Запуск крона. Задач: " . count($this->tasks), 'INFO');

        $result = [
            'executed' => 0,
            'skipped' => 0,
            'errors' => 0,
        ];

        foreach ($this->tasks as $taskName => $taskConfig) {
            try {
                // Проверяем, включена ли задача
                if (isset($taskConfig['options']['enabled']) && !$taskConfig['options']['enabled']) {
                    $this->log("Задача $taskName отключена", 'DEBUG');
                    $result['skipped']++;
                    continue;
                }

                if ($this->shouldRun($taskName, $taskConfig['schedule'])) {
                    $callback = $taskConfig['controller'];
                    $options = $taskConfig['options'] ?? [];

                    if ($this->executeTask($taskName, $callback, $options)) {
                        $result['executed']++;
                    } else {
                        $result['errors']++;
                    }
                } else {
                    $result['skipped']++;
                }
            } catch (\Throwable $e) {
                $result['errors']++;
                $this->log("❌ Ошибка в задаче {$taskName}: " . $e->getMessage(), 'ERROR');
            }
        }

        $totalDuration = time() - $startTime;
        $this->log("[" . date('Y-m-d H:i:s') . "] Крон завершен. Выполнено: {$result['executed']}, Пропущено: {$result['skipped']}, Ошибок: {$result['errors']}, Время: {$totalDuration}сек", 'INFO');

        return $result;
    }

    public function runSingleTask(string $taskName, $params_json = null): bool
    {
        //file_put_contents(Yii::getAlias('@runtime/asasasas.txt'), print_r([$taskName, $params_json], true));
        $this->log("Принудительный запуск задачи: {$taskName}", 'INFO');

        if (!isset($this->tasks[$taskName])) {
            //file_put_contents(Yii::getAlias('@runtime/failfail.txt'), print_r([$taskName, $params_json], true));
            $this->log("❌ Задача '{$taskName}' не найдена", 'ERROR');
            return false;
        }

        $taskConfig = $this->tasks[$taskName];

        // Проверяем, включена ли задача
        if (
            isset($taskConfig['options']['enabled']) &&
            !$taskConfig['options']['enabled'] &&
            (
                !isset($taskConfig['options']['no_auto']) ||
                !$taskConfig['options']['no_auto']
            )
        ) {
//            file_put_contents(Yii::getAlias('@runtime/errrorrr.txt'), print_r([
//                $taskName,
//                $params_json,
//                $taskConfig
//            ], true));
            $this->log("❌ Задача '{$taskName}' отключена", 'ERROR');
            return false;
        }
//        if (
//            isset($taskConfig['options']['enabled']) &&
//            !$taskConfig['options']['enabled'] &&
//            (
//                !isset($taskConfig['options']['no_auto']) ||
//                !$taskConfig['options']['no_auto'])
//        ) {
//            file_put_contents(Yii::getAlias('@runtime/errrorrr.txt'), print_r([$taskName, $params_json], true));
//            $this->log("❌ Задача '{$taskName}' отключена", 'ERROR');
//            return false;
//        }

        $callback = $taskConfig['controller'];
        $options = $taskConfig['options'] ?? [];

        //return $this->executeTask($taskName, $callback($params_json), $options);

        //file_put_contents(Yii::getAlias('@runtime/ttttttttt.txt'), print_r($callback, true));

        // Если callback - это массив [controller, action], то создаем функцию-обертку
        if (is_array($callback) && count($callback) >= 2) {
            $controllerCallback = function () use ($callback, $params_json) {
                [$controllerClass, $action] = $callback;

                // Если есть дополнительные параметры
                $params = [];
                if (isset($callback[2])) {
                    $params = $callback[2];
                }

                // Добавляем JSON параметры, если они есть
                if ($params_json !== null && json_validate($params_json)) {
                    $jsonParams = json_decode($params_json, true);
                    if (is_array($jsonParams)) {
                        $params = array_merge($params, $jsonParams);
                    }
                }

                // Создаем экземпляр контроллера и вызываем действие
                $controller = new $controllerClass();
                return $controller->$action(...$params);
            };

            file_put_contents(Yii::getAlias('@runtime/ededed1.txt'), print_r($options, true));

            return $this->executeTask($taskName, $controllerCallback, $options);
        }

        // Если callback - это уже callable (замыкание или функция)
        if (is_callable($callback)) {
            $wrappedCallback = function () use ($callback, $params_json) {
                // Если переданы JSON параметры, пробуем их использовать
                if ($params_json !== null && json_validate($params_json)) {
                    $params = json_decode($params_json, true);

                    file_put_contents(Yii::getAlias('@runtime/ededed3.txt'), print_r($params, true));
                    return call_user_func($callback, $params_json);
                    // Записываем ВСЮ информацию в ededed3.txt
//                    file_put_contents(Yii::getAlias('@runtime/ededed3.txt'),
//                        print_r([
//                            'TIMESTAMP' => date('Y-m-d H:i:s'),
//                            'PARAMS_JSON' => $params_json,
//                            'PARAMS_DECODED' => $params,
//                            'CALLBACK' => $callback,
//                            'CALLBACK_TYPE' => gettype($callback),
//                            'IS_CLOSURE' => ($callback instanceof \Closure) ? 'YES' : 'NO',
//                            'IS_ARRAY' => is_array($callback) ? 'YES, length: ' . count($callback) : 'NO',
//                            'IS_STRING' => is_string($callback) ? 'YES: ' . $callback : 'NO',
//                            'CALLABLE_CHECK' => 'PASSED',
//                            'MEMORY_USAGE' => memory_get_usage() . ' bytes',
//                            'BACKTRACE' => debug_backtrace(DEBUG_BACKTRACE_IGNORE_ARGS, 3)
//                        ], true) . "\n\n" .
//                        "==========================================\n\n",
//                        FILE_APPEND);
//
//                    // Пробуем вызвать callback и записать результат
//                    try {
//                        $result = call_user_func($callback, $params_json);
//
//                        // Записываем результат вызова
//                        file_put_contents(Yii::getAlias('@runtime/ededed3.txt'),
//                            "\n\nCALLBACK RESULT: " . print_r($result, true) . "\n" .
//                            "CALLBACK EXECUTED SUCCESSFULLY\n" .
//                            "==========================================\n\n",
//                            FILE_APPEND);
//
//                        return $result;
//                    } catch (\Throwable $e) {
//                        // Записываем ошибку
//                        file_put_contents(Yii::getAlias('@runtime/ededed3.txt'),
//                            "\n\nCALLBACK ERROR:\n" .
//                            "Message: " . $e->getMessage() . "\n" .
//                            "File: " . $e->getFile() . ":" . $e->getLine() . "\n" .
//                            "Trace:\n" . $e->getTraceAsString() . "\n" .
//                            "==========================================\n\n",
//                            FILE_APPEND);
//                        throw $e;
//                    }
                }
                return call_user_func($callback);
            };

            //file_put_contents(Yii::getAlias('@runtime/ededed2.txt'), print_r($options, true));

            return $this->executeTask($taskName, $wrappedCallback, $options);
        }

        $this->log("❌ Некорректный callback для задачи '{$taskName}'", 'ERROR');
        return false;
    }

    public function listTasks(): array
    {
        $list = [];

        foreach ($this->tasks as $taskName => $taskConfig) {
            $lastRun = $this->lastRun[$taskName]['date'] ?? 'никогда';
            $totalRuns = $this->stats[$taskName]['total_runs'] ?? 0;
            $isRunning = $this->isTaskRunning($taskName);
            $isEnabled = $taskConfig['options']['enabled'] ?? true;

            $list[$taskName] = [
                'schedule' => $taskConfig['schedule'],
                'controller' => $taskConfig['controller'],
                'last_run' => $lastRun,
                'total_runs' => $totalRuns,
                'is_running' => $isRunning,
                'is_enabled' => $isEnabled,
                'description' => $taskConfig['options']['description'] ?? '',
            ];
        }

        return $list;
    }

    public function showLogs(int $lines = 50): array
    {
        $logFile = $this->config['log_file'];

        if (!file_exists($logFile)) {
            return [];
        }

        $content = file($logFile, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
        return array_slice($content, -$lines);
    }

    public function cleanupLogs(): void
    {
        $logFile = $this->config['log_file'];
        $maxFiles = $this->config['log_max_files'];

        // Удаляем архивные логи сверх лимита
        for ($i = $maxFiles + 1; $i <= 20; $i++) {
            $oldLog = $logFile . '.' . $i;
            if (file_exists($oldLog)) {
                @unlink($oldLog);
            }
        }

        // Очищаем текущий лог если он слишком большой
        if (file_exists($logFile) && filesize($logFile) > $this->config['log_max_size']) {
            $this->rotateLogFile($logFile);
        }
    }

    /**
     * Получает список выполняющихся задач
     * @return array
     */
    public function getRunningTasks(): array
    {
        $running = [];

        foreach (array_keys($this->tasks) as $taskName) {
            if ($this->isTaskRunning($taskName)) {
                $running[$taskName] = $this->getTaskInfo($taskName);
            }
        }

        return $running;
    }

    /**
     * Получает список зависших задач
     * @return array
     */
    public function getStuckTasks(): array
    {
        $stuck = [];

        foreach ($this->runningTasks as $taskName => $taskInfo) {
            $startTime = $taskInfo['start_time'] ?? 0;
            $timeout = $taskInfo['timeout'] ?? $this->config['task_timeout'];

            if (time() - $startTime > $timeout) {
                $stuck[$taskName] = $taskInfo;
            }
        }

        return $stuck;
    }

    /**
     * Очищает все зависшие задачи
     * @return array Статистика очистки
     */
    public function cleanupAllStuckTasks(): array
    {
        $stuckTasks = $this->getStuckTasks();
        $result = [
            'total' => count($stuckTasks),
            'cleaned' => 0,
            'failed' => 0,
            'tasks' => [],
        ];

        foreach ($stuckTasks as $taskName => $taskInfo) {
            try {
                $this->cleanupStuckTask($taskName);
                $result['cleaned']++;
                $result['tasks'][$taskName] = 'cleaned';
            } catch (\Throwable $e) {
                $result['failed']++;
                $result['tasks'][$taskName] = 'failed';
            }
        }

        return $result;
    }

    public function addImmediateTask(string $taskName, ?array $params = []): void
    {
        $flagFile = Yii::getAlias('@runtime/cron_immediate_' . $taskName . '.flag');
        //$flagFile2 = Yii::getAlias('@runtime/11-cron_immediate_' . $taskName . '.flag');

        if (!empty($params)) {
            // Сохраняем параметры в JSON
            file_put_contents($flagFile, json_encode($params, JSON_UNESCAPED_UNICODE));
            //file_put_contents($flagFile2, json_encode($params, JSON_UNESCAPED_UNICODE));
        } else {
            // Создаем пустой файл
            touch($flagFile);
        }
        //$this->log("Добавлена немедленная задача: {$taskName} с параметрами: " . json_encode($params), 'INFO');
    }

    public function runImmediateTasks(): void
    {
        $files = glob(Yii::getAlias('@runtime/cron_immediate_*.flag'));

        foreach ($files as $file) {

            $json_string = file_get_contents($file);
            $json_params = null;

            // Пробуем декодировать JSON параметры
            if ($json_string !== false && !empty($json_string)) {
                if (json_validate($json_string)) {
                    $json_params = $json_string;
                }
            }

            $taskName = str_replace(['cron_immediate_', '.flag'], '', basename($file));

            //file_put_contents(Yii::getAlias('@runtime/fffff.txt'), print_r([$file, $json_params], true), FILE_APPEND);

            if (isset($this->tasks[$taskName])) {
                $this->runSingleTask($taskName, $json_params);
            }

            @unlink($file); // Удаляем флаг
        }
    }

    /**
     * Проверяет, есть ли флаг для немедленного запуска задачи
     */
    public function hasImmediateTask(string $taskName): bool
    {
        $pattern = Yii::getAlias('@runtime/cron_immediate_' . $taskName . '.flag');
        $files = glob($pattern);
        return !empty($files);
    }

    /**
     * Проверяет, есть ли хотя бы один флаг
     */
    public function hasAnyImmediateTasks(): bool
    {
        $files = glob(Yii::getAlias('@runtime/cron_immediate_*.flag'));
        return !empty($files);
    }

    /**
     * Получает список всех задач с флагами
     */
    public function getImmediateTasks(): array
    {
        $tasks = [];
        $files = glob(Yii::getAlias('@runtime/cron_immediate_*.flag'));

        foreach ($files as $file) {
            $taskName = str_replace(['cron_immediate_', '.flag'], '', basename($file));
            $created = filemtime($file);
            $age = time() - $created;

            $tasks[] = [
                'task' => $taskName,
                'file' => $file,
                'created' => date('Y-m-d H:i:s', $created),
                'age_seconds' => $age,
            ];
        }

        return $tasks;
    }
}
