<?php

namespace app\core\services\backup;

use Exception;
use FilesystemIterator;
use RecursiveDirectoryIterator;
use RecursiveIteratorIterator;
use Yii;
use yii\base\Component;
use yii\helpers\FileHelper;
use yii\db\Connection;
use ZipArchive;

class BackupManager03 extends Component
{
    public string $backupPath = '@app/backups';
    public $maxBackups = 50;
    public $maxSqlBackups = 30;
    public $maxFullBackups = 7;

    // Исключения для полного бэкапа
    public $excludePatterns = [
        // Системные папки которые не нужно бэкапить
        '/^.git\//',                 // GIT
        '/^tests\//',                // Vue3
        '/^docker\//',               // Docker
        '/^backups\//',              // Сами бэкапы
        '/^vue3\//',                 // Vue3
        '/^node_modules\//',         // Node.js модули
        '/^vendor\/[^\/]+\/[^\/]+\/tests?\//', // Тесты вендоров
        '/^vendor\/[^\/]+\/[^\/]+\/docs?\//',  // Документация вендоров
        '/^runtime\/cache\//',       // Кэш
        '/^runtime\/debug\//',       // Debug данные
        '/^\.idea\//',               // IDE файлы
        '/^\.vscode\//',             // VSCode
        '/^\.DS_Store$/',            // Mac системные файлы
        '/^Thumbs\.db$/',            // Windows системные файлы
        '/\.bak$/',                  // Резервные файлы
        '/\.tmp$/',                  // Временные файлы
        '/\.swp$/',                  // Vim swap файлы
    ];

    /**
     * Создать новый бэкап
     */
    public function createBackup(string $type, string $description = null, string $createdBy = null): array
    {
        if (!in_array($type, ['sql', 'full'])) {
            throw new \InvalidArgumentException("Неправильный тип бэкапа: {$type}. Допустимо: sql, full");
        }

        $timestamp = date('Y-m-d_H-i-s');
        $backupId = "backup_{$timestamp}_{$type}";
        $backupDir = Yii::getAlias("{$this->backupPath}/{$backupId}");

        FileHelper::createDirectory($backupDir);

        $manifest = [
            'id' => $backupId,
            'type' => $type,
            'description' => $description,
            'created_at' => date('c'),
            'created_by' => $createdBy,
            'system_info' => $this->getSystemInfo(),
            'excluded_patterns' => $this->excludePatterns,
        ];

        try {
            // Создаем бэкап БД (для обоих типов)
            $manifest['database'] = $this->backupDatabase($backupDir);

            // Архивируем файлы проекта (только для FULL)
            if ($type === 'full') {
                $manifest['project_files'] = $this->backupProjectFiles($backupDir);
            }

            // Сохраняем манифест
            $manifestFile = "{$backupDir}/manifest.json";
            file_put_contents($manifestFile,
                json_encode($manifest, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE)
            );

            // Ротация
            $deletedBackups = $this->rotateBackups();
            $totalSize = $this->calculateDirectorySize($backupDir);

            return [
                'success' => true,
                'backup_id' => $backupId,
                'path' => $backupDir,
                'manifest' => $manifest,
                'deleted_old_backups' => $deletedBackups,
                'total_size' => $totalSize,
                'total_size_formatted' => $this->formatBytes($totalSize),
            ];

        } catch (Exception $e) {
            if (file_exists($backupDir)) {
                FileHelper::removeDirectory($backupDir);
            }
            throw $e;
        }
    }

    /**
     * Создать дамп БД
     */
    private function backupDatabase(string $backupDir): array
    {
        $db = Yii::$app->db;
        $dumpFile = "{$backupDir}/database.sql";

        if ($db->driverName !== 'pgsql') {
            throw new Exception("Поддерживается только PostgreSQL");
        }

        $dumpContent = $this->generateSqlDump($db);

        if (file_put_contents($dumpFile, $dumpContent) === false) {
            throw new Exception("Не удалось записать дамп БД в файл");
        }

        $tablesCount = $this->countTables($db);

        return [
            'file' => 'database.sql',
            'size' => filesize($dumpFile),
            'tables_count' => $tablesCount,
            'encoding' => 'UTF-8',
            'generated_by' => 'php',
        ];
    }

    /**
     * Генерация SQL дампа средствами PHP
     */
    private function generateSqlDump(Connection $db): string
    {
        $output = "";

        // Простой заголовок
        $output .= "-- PostgreSQL database dump\n";
        $output .= "-- Generated by BackupManager\n";
        $output .= "-- Date: " . date('Y-m-d H:i:s') . "\n\n";

        // Получаем все таблицы
        $tables = $this->getAllTables($db);

        // Сначала создаем все таблицы (без данных)
        foreach ($tables as $table) {
            try {
                $createSql = $this->getTableCreateSql($db, $table);
                if ($createSql) {
                    $output .= $createSql . ";\n\n";
                }
            } catch (\Exception $e) {
                $output .= "-- Error creating table {$table}: " . $e->getMessage() . "\n\n";
            }
        }

        // Затем вставляем данные
        foreach ($tables as $table) {
            try {
                $dataSql = $this->getTableDataSql($db, $table);
                if (!empty(trim($dataSql))) {
                    $output .= $dataSql . "\n";
                }
            } catch (\Exception $e) {
                $output .= "-- Error inserting data into {$table}: " . $e->getMessage() . "\n\n";
            }
        }

        return $output;
    }

//    private function findProjectRoot(): string
//    {
//        // Начинаем с app директории и поднимаемся вверх
//        $currentDir = realpath(\Yii::getAlias('@app'));
//
//        while ($currentDir && $currentDir !== '/') {
//            $composerFile = $currentDir . '/composer.json';
//            if (file_exists($composerFile)) {
//                return $currentDir;
//            }
//            $currentDir = dirname($currentDir);
//        }
//
//        // Если composer.json не найден, используем директорию app
//        return realpath(\Yii::getAlias('@app'));
//    }

    public function lastBackupsDate(): ?int
    {
        $backupPath = Yii::getAlias($this->backupPath);

        if (!is_dir($backupPath)) {
            return null;
        }

        return \app\core\helpers\FileHelper::getNewestFolderDate($backupPath);
    }

    public function sizeBackups()
    {
        $backupPath = Yii::getAlias($this->backupPath);

        if (!is_dir($backupPath)) {
            return 0;
        }

        $size = 0;
        $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($backupPath, FilesystemIterator::SKIP_DOTS)
        );

        foreach ($iterator as $file) {
            if ($file->isFile()) {
                $size += $file->getSize();
            }
        }

        return $size;
    }

    /**
     * Бэкап всех файлов проекта (кроме исключенных)
     */
    private function backupProjectFiles(string $backupDir): array
    {
        // Определяем корень проекта по composer.json
        $projectRoot = realpath(Yii::getAlias('@app'));
        if (!$projectRoot) {
            throw new \Exception("Не удалось определить корневую директорию проекта (composer.json не найден)");
        }

        $archiveFile = "{$backupDir}/project.zip";

        // Создаем ZIP архив
        $zip = new \ZipArchive();

        if ($zip->open($archiveFile, \ZipArchive::CREATE | \ZipArchive::OVERWRITE) !== true) {
            throw new \Exception("Не удалось создать ZIP архив");
        }

        $filesCount = 0;
        $totalSize = 0;
        $errors = [];

        // Функция для рекурсивного сканирования с обработкой ошибок
        $scanDirectory = function($directory, $basePath) use (&$scanDirectory, &$zip, &$filesCount, &$totalSize, &$errors, $projectRoot) {
            try {
                $items = scandir($directory);
            } catch (\Exception $e) {
                $errors[] = "Не удалось прочитать директорию {$directory}: " . $e->getMessage();
                return;
            }

            foreach ($items as $item) {
                if ($item === '.' || $item === '..') {
                    continue;
                }

                $fullPath = $directory . '/' . $item;
                $relativePath = substr($fullPath, strlen($projectRoot) + 1);

                // Проверяем на исключения
                if ($this->shouldExcludeFile($relativePath)) {
                    continue;
                }

                try {
                    if (is_dir($fullPath)) {
                        // Рекурсивно сканируем поддиректорию
                        $scanDirectory($fullPath, $basePath);
                    } elseif (is_file($fullPath)) {
                        if ($zip->addFile($fullPath, $relativePath)) {
                            $filesCount++;
                            $totalSize += filesize($fullPath);
                        }
                    }
                } catch (\Exception $e) {
                    $errors[] = "Ошибка обработки {$fullPath}: " . $e->getMessage();
                    continue;
                }
            }
        };

        // Запускаем сканирование
        $scanDirectory($projectRoot, $projectRoot);

        // Добавляем README с информацией о бэкапе
        $readmeContent = "FULL BACKUP\n";
        $readmeContent .= "Created: " . date('Y-m-d H:i:s') . "\n";
        $readmeContent .= "Project root: " . $projectRoot . "\n";
        $readmeContent .= "Files: {$filesCount}\n";
        $readmeContent .= "Total size: " . $this->formatBytes($totalSize) . "\n";
        $readmeContent .= "Excluded patterns:\n";
        foreach ($this->excludePatterns as $pattern) {
            $readmeContent .= "- " . $pattern . "\n";
        }

        if (!empty($errors)) {
            $readmeContent .= "\nWarnings/Errors (" . count($errors) . "):\n";
            foreach (array_slice($errors, 0, 10) as $error) {
                $readmeContent .= "- " . substr($error, 0, 200) . "\n";
            }
        }

        $zip->addFromString('BACKUP_INFO.txt', $readmeContent);

        $zip->close();

        if (!file_exists($archiveFile)) {
            throw new \Exception("Архив создан, но файл отсутствует");
        }

        return [
            'file' => 'project.zip',
            'size' => filesize($archiveFile),
            'files_count' => $filesCount,
            'original_size' => $totalSize,
            'compression_ratio' => $totalSize > 0 ? round(filesize($archiveFile) / $totalSize * 100, 2) : 0,
            'format' => 'zip',
            'project_root' => $projectRoot,
            'errors_count' => count($errors),
            'errors' => array_slice($errors, 0, 5),
        ];
    }


    /**
     * Проверить, нужно ли исключить файл
     */
    private function shouldExcludeFile(string $relativePath): bool
    {
        // Преобразуем путь для универсальности
        $path = str_replace('\\', '/', $relativePath);

        foreach ($this->excludePatterns as $pattern) {
            if (preg_match($pattern, $path)) {
                return true;
            }
        }

        return false;
    }

    /**
     * Получить список всех бэкапов
     */
    public function listBackups(): array
    {
        $backupPath = Yii::getAlias($this->backupPath);

        if (!file_exists($backupPath)) {
            FileHelper::createDirectory($backupPath);
            return [];
        }

        $backupDirs = glob("{$backupPath}/backup_*", GLOB_ONLYDIR);
        $result = [];

        foreach ($backupDirs as $backupDir) {
            $manifestFile = "{$backupDir}/manifest.json";

            if (!file_exists($manifestFile)) {
                continue;
            }

            try {
                $manifestContent = file_get_contents($manifestFile);
                $manifest = json_decode($manifestContent, true);

                if (!is_array($manifest)) {
                    continue;
                }

                $size = $this->calculateDirectorySize($backupDir);
                $backupId = basename($backupDir);

                $result[] = [
                    'id' => $backupId,
                    'path' => $backupDir,
                    'manifest' => $manifest,
                    'size' => $size,
                    'size_formatted' => $this->formatBytes($size),
                    'created_at' => strtotime($manifest['created_at'] ?? filemtime($backupDir)),
                    'type' => $manifest['type'] ?? 'unknown',
                ];
            } catch (Exception $e) {
                continue;
            }
        }

        usort($result, function($a, $b) {
            return $b['created_at'] <=> $a['created_at'];
        });

        return $result;
    }

    /**
     * Удалить бэкап
     */
    public function deleteBackup(string $backupId): bool
    {
        $backupDir = Yii::getAlias("{$this->backupPath}/{$backupId}");

        if (!file_exists($backupDir) || !is_dir($backupDir)) {
            return false;
        }

        try {
            FileHelper::removeDirectory($backupDir);
            return true;
        } catch (Exception $e) {
            return false;
        }
    }

    /**
     * Восстановить из бэкапа
     */
    public function restoreBackup(string $backupId, bool $restoreDatabase = true, bool $restoreFiles = true): array
    {
        $backupDir = Yii::getAlias("{$this->backupPath}/{$backupId}");

        if (!file_exists($backupDir)) {
            throw new Exception("Бэкап не найден: {$backupId}");
        }

        $manifestFile = "{$backupDir}/manifest.json";
        if (!file_exists($manifestFile)) {
            throw new Exception("Манифест бэкапа не найден");
        }

        $manifest = json_decode(file_get_contents($manifestFile), true);
        if (!is_array($manifest)) {
            throw new Exception("Неверный формат манифеста");
        }

        $type = $manifest['type'] ?? 'unknown';
        $steps = [];

        // Восстанавливаем БД
        if ($restoreDatabase) {
            $steps[] = $this->restoreDatabase($backupDir);
        }

        // Восстанавливаем файлы (только для FULL)
        if ($restoreFiles && $type === 'full') {
            $steps[] = $this->restoreProjectFiles($backupDir);
        }

        return [
            'success' => true,
            'backup_id' => $backupId,
            'type' => $type,
            'manifest' => $manifest,
            'steps' => $steps,
            'message' => 'Восстановление выполнено успешно',
        ];
    }

    /**
     * Восстановить БД из SQL
     */
    private function restoreDatabase(string $backupDir): array
    {
        $dumpFile = "{$backupDir}/database.sql";

        if (!file_exists($dumpFile)) {
            throw new \Exception("Файл дампа БД не найден");
        }

        $db = \Yii::$app->db;
        $sqlContent = file_get_contents($dumpFile);

        // Очищаем SQL от потенциальных проблем
        $sqlContent = $this->cleanupSqlForRestore($sqlContent);

        // Разбиваем на запросы
        $queries = $this->splitSqlQueries($sqlContent);

        $executed = 0;
        $totalQueries = count($queries);

        // Начинаем БОЛЬШУЮ транзакцию
        $transaction = $db->beginTransaction();

        try {
            // 1. Сначала удаляем все существующие таблицы (если нужно)
            $this->dropAllTables($db);

            // 2. Выполняем все запросы из дампа
            foreach ($queries as $query) {
                $query = trim($query);

                if (empty($query) || strpos($query, '--') === 0) {
                    continue;
                }

                $db->createCommand($query)->execute();
                $executed++;
            }

            $transaction->commit();

            return [
                'step' => 'database',
                'success' => true,
                'queries_executed' => $executed,
                'total_queries' => $totalQueries,
                'message' => 'База данных полностью восстановлена'
            ];

        } catch (\Exception $e) {
            $transaction->rollBack();
            throw new \Exception("Ошибка восстановления БД. Все изменения отменены: " . $e->getMessage());
        }
    }

    private function cleanupSqlForRestore(string $sql): string
    {
        // Удаляем команды SET которые могут вызвать ошибки
        $patternsToRemove = [
            '/^SET\s+session_replication_role.*$/mi',
            '/^SELECT\s+pg_catalog\.set_config.*$/mi',
            '/^SET\s+client_min_messages.*$/mi',
            '/^SET\s+row_security.*$/mi',
        ];

        foreach ($patternsToRemove as $pattern) {
            $sql = preg_replace($pattern, '', $sql);
        }

        return trim($sql);
    }

    private function dropAllTables(Connection $db): void
    {
        // Получаем все таблицы
        $tables = $this->getAllTables($db);

        // Отключаем проверку внешних ключей
        try {
            $db->createCommand("SET CONSTRAINTS ALL DEFERRED")->execute();
        } catch (\Exception $e) {
            // Игнорируем если нет прав
        }

        // Удаляем все таблицы
        foreach ($tables as $table) {
            try {
                $db->createCommand("DROP TABLE IF EXISTS \"{$table}\" CASCADE")->execute();
            } catch (\Exception $e) {
                throw new \Exception("Не удалось удалить таблицу {$table}: " . $e->getMessage());
            }
        }
    }

    /**
     * Восстановить файлы проекта
     */
    private function restoreProjectFiles(string $backupDir): array
    {
        $archiveFile = "{$backupDir}/project.zip";

        if (!file_exists($archiveFile)) {
            throw new Exception("Архив проекта не найден");
        }

        $projectRoot = Yii::getAlias('@app/..');

        // Распаковываем архив
        $zip = new \ZipArchive();

        if ($zip->open($archiveFile) !== true) {
            throw new Exception("Не удалось открыть ZIP архив проекта");
        }

        // Удаляем все существующие файлы (кроме исключений)
        $this->cleanProjectDirectory($projectRoot);

        // Извлекаем все файлы
        $extracted = $zip->extractTo($projectRoot);
        $zip->close();

        if (!$extracted) {
            throw new Exception("Не удалось распаковать архив проекта");
        }

        return [
            'step' => 'project_files',
            'success' => true,
            'archive' => $archiveFile,
            'destination' => $projectRoot,
        ];
    }

    private function cleanProjectDirectory(string $projectRoot): void
    {
        $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($projectRoot, RecursiveDirectoryIterator::SKIP_DOTS),
            RecursiveIteratorIterator::CHILD_FIRST
        );

        foreach ($iterator as $item) {
            $relativePath = substr($item->getRealPath(), strlen($projectRoot) + 1);

            // Не удаляем файлы из списка исключений (чтобы не удалить сами бэкапы и т.д.)
            if ($this->shouldExcludeFile($relativePath)) {
                continue;
            }

            if ($item->isDir()) {
                rmdir($item->getRealPath());
            } else {
                unlink($item->getRealPath());
            }
        }
    }

    /**
     * Создать бэкап текущих файлов проекта
     */
    private function backupCurrentProject(string $sourceDir, string $backupDir): void
    {
        FileHelper::createDirectory($backupDir);

        // Копируем только важные файлы (исключая временные)
        $importantPatterns = [
            '/\.php$/',
            '/\.(js|css)$/',
            '/\.(json|yaml|yml)$/',
            '/\.(sql|txt|md)$/',
            '/^config\//',
            '/^controllers\//',
            '/^models\//',
            '/^views\//',
            '/^web\/(?!assets)/',
            '/^migrations\//',
        ];

        $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($sourceDir, RecursiveDirectoryIterator::SKIP_DOTS),
            RecursiveIteratorIterator::SELF_FIRST
        );

        foreach ($iterator as $item) {
            if ($item->isFile()) {
                $filePath = $item->getRealPath();
                $relativePath = substr($filePath, strlen($sourceDir) + 1);

                // Проверяем, важный ли это файл
                $isImportant = false;
                foreach ($importantPatterns as $pattern) {
                    if (preg_match($pattern, $relativePath)) {
                        $isImportant = true;
                        break;
                    }
                }

                if ($isImportant && !$this->shouldExcludeFile($relativePath)) {
                    $destPath = $backupDir . '/' . $relativePath;
                    FileHelper::createDirectory(dirname($destPath));
                    copy($filePath, $destPath);
                }
            }
        }
    }

    /**
     * Восстановить из бэкапа
     */
    private function restoreFromBackup(string $backupDir, string $destination): void
    {
        FileHelper::copyDirectory($backupDir, $destination);
    }

    /**
     * Автоматическая ротация бэкапов
     */
    private function rotateBackups(): array
    {
        $allBackups = $this->listBackups();
        $deleted = [];

        $sqlBackups = [];
        $fullBackups = [];

        foreach ($allBackups as $backup) {
            if ($backup['type'] === 'sql') {
                $sqlBackups[] = $backup;
            } elseif ($backup['type'] === 'full') {
                $fullBackups[] = $backup;
            }
        }

        if (count($sqlBackups) > $this->maxSqlBackups) {
            $toDelete = array_slice($sqlBackups, $this->maxSqlBackups);
            foreach ($toDelete as $backup) {
                if ($this->deleteBackup($backup['id'])) {
                    $deleted[] = $backup['id'];
                }
            }
        }

        if (count($fullBackups) > $this->maxFullBackups) {
            $toDelete = array_slice($fullBackups, $this->maxFullBackups);
            foreach ($toDelete as $backup) {
                if ($this->deleteBackup($backup['id'])) {
                    $deleted[] = $backup['id'];
                }
            }
        }

        $totalBackups = count($sqlBackups) + count($fullBackups);
        if ($totalBackups > $this->maxBackups) {
            $allBackupsSorted = $allBackups;
            usort($allBackupsSorted, function($a, $b) {
                return $a['created_at'] <=> $b['created_at'];
            });

            $toDelete = array_slice($allBackupsSorted, 0, $totalBackups - $this->maxBackups);
            foreach ($toDelete as $backup) {
                if (!in_array($backup['id'], $deleted) && $this->deleteBackup($backup['id'])) {
                    $deleted[] = $backup['id'];
                }
            }
        }

        return $deleted;
    }

    /**
     * Получить информацию о системе
     */
    private function getSystemInfo(): array
    {
        $db = Yii::$app->db;

        return [
            'app' => [
                'name' => Yii::$app->name,
                'environment' => YII_ENV,
            ],
            'server' => [
                'php_version' => PHP_VERSION,
                'yii_version' => Yii::getVersion(),
            ],
            'database' => [
                'driver' => $db->driverName,
            ],
            'timestamp' => time(),
            'date' => date('c'),
        ];
    }

    /**
     * Получить имя БД из DSN
     */
    private function getDbNameFromDsn(string $dsn): string
    {
        preg_match('/dbname=([^;]+)/', $dsn, $matches);
        return $matches[1] ?? 'unknown';
    }

    /**
     * Получить все таблицы БД
     */
    private function getAllTables(Connection $db): array
    {
        try {
            $sql = "SELECT table_name 
                    FROM information_schema.tables 
                    WHERE table_schema = 'public' 
                    AND table_type = 'BASE TABLE'
                    ORDER BY table_name";

            return $db->createCommand($sql)->queryColumn();
        } catch (Exception $e) {
            return [];
        }
    }

    /**
     * Получить SQL для создания таблицы
     */
    private function getTableCreateSql(Connection $db, string $table): string
    {
        try {
            $sql = "SELECT 
                        'CREATE TABLE ' || quote_ident(table_name) || ' (' || 
                        string_agg(
                            quote_ident(column_name) || ' ' || 
                            data_type || 
                            CASE 
                                WHEN character_maximum_length IS NOT NULL 
                                THEN '(' || character_maximum_length || ')' 
                                ELSE '' 
                            END || 
                            CASE 
                                WHEN is_nullable = 'NO' THEN ' NOT NULL' 
                                ELSE '' 
                            END || 
                            CASE 
                                WHEN column_default IS NOT NULL 
                                THEN ' DEFAULT ' || column_default 
                                ELSE '' 
                            END,
                            ', '
                            ORDER BY ordinal_position
                        ) || 
                        ')' as create_sql
                    FROM information_schema.columns 
                    WHERE table_schema = 'public' 
                    AND table_name = :table
                    GROUP BY table_name";

            $result = $db->createCommand($sql, [':table' => $table])->queryOne();
            return $result['create_sql'] ?? "CREATE TABLE \"{$table}\" (id SERIAL PRIMARY KEY)";
        } catch (Exception $e) {
            return "-- Error getting structure for table {$table}: " . $e->getMessage();
        }
    }

    /**
     * Получить данные таблицы в виде INSERT запросов
     */
    private function getTableDataSql(Connection $db, string $table): string
    {
        $output = "";

        try {
            // Получаем все данные сразу (для небольших таблиц)
            // Для больших таблиц нужно пагинировать
            $rows = $db->createCommand("SELECT * FROM \"{$table}\"")->queryAll();

            if (empty($rows)) {
                return "";
            }

            // Получаем имена колонок
            $columns = array_keys($rows[0]);
            $columnNames = implode(', ', array_map(function($col) {
                return "\"{$col}\"";
            }, $columns));

            // Вставляем данные
            foreach ($rows as $row) {
                $values = [];
                foreach ($columns as $column) {
                    $value = $row[$column] ?? null;
                    $values[] = $this->escapeSqlValue($value);
                }
                $valuesStr = implode(', ', $values);
                $output .= "INSERT INTO \"{$table}\" ({$columnNames}) VALUES ({$valuesStr});\n";
            }

        } catch (\Exception $e) {
            return "-- Error: " . $e->getMessage();
        }

        return $output;
    }

    /**
     * Экранирование SQL значений
     */
    private function escapeSqlValue($value): string
    {
        if ($value === null) {
            return 'NULL';
        }

        if (is_numeric($value)) {
            return $value;
        }

        if (is_bool($value)) {
            return $value ? 'TRUE' : 'FALSE';
        }

        if (is_string($value)) {
            $value = str_replace("'", "''", $value);
            $value = str_replace("\\", "\\\\", $value);
            return "'{$value}'";
        }

        return "'" . strval($value) . "'";
    }

    /**
     * Разбить SQL на отдельные запросы
     */
    private function splitSqlQueries(string $sql): array
    {
        $sql = preg_replace('/--.*$/m', '', $sql);

        $queries = [];
        $currentQuery = '';
        $inString = false;
        $stringChar = '';

        for ($i = 0; $i < strlen($sql); $i++) {
            $char = $sql[$i];

            if ($char === "'" || $char === '"') {
                if (!$inString) {
                    $inString = true;
                    $stringChar = $char;
                } elseif ($stringChar === $char && $sql[$i-1] !== '\\') {
                    $inString = false;
                }
            }

            $currentQuery .= $char;

            if ($char === ';' && !$inString) {
                $queries[] = trim($currentQuery);
                $currentQuery = '';
            }
        }

        if (!empty(trim($currentQuery))) {
            $queries[] = trim($currentQuery);
        }

        return $queries;
    }

    /**
     * Посчитать количество таблиц в БД
     */
    private function countTables(Connection $db): int
    {
        try {
            $sql = "SELECT COUNT(*) FROM information_schema.tables 
                    WHERE table_schema = 'public' 
                    AND table_type = 'BASE TABLE'";
            return (int) $db->createCommand($sql)->queryScalar();
        } catch (Exception $e) {
            return 0;
        }
    }

    /**
     * Посчитать размер директории
     */
    private function calculateDirectorySize(string $directory): int
    {
        if (!file_exists($directory) || !is_dir($directory)) {
            return 0;
        }

        $size = 0;
        $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($directory, RecursiveDirectoryIterator::SKIP_DOTS)
        );

        foreach ($iterator as $file) {
            if ($file->isFile()) {
                $size += $file->getSize();
            }
        }

        return $size;
    }

    /**
     * Форматировать размер в байтах
     */
    public function formatBytes(int $bytes, int $precision = 2): string
    {
        $units = ['B', 'KB', 'MB', 'GB', 'TB'];

        $bytes = max($bytes, 0);
        $pow = floor(($bytes ? log($bytes) : 0) / log(1024));
        $pow = min($pow, count($units) - 1);

        $bytes /= pow(1024, $pow);

        return round($bytes, $precision) . ' ' . $units[$pow];
    }

    /**
     * Получить статистику по бэкапам
     */
    public function getStats(): array
    {
        $backups = $this->listBackups();

        $sqlCount = 0;
        $fullCount = 0;
        $totalSize = 0;
        $sqlSize = 0;
        $fullSize = 0;

        foreach ($backups as $backup) {
            $totalSize += $backup['size'];

            if ($backup['type'] === 'sql') {
                $sqlCount++;
                $sqlSize += $backup['size'];
            } elseif ($backup['type'] === 'full') {
                $fullCount++;
                $fullSize += $backup['size'];
            }
        }

        $backupPath = Yii::getAlias($this->backupPath);
        $diskFree = disk_free_space(dirname($backupPath));
        $diskTotal = disk_total_space(dirname($backupPath));

        return [
            'total' => count($backups),
            'sql_count' => $sqlCount,
            'full_count' => $fullCount,
            'size' => [
                'total' => $totalSize,
                'total_formatted' => $this->formatBytes($totalSize),
                'sql' => $sqlSize,
                'sql_formatted' => $this->formatBytes($sqlSize),
                'full' => $fullSize,
                'full_formatted' => $this->formatBytes($fullSize),
            ],
            'disk' => [
                'free' => $diskFree,
                'free_formatted' => $this->formatBytes($diskFree),
                'total' => $diskTotal,
                'total_formatted' => $this->formatBytes($diskTotal),
                'used_percent' => $diskTotal > 0 ? round(($diskTotal - $diskFree) / $diskTotal * 100, 2) : 0,
            ],
            'limits' => [
                'max_backups' => $this->maxBackups,
                'max_sql_backups' => $this->maxSqlBackups,
                'max_full_backups' => $this->maxFullBackups,
            ],
        ];
    }

    /**
     * Проверить требования системы
     */
    public function checkRequirements(): array
    {
        $requirements = [];

        $requirements['zip_extension'] = [
            'available' => extension_loaded('zip'),
            'message' => extension_loaded('zip')
                ? 'Расширение ZIP доступно'
                : 'Установите расширение ZIP'
        ];

        $backupPath = Yii::getAlias($this->backupPath);
        $requirements['backup_directory'] = [
            'exists' => file_exists($backupPath),
            'writable' => is_writable($backupPath) || (!file_exists($backupPath) && is_writable(dirname($backupPath))),
            'path' => $backupPath,
        ];

        $requirements['all_ok'] =
            $requirements['zip_extension']['available'] &&
            $requirements['backup_directory']['writable'];

        return $requirements;
    }

    /**
     * Скачать бэкап как архив
     */
    public function download(string $backupId): array
    {
        $backupDir = Yii::getAlias("{$this->backupPath}/{$backupId}");

        if (!file_exists($backupDir) || !is_dir($backupDir)) {
            throw new Exception("Бэкап не найден: {$backupId}");
        }

        // Создаем временный архив
        $tempDir = Yii::getAlias('@runtime/temp_backup_download');
        FileHelper::createDirectory($tempDir);

        $tempArchive = $tempDir . '/' . $backupId . '.tar.gz';

        // Архивируем папку бэкапа
        $this->createTarGzArchive($backupDir, $tempArchive);

        if (!file_exists($tempArchive)) {
            throw new Exception("Не удалось создать архив для скачивания");
        }

        return [
            'success' => true,
            'backup_id' => $backupId,
            'archive_path' => $tempArchive,
            'archive_name' => $backupId . '.tar.gz',
            'size' => filesize($tempArchive),
        ];
    }

    /**
     * Создать tar.gz архив средствами PHP
     */
    private function createTarGzArchive(string $sourceDir, string $outputFile): void
    {
        // Если есть команда tar, используем ее (быстрее)
        if (function_exists('exec')) {
            exec('which tar 2>/dev/null', $output, $returnCode);
            if ($returnCode === 0) {
                $command = sprintf(
                    'tar -czf %s -C %s . 2>&1',
                    escapeshellarg($outputFile),
                    escapeshellarg($sourceDir)
                );
                exec($command, $output, $returnCode);
                if ($returnCode === 0) {
                    return;
                }
            }
        }

        // Иначе создаем ZIP архив (менее эффективно, но работает везде)
        $this->createZipArchive($sourceDir, $outputFile);
    }

    /**
     * Создать ZIP архив (альтернатива tar.gz)
     */
    private function createZipArchive(string $sourceDir, string $outputFile): void
    {
        // Меняем расширение на .zip
        $outputFile = preg_replace('/\.tar\.gz$/', '.zip', $outputFile);

        $zip = new ZipArchive();

        if ($zip->open($outputFile, ZipArchive::CREATE | ZipArchive::OVERWRITE) !== true) {
            throw new Exception("Не удалось создать ZIP архив");
        }

        $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($sourceDir, RecursiveDirectoryIterator::SKIP_DOTS),
            RecursiveIteratorIterator::SELF_FIRST
        );

        foreach ($iterator as $item) {
            if ($item->isFile()) {
                $filePath = $item->getRealPath();
                $relativePath = substr($filePath, strlen($sourceDir) + 1);

                $zip->addFile($filePath, $relativePath);
            }
        }

        $zip->close();
    }

    /**
     * Отправить файл для скачивания
     */
    public function sendDownloadResponse(string $backupId): \yii\web\Response
    {
        $downloadInfo = $this->download($backupId);
        $archivePath = $downloadInfo['archive_path'];
        $archiveName = $downloadInfo['archive_name'];

        if (!file_exists($archivePath)) {
            throw new Exception("Архив не найден");
        }

        // Отправляем файл
        $response = Yii::$app->response;
        $response->sendFile($archivePath, $archiveName, [
            'mimeType' => $this->getMimeType($archivePath),
            'inline' => false,
        ]);

        // Удаляем временный файл после отправки
        register_shutdown_function(function() use ($archivePath) {
            if (file_exists($archivePath)) {
                unlink($archivePath);
            }
            // Удаляем временную директорию если пуста
            $tempDir = dirname($archivePath);
            if (file_exists($tempDir) && count(scandir($tempDir)) <= 2) {
                FileHelper::removeDirectory($tempDir);
            }
        });

        return $response;
    }

    /**
     * Получить MIME тип файла
     */
    private function getMimeType(string $filePath): string
    {
        $extension = strtolower(pathinfo($filePath, PATHINFO_EXTENSION));

        $mimeTypes = [
            'tar.gz' => 'application/gzip',
            'tgz' => 'application/gzip',
            'zip' => 'application/zip',
            'sql' => 'application/sql',
            'json' => 'application/json',
        ];

        return $mimeTypes[$extension] ?? 'application/octet-stream';
    }
}