<?php

namespace app\core\services\backup;

use Exception;
use FilesystemIterator;
use RecursiveDirectoryIterator;
use RecursiveIteratorIterator;
use Yii;
use yii\base\Component;
use yii\helpers\FileHelper;
use yii\db\Connection;
use ZipArchive;

class BackupManager04 extends Component
{
    public string $backupPath = '@app/backups';
    public $maxBackups = 50;
    public $maxSqlBackups = 30;
    public $maxFullBackups = 7;

    // Исключения для полного бэкапа
    public $excludePatterns = [
        '/^.git\//',
        '/^tests\//',
        '/^docker\//',
        '/^backups\//',
        '/^vue3\//',
        '/^node_modules\//',
        '/^vendor\/[^\/]+\/[^\/]+\/tests?\//',
        '/^vendor\/[^\/]+\/[^\/]+\/docs?\//',
        '/^runtime\/cache\//',
        '/^runtime\/debug\//',
        '/^\.idea\//',
        '/^\.vscode\//',
        '/^\.DS_Store$/',
        '/^Thumbs\.db$/',
        '/\.bak$/',
        '/\.tmp$/',
        '/\.swp$/',
    ];

    /**
     * Создать новый бэкап
     */
    public function createBackup(string $type, string $description = null, string $createdBy = null): array
    {
        if (!in_array($type, ['sql', 'full'])) {
            throw new \InvalidArgumentException("Неправильный тип бэкапа: {$type}. Допустимо: sql, full");
        }

        $timestamp = date('Y-m-d_H-i-s');
        $backupId = "backup_{$timestamp}_{$type}";
        $backupDir = Yii::getAlias("{$this->backupPath}/{$backupId}");

        FileHelper::createDirectory($backupDir);

        $manifest = [
            'id' => $backupId,
            'type' => $type,
            'description' => $description,
            'created_at' => date('c'),
            'created_by' => $createdBy,
            'system_info' => $this->getSystemInfo(),
            'excluded_patterns' => $this->excludePatterns,
        ];

        try {
            // Создаем бэкап БД (для обоих типов)
            $manifest['database'] = $this->backupDatabase($backupDir);

            // Архивируем файлы проекта (только для FULL)
            if ($type === 'full') {
                $manifest['project_files'] = $this->backupProjectFiles($backupDir);
            }

            // Сохраняем манифест
            $manifestFile = "{$backupDir}/manifest.json";
            file_put_contents($manifestFile,
                json_encode($manifest, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE)
            );

            // Ротация
            $deletedBackups = $this->rotateBackups();
            $totalSize = $this->calculateDirectorySize($backupDir);

            return [
                'success' => true,
                'backup_id' => $backupId,
                'path' => $backupDir,
                'manifest' => $manifest,
                'deleted_old_backups' => $deletedBackups,
                'total_size' => $totalSize,
                'total_size_formatted' => $this->formatBytes($totalSize),
            ];

        } catch (Exception $e) {
            if (file_exists($backupDir)) {
                FileHelper::removeDirectory($backupDir);
            }
            throw $e;
        }
    }

    /**
     * Создать дамп БД (ТОЛЬКО данные)
     */
    private function backupDatabase(string $backupDir): array
    {
        $db = Yii::$app->db;
        $dumpFile = "{$backupDir}/database.sql";

        if ($db->driverName !== 'pgsql') {
            throw new Exception("Поддерживается только PostgreSQL");
        }

        $dumpContent = $this->generateSqlDump($db);

        if (file_put_contents($dumpFile, $dumpContent) === false) {
            throw new Exception("Не удалось записать дамп БД в файл");
        }

        $tablesCount = $this->countTables($db);

        return [
            'file' => 'database.sql',
            'size' => filesize($dumpFile),
            'tables_count' => $tablesCount,
            'encoding' => 'UTF-8',
            'generated_by' => 'php',
            'data_only' => true,
        ];
    }

    /**
     * Генерация SQL дампа (ТОЛЬКО данные, без SET команд)
     */
    private function generateSqlDump(Connection $db): string
    {
        $output = "";

        $output .= "-- PostgreSQL DATA ONLY dump\n";
        $output .= "-- Generated: " . date('Y-m-d H:i:s') . "\n";
        $output .= "-- IMPORTANT: This dump contains ONLY data, not table structure!\n";
        $output .= "-- Tables should be created via Yii2 migrations before restoring this data.\n\n";

        // Получаем все таблицы
        $tables = $this->getAllTables($db);

        // Сортируем таблицы для правильного порядка вставки (сначала родительские)
        $sortedTables = $this->sortTablesByDependencies($db, $tables);

        foreach ($sortedTables as $table) {
            try {
                $dataSql = $this->getTableDataSql($db, $table);
                if (!empty(trim($dataSql))) {
                    $output .= "-- Table: {$table}\n";
                    $output .= $dataSql . "\n";
                }
            } catch (\Exception $e) {
                $output .= "-- Error getting data for table {$table}: " . $e->getMessage() . "\n\n";
            }
        }

        return $output;
    }

    private function sortTablesByDependencies(Connection $db, array $tables): array
    {
        // Получаем информацию о внешних ключах
        $dependencies = [];
        foreach ($tables as $table) {
            $dependencies[$table] = $this->getTableDependencies($db, $table);
        }

        // Простая топологическая сортировка
        $sorted = [];
        $visited = [];

        foreach ($tables as $table) {
            $this->visitTable($table, $dependencies, $visited, $sorted);
        }

        return $sorted;
    }

    private function getTableDependencies(Connection $db, string $table): array
    {
        $sql = "SELECT
                confrelid::regclass::text as referenced_table
            FROM pg_constraint
            WHERE conrelid = :table::regclass
            AND contype = 'f'";

        try {
            return $db->createCommand($sql, [':table' => $table])->queryColumn();
        } catch (\Exception $e) {
            return [];
        }
    }

    private function visitTable(string $table, array $dependencies, array &$visited, array &$sorted): void
    {
        if (in_array($table, $visited)) {
            return;
        }

        $visited[] = $table;

        foreach ($dependencies[$table] ?? [] as $dependency) {
            $this->visitTable($dependency, $dependencies, $visited, $sorted);
        }

        $sorted[] = $table;
    }

    public function lastBackupsDate(): ?int
    {
        $backupPath = Yii::getAlias($this->backupPath);

        if (!is_dir($backupPath)) {
            return null;
        }

        return \app\core\helpers\FileHelper::getNewestFolderDate($backupPath);
    }

    public function sizeBackups()
    {
        $backupPath = Yii::getAlias($this->backupPath);

        if (!is_dir($backupPath)) {
            return 0;
        }

        $size = 0;
        $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($backupPath, FilesystemIterator::SKIP_DOTS)
        );

        foreach ($iterator as $file) {
            if ($file->isFile()) {
                $size += $file->getSize();
            }
        }

        return $size;
    }

    /**
     * Бэкап всех файлов проекта (кроме исключенных)
     */
    private function backupProjectFiles(string $backupDir): array
    {
        // Определяем корень проекта по composer.json
        $projectRoot = realpath(Yii::getAlias('@app'));
        if (!$projectRoot) {
            throw new \Exception("Не удалось определить корневую директорию проекта (composer.json не найден)");
        }

        $archiveFile = "{$backupDir}/project.zip";

        // Создаем ZIP архив
        $zip = new \ZipArchive();

        if ($zip->open($archiveFile, \ZipArchive::CREATE | \ZipArchive::OVERWRITE) !== true) {
            throw new \Exception("Не удалось создать ZIP архив");
        }

        $filesCount = 0;
        $totalSize = 0;
        $errors = [];

        // Функция для рекурсивного сканирования с обработкой ошибок
        $scanDirectory = function($directory, $basePath) use (&$scanDirectory, &$zip, &$filesCount, &$totalSize, &$errors, $projectRoot) {
            try {
                $items = scandir($directory);
            } catch (\Exception $e) {
                $errors[] = "Не удалось прочитать директорию {$directory}: " . $e->getMessage();
                return;
            }

            foreach ($items as $item) {
                if ($item === '.' || $item === '..') {
                    continue;
                }

                $fullPath = $directory . '/' . $item;
                $relativePath = substr($fullPath, strlen($projectRoot) + 1);

                // Проверяем на исключения
                if ($this->shouldExcludeFile($relativePath)) {
                    continue;
                }

                try {
                    if (is_dir($fullPath)) {
                        // Рекурсивно сканируем поддиректорию
                        $scanDirectory($fullPath, $basePath);
                    } elseif (is_file($fullPath)) {
                        if ($zip->addFile($fullPath, $relativePath)) {
                            $filesCount++;
                            $totalSize += filesize($fullPath);
                        }
                    }
                } catch (\Exception $e) {
                    $errors[] = "Ошибка обработки {$fullPath}: " . $e->getMessage();
                    continue;
                }
            }
        };

        // Запускаем сканирование
        $scanDirectory($projectRoot, $projectRoot);

        // Добавляем README с информацией о бэкапе
        $readmeContent = "FULL BACKUP\n";
        $readmeContent .= "Created: " . date('Y-m-d H:i:s') . "\n";
        $readmeContent .= "Project root: " . $projectRoot . "\n";
        $readmeContent .= "Files: {$filesCount}\n";
        $readmeContent .= "Total size: " . $this->formatBytes($totalSize) . "\n";
        $readmeContent .= "Excluded patterns:\n";
        foreach ($this->excludePatterns as $pattern) {
            $readmeContent .= "- " . $pattern . "\n";
        }

        if (!empty($errors)) {
            $readmeContent .= "\nWarnings/Errors (" . count($errors) . "):\n";
            foreach (array_slice($errors, 0, 10) as $error) {
                $readmeContent .= "- " . substr($error, 0, 200) . "\n";
            }
        }

        $zip->addFromString('BACKUP_INFO.txt', $readmeContent);

        $zip->close();

        if (!file_exists($archiveFile)) {
            throw new \Exception("Архив создан, но файл отсутствует");
        }

        return [
            'file' => 'project.zip',
            'size' => filesize($archiveFile),
            'files_count' => $filesCount,
            'original_size' => $totalSize,
            'compression_ratio' => $totalSize > 0 ? round(filesize($archiveFile) / $totalSize * 100, 2) : 0,
            'format' => 'zip',
            'project_root' => $projectRoot,
            'errors_count' => count($errors),
            'errors' => array_slice($errors, 0, 5),
        ];
    }

    /**
     * Проверить, нужно ли исключить файл
     */
    private function shouldExcludeFile(string $relativePath): bool
    {
        $path = str_replace('\\', '/', $relativePath);

        foreach ($this->excludePatterns as $pattern) {
            if (preg_match($pattern, $path)) {
                return true;
            }
        }

        return false;
    }

    /**
     * Получить список всех бэкапов
     */
    public function listBackups(): array
    {
        $backupPath = Yii::getAlias($this->backupPath);

        if (!file_exists($backupPath)) {
            FileHelper::createDirectory($backupPath);
            return [];
        }

        $backupDirs = glob("{$backupPath}/backup_*", GLOB_ONLYDIR);
        $result = [];

        foreach ($backupDirs as $backupDir) {
            $manifestFile = "{$backupDir}/manifest.json";

            if (!file_exists($manifestFile)) {
                continue;
            }

            try {
                $manifestContent = file_get_contents($manifestFile);
                $manifest = json_decode($manifestContent, true);

                if (!is_array($manifest)) {
                    continue;
                }

                $size = $this->calculateDirectorySize($backupDir);
                $backupId = basename($backupDir);

                $result[] = [
                    'id' => $backupId,
                    'path' => $backupDir,
                    'manifest' => $manifest,
                    'size' => $size,
                    'size_formatted' => $this->formatBytes($size),
                    'created_at' => strtotime($manifest['created_at'] ?? filemtime($backupDir)),
                    'type' => $manifest['type'] ?? 'unknown',
                ];
            } catch (Exception $e) {
                continue;
            }
        }

        usort($result, function($a, $b) {
            return $b['created_at'] <=> $a['created_at'];
        });

        return $result;
    }

    /**
     * Удалить бэкап
     */
    public function deleteBackup(string $backupId): bool
    {
        $backupDir = Yii::getAlias("{$this->backupPath}/{$backupId}");

        if (!file_exists($backupDir) || !is_dir($backupDir)) {
            return false;
        }

        try {
            FileHelper::removeDirectory($backupDir);
            return true;
        } catch (Exception $e) {
            return false;
        }
    }

    /**
     * Восстановить из бэкапа
     */
    public function restoreBackup(string $backupId, bool $restoreDatabase = true, bool $restoreFiles = true): array
    {
        $backupDir = Yii::getAlias("{$this->backupPath}/{$backupId}");

        if (!file_exists($backupDir)) {
            throw new Exception("Бэкап не найден: {$backupId}");
        }

        $manifestFile = "{$backupDir}/manifest.json";
        if (!file_exists($manifestFile)) {
            throw new Exception("Манифест бэкапа не найден");
        }

        $manifest = json_decode(file_get_contents($manifestFile), true);
        if (!is_array($manifest)) {
            throw new Exception("Неверный формат манифеста");
        }

        $type = $manifest['type'] ?? 'unknown';
        $steps = [];

        // Восстанавливаем файлы (только для FULL)
        if ($restoreFiles && $type === 'full') {
            $steps[] = $this->restoreProjectFiles($backupDir);

            // После восстановления файлов, запускаем миграции
            $steps[] = $this->runMigrations();
        }

        // Восстанавливаем БД
        if ($restoreDatabase) {
            $steps[] = $this->restoreDatabase($backupDir, $type === 'full');
        }

        return [
            'success' => true,
            'backup_id' => $backupId,
            'type' => $type,
            'manifest' => $manifest,
            'steps' => $steps,
            'message' => 'Восстановление выполнено успешно',
        ];
    }

    /**
     * Восстановить БД из SQL
     */
    private function restoreDatabase(string $backupDir, bool $isFullBackup = false): array
    {
        $dumpFile = "{$backupDir}/database.sql";

        if (!file_exists($dumpFile)) {
            throw new \Exception("Файл дампа БД не найден");
        }

        $db = \Yii::$app->db;
        $sqlContent = file_get_contents($dumpFile);

        // Проверяем, что это дамп только данных
        if (strpos($sqlContent, 'DATA ONLY dump') === false) {
            throw new \Exception("Этот дамп содержит структуру таблиц. Используйте миграции для создания таблиц.");
        }

        // Для полного бэкапа таблицы уже созданы миграциями
        // Для SQL бэкапа - предполагаем что таблицы уже существуют
        // Очищаем таблицы в любом случае
        $this->clearAllTablesData($db);

        // Разбиваем на запросы (упрощенная версия без обработки SET команд)
        $queries = $this->splitSimpleSqlQueries($sqlContent);

        $executed = 0;
        $totalQueries = count($queries);
        $errors = [];

        // Начинаем транзакцию
        $transaction = $db->beginTransaction();

        try {
            // Выполняем запросы вставки (только INSERT)
            foreach ($queries as $query) {
                $query = trim($query);

                if (empty($query) || strpos($query, '--') === 0) {
                    continue;
                }

                // Пропускаем SET команды
                if (strtoupper(substr($query, 0, 3)) === 'SET') {
                    continue;
                }

                // Пропускаем комментарии
                if (strpos($query, '--') === 0) {
                    continue;
                }

                try {
                    $db->createCommand($query)->execute();
                    $executed++;
                } catch (\Exception $e) {
                    $errors[] = "Ошибка выполнения запроса: " . $e->getMessage() . " | Query: " . substr($query, 0, 100);
                    // Продолжаем выполнение
                }
            }

            $transaction->commit();

            return [
                'step' => 'database',
                'success' => true,
                'queries_executed' => $executed,
                'total_queries' => $totalQueries,
                'errors' => $errors,
                'message' => 'Данные БД восстановлены'
            ];

        } catch (\Exception $e) {
            $transaction->rollBack();
            throw new \Exception("Ошибка восстановления данных БД: " . $e->getMessage());
        }
    }

    /**
     * Очистить данные всех таблиц (без удаления таблиц)
     */
    private function clearAllTablesData(Connection $db): void
    {
        $tables = $this->getAllTables($db);

        // Отключаем проверку внешних ключей временно
        try {
            $db->createCommand("SET session_replication_role = 'replica'")->execute();
        } catch (\Exception $e) {
            // Пропускаем если нет прав
        }

        // Очищаем таблицы в обратном порядке (сначала дочерние)
        $reversedTables = array_reverse($tables);

        foreach ($reversedTables as $table) {
            try {
                $db->createCommand("TRUNCATE TABLE \"{$table}\" CASCADE")->execute();
            } catch (\Exception $e) {
                // Если не удалось очистить через TRUNCATE, пробуем DELETE
                try {
                    $db->createCommand("DELETE FROM \"{$table}\"")->execute();
                } catch (\Exception $e2) {
                    // Пропускаем если не удалось
                }
            }
        }

        // Включаем проверку обратно
        try {
            $db->createCommand("SET session_replication_role = 'origin'")->execute();
        } catch (\Exception $e) {
            // Пропускаем
        }
    }

    /**
     * Запустить Yii2 миграции
     */
    private function runMigrations(): array
    {
        $result = [
            'step' => 'migrations',
            'success' => true,
            'output' => [],
            'message' => ''
        ];

        try {
            // Получаем объект миграций
            $migration = new \yii\console\controllers\MigrateController('migrate', Yii::$app);
            $migration->interactive = false;
            $migration->color = false;

            // Создаем временный файл для вывода
            $tempFile = tempnam(sys_get_temp_dir(), 'migrate_');

            // Захватываем вывод
            ob_start();

            // Выполняем все новые миграции
            $exitCode = $migration->runAction('up', [
                'interactive' => false,
                'migrationPath' => null,
            ]);

            $output = ob_get_clean();

            if ($exitCode === 0) {
                $result['message'] = 'Миграции успешно выполнены';
                $result['output'] = explode("\n", $output);
            } else {
                $result['success'] = false;
                $result['message'] = 'Ошибка выполнения миграций';
                $result['output'] = explode("\n", $output);
                throw new Exception('Миграции не выполнены: ' . $output);
            }

        } catch (\Exception $e) {
            $result['success'] = false;
            $result['message'] = 'Ошибка при запуске миграций: ' . $e->getMessage();
            throw new Exception('Не удалось выполнить миграции: ' . $e->getMessage());
        }

        return $result;
    }

    /**
     * Упрощенное разбиение SQL на запросы (без обработки SET)
     */
    private function splitSimpleSqlQueries(string $sql): array
    {
        // Удаляем все SET команды и другие системные команды
        $sql = preg_replace('/^SET\s+.*$/mi', '', $sql);
        $sql = preg_replace('/^SELECT\s+pg_catalog\.set_config.*$/mi', '', $sql);

        // Разбиваем по точкам с запятой
        $queries = explode(';', $sql);

        // Очищаем запросы
        $cleanedQueries = [];
        foreach ($queries as $query) {
            $query = trim($query);
            if (!empty($query) && strpos($query, '--') !== 0) {
                // Удаляем комментарии в конце строк
                $query = preg_replace('/\s*--.*$/m', '', $query);
                if (!empty(trim($query))) {
                    $cleanedQueries[] = $query . ';';
                }
            }
        }

        return $cleanedQueries;
    }

    /**
     * Восстановить файлы проекта
     */
    private function restoreProjectFiles(string $backupDir): array
    {
        $archiveFile = "{$backupDir}/project.zip";

        if (!file_exists($archiveFile)) {
            throw new Exception("Архив проекта не найден");
        }

        $projectRoot = realpath(Yii::getAlias('@app'));
        if (!$projectRoot) {
            throw new Exception("Не удалось определить корневую директорию проекта");
        }

        // Распаковываем архив
        $zip = new \ZipArchive();

        if ($zip->open($archiveFile) !== true) {
            throw new Exception("Не удалось открыть ZIP архив проекта");
        }

        // Создаем бэкап текущих файлов на случай отката
        $backupBeforeRestore = $this->backupCurrentFilesBeforeRestore($projectRoot);

        try {
            // Очищаем проект (кроме исключенных папок)
            $this->cleanProjectDirectory($projectRoot);

            // Извлекаем все файлы
            $extracted = $zip->extractTo($projectRoot);
            $zip->close();

            if (!$extracted) {
                throw new Exception("Не удалось распаковать архив проекта");
            }

            return [
                'step' => 'project_files',
                'success' => true,
                'archive' => $archiveFile,
                'destination' => $projectRoot,
                'backup_before_restore' => $backupBeforeRestore,
            ];

        } catch (\Exception $e) {
            // В случае ошибки восстанавливаем из временного бэкапа
            if (isset($backupBeforeRestore['temp_dir'])) {
                $this->restoreFromTempBackup($backupBeforeRestore['temp_dir'], $projectRoot);
            }
            throw $e;
        }
    }

    /**
     * Создать временный бэкап текущих файлов перед восстановлением
     */
    private function backupCurrentFilesBeforeRestore(string $projectRoot): array
    {
        $tempDir = Yii::getAlias('@runtime/temp_backup_' . date('Y-m-d_H-i-s'));
        FileHelper::createDirectory($tempDir);

        // Копируем только важные файлы (исключая временные)
        $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($projectRoot, RecursiveDirectoryIterator::SKIP_DOTS),
            RecursiveIteratorIterator::SELF_FIRST
        );

        $filesCopied = 0;
        foreach ($iterator as $item) {
            if ($item->isFile()) {
                $filePath = $item->getRealPath();
                $relativePath = substr($filePath, strlen($projectRoot) + 1);

                // Пропускаем исключенные файлы
                if ($this->shouldExcludeFile($relativePath)) {
                    continue;
                }

                $destPath = $tempDir . '/' . $relativePath;
                FileHelper::createDirectory(dirname($destPath));
                copy($filePath, $destPath);
                $filesCopied++;
            }
        }

        return [
            'temp_dir' => $tempDir,
            'files_copied' => $filesCopied,
            'created_at' => date('Y-m-d H:i:s')
        ];
    }

    /**
     * Восстановить из временного бэкапа
     */
    private function restoreFromTempBackup(string $tempDir, string $projectRoot): void
    {
        FileHelper::copyDirectory($tempDir, $projectRoot);
        FileHelper::removeDirectory($tempDir);
    }

    /**
     * Очистить проект (кроме исключенных папок)
     */
    private function cleanProjectDirectory(string $projectRoot): void
    {
        $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($projectRoot, RecursiveDirectoryIterator::SKIP_DOTS),
            RecursiveIteratorIterator::CHILD_FIRST
        );

        foreach ($iterator as $item) {
            $relativePath = substr($item->getRealPath(), strlen($projectRoot) + 1);
            $relativePath = str_replace('\\', '/', $relativePath);

            // Пропускаем исключенные файлы/папки
            if ($this->shouldExcludeFile($relativePath)) {
                continue;
            }

            // Пропускаем саму папку backups
            if (strpos($relativePath, 'backups') === 0) {
                continue;
            }

            // Пропускаем runtime папку (там логи и кэш)
            if (strpos($relativePath, 'runtime') === 0) {
                continue;
            }

            try {
                if ($item->isDir()) {
                    // Пропускаем если директория не пуста
                    $isEmpty = !(new \FilesystemIterator($item->getRealPath()))->valid();
                    if ($isEmpty) {
                        rmdir($item->getRealPath());
                    }
                } else {
                    unlink($item->getRealPath());
                }
            } catch (\Exception $e) {
                // Пропускаем ошибки удаления
                continue;
            }
        }
    }

    /**
     * Автоматическая ротация бэкапов
     */
    private function rotateBackups(): array
    {
        $allBackups = $this->listBackups();
        $deleted = [];

        $sqlBackups = [];
        $fullBackups = [];

        foreach ($allBackups as $backup) {
            if ($backup['type'] === 'sql') {
                $sqlBackups[] = $backup;
            } elseif ($backup['type'] === 'full') {
                $fullBackups[] = $backup;
            }
        }

        if (count($sqlBackups) > $this->maxSqlBackups) {
            $toDelete = array_slice($sqlBackups, $this->maxSqlBackups);
            foreach ($toDelete as $backup) {
                if ($this->deleteBackup($backup['id'])) {
                    $deleted[] = $backup['id'];
                }
            }
        }

        if (count($fullBackups) > $this->maxFullBackups) {
            $toDelete = array_slice($fullBackups, $this->maxFullBackups);
            foreach ($toDelete as $backup) {
                if ($this->deleteBackup($backup['id'])) {
                    $deleted[] = $backup['id'];
                }
            }
        }

        $totalBackups = count($sqlBackups) + count($fullBackups);
        if ($totalBackups > $this->maxBackups) {
            $allBackupsSorted = $allBackups;
            usort($allBackupsSorted, function($a, $b) {
                return $a['created_at'] <=> $b['created_at'];
            });

            $toDelete = array_slice($allBackupsSorted, 0, $totalBackups - $this->maxBackups);
            foreach ($toDelete as $backup) {
                if (!in_array($backup['id'], $deleted) && $this->deleteBackup($backup['id'])) {
                    $deleted[] = $backup['id'];
                }
            }
        }

        return $deleted;
    }

    /**
     * Получить информацию о системе
     */
    private function getSystemInfo(): array
    {
        $db = \Yii::$app->db;

        // Получаем текущую миграцию
        $migration = '';
        $migrationTime = '';
        try {
            $migrationData = $db->createCommand("SELECT version, apply_time FROM migration ORDER BY apply_time DESC LIMIT 1")
                ->queryOne();
            if ($migrationData) {
                $migration = $migrationData['version'];
                $migrationTime = date('Y-m-d H:i:s', $migrationData['apply_time']);
            }
        } catch (\Exception $e) {
            // Таблица миграций может не существовать
        }

        return [
            'app' => [
                'name' => \Yii::$app->name,
                'environment' => YII_ENV,
            ],
            'server' => [
                'php_version' => PHP_VERSION,
                'yii_version' => \Yii::getVersion(),
            ],
            'database' => [
                'driver' => $db->driverName,
                'tables_count' => $this->countTables($db),
                'last_migration' => $migration,
                'last_migration_time' => $migrationTime,
            ],
            'timestamp' => time(),
            'date' => date('c'),
        ];
    }

    /**
     * Получить все таблицы БД
     */
    private function getAllTables(Connection $db): array
    {
        try {
            $sql = "SELECT table_name 
                    FROM information_schema.tables 
                    WHERE table_schema = 'public' 
                    AND table_type = 'BASE TABLE'
                    ORDER BY table_name";

            return $db->createCommand($sql)->queryColumn();
        } catch (Exception $e) {
            return [];
        }
    }

    /**
     * Получить данные таблицы в виде INSERT запросов
     */
    private function getTableDataSql(Connection $db, string $table): string
    {
        $output = "";

        try {
            // Получаем данные пачками по 1000 записей
            $offset = 0;
            $limit = 1000;

            while (true) {
                $rows = $db->createCommand("SELECT * FROM \"{$table}\" LIMIT {$limit} OFFSET {$offset}")->queryAll();

                if (empty($rows)) {
                    break;
                }

                // Получаем имена колонок
                $columns = array_keys($rows[0]);
                $columnNames = implode(', ', array_map(function($col) {
                    return "\"{$col}\"";
                }, $columns));

                // Вставляем данные
                foreach ($rows as $row) {
                    $values = [];
                    foreach ($columns as $column) {
                        $value = $row[$column] ?? null;
                        $values[] = $this->escapeSqlValue($value);
                    }
                    $valuesStr = implode(', ', $values);
                    $output .= "INSERT INTO \"{$table}\" ({$columnNames}) VALUES ({$valuesStr});\n";
                }

                $offset += $limit;

                // Если получили меньше чем лимит, значит это последняя пачка
                if (count($rows) < $limit) {
                    break;
                }
            }

        } catch (\Exception $e) {
            return "-- Error: " . $e->getMessage();
        }

        return $output;
    }

    /**
     * Экранирование SQL значений
     */
    private function escapeSqlValue($value): string
    {
        if ($value === null) {
            return 'NULL';
        }

        if (is_numeric($value)) {
            return $value;
        }

        if (is_bool($value)) {
            return $value ? 'TRUE' : 'FALSE';
        }

        if (is_string($value)) {
            $value = str_replace("'", "''", $value);
            $value = str_replace("\\", "\\\\", $value);
            return "'{$value}'";
        }

        return "'" . strval($value) . "'";
    }

    /**
     * Посчитать количество таблиц в БД
     */
    private function countTables(Connection $db): int
    {
        try {
            $sql = "SELECT COUNT(*) FROM information_schema.tables 
                    WHERE table_schema = 'public' 
                    AND table_type = 'BASE TABLE'";
            return (int) $db->createCommand($sql)->queryScalar();
        } catch (Exception $e) {
            return 0;
        }
    }

    /**
     * Посчитать размер директории
     */
    private function calculateDirectorySize(string $directory): int
    {
        if (!file_exists($directory) || !is_dir($directory)) {
            return 0;
        }

        $size = 0;
        $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($directory, RecursiveDirectoryIterator::SKIP_DOTS)
        );

        foreach ($iterator as $file) {
            if ($file->isFile()) {
                $size += $file->getSize();
            }
        }

        return $size;
    }

    /**
     * Форматировать размер в байтах
     */
    public function formatBytes(int $bytes, int $precision = 2): string
    {
        $units = ['B', 'KB', 'MB', 'GB', 'TB'];

        $bytes = max($bytes, 0);
        $pow = floor(($bytes ? log($bytes) : 0) / log(1024));
        $pow = min($pow, count($units) - 1);

        $bytes /= pow(1024, $pow);

        return round($bytes, $precision) . ' ' . $units[$pow];
    }

    /**
     * Получить статистику по бэкапам
     */
    public function getStats(): array
    {
        $backups = $this->listBackups();

        $sqlCount = 0;
        $fullCount = 0;
        $totalSize = 0;
        $sqlSize = 0;
        $fullSize = 0;

        foreach ($backups as $backup) {
            $totalSize += $backup['size'];

            if ($backup['type'] === 'sql') {
                $sqlCount++;
                $sqlSize += $backup['size'];
            } elseif ($backup['type'] === 'full') {
                $fullCount++;
                $fullSize += $backup['size'];
            }
        }

        $backupPath = Yii::getAlias($this->backupPath);
        $diskFree = disk_free_space(dirname($backupPath));
        $diskTotal = disk_total_space(dirname($backupPath));

        return [
            'total' => count($backups),
            'sql_count' => $sqlCount,
            'full_count' => $fullCount,
            'size' => [
                'total' => $totalSize,
                'total_formatted' => $this->formatBytes($totalSize),
                'sql' => $sqlSize,
                'sql_formatted' => $this->formatBytes($sqlSize),
                'full' => $fullSize,
                'full_formatted' => $this->formatBytes($fullSize),
            ],
            'disk' => [
                'free' => $diskFree,
                'free_formatted' => $this->formatBytes($diskFree),
                'total' => $diskTotal,
                'total_formatted' => $this->formatBytes($diskTotal),
                'used_percent' => $diskTotal > 0 ? round(($diskTotal - $diskFree) / $diskTotal * 100, 2) : 0,
            ],
            'limits' => [
                'max_backups' => $this->maxBackups,
                'max_sql_backups' => $this->maxSqlBackups,
                'max_full_backups' => $this->maxFullBackups,
            ],
        ];
    }

    /**
     * Проверить требования системы
     */
    public function checkRequirements(): array
    {
        $requirements = [];

        $requirements['zip_extension'] = [
            'available' => extension_loaded('zip'),
            'message' => extension_loaded('zip')
                ? 'Расширение ZIP доступно'
                : 'Установите расширение ZIP'
        ];

        $backupPath = Yii::getAlias($this->backupPath);
        $requirements['backup_directory'] = [
            'exists' => file_exists($backupPath),
            'writable' => is_writable($backupPath) || (!file_exists($backupPath) && is_writable(dirname($backupPath))),
            'path' => $backupPath,
        ];

        $requirements['all_ok'] =
            $requirements['zip_extension']['available'] &&
            $requirements['backup_directory']['writable'];

        return $requirements;
    }

    /**
     * Скачать бэкап как архив
     */
    public function download(string $backupId): array
    {
        $backupDir = Yii::getAlias("{$this->backupPath}/{$backupId}");

        if (!file_exists($backupDir) || !is_dir($backupDir)) {
            throw new Exception("Бэкап не найден: {$backupId}");
        }

        // Создаем временный архив
        $tempDir = Yii::getAlias('@runtime/temp_backup_download');
        FileHelper::createDirectory($tempDir);

        $tempArchive = $tempDir . '/' . $backupId . '.zip';

        // Создаем ZIP архив
        $this->createZipArchive($backupDir, $tempArchive);

        if (!file_exists($tempArchive)) {
            throw new Exception("Не удалось создать архив для скачивания");
        }

        return [
            'success' => true,
            'backup_id' => $backupId,
            'archive_path' => $tempArchive,
            'archive_name' => $backupId . '.zip',
            'size' => filesize($tempArchive),
        ];
    }

    /**
     * Создать ZIP архив
     */
    private function createZipArchive(string $sourceDir, string $outputFile): void
    {
        $zip = new ZipArchive();

        if ($zip->open($outputFile, ZipArchive::CREATE | ZipArchive::OVERWRITE) !== true) {
            throw new Exception("Не удалось создать ZIP архив");
        }

        $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($sourceDir, RecursiveDirectoryIterator::SKIP_DOTS),
            RecursiveIteratorIterator::SELF_FIRST
        );

        foreach ($iterator as $item) {
            if ($item->isFile()) {
                $filePath = $item->getRealPath();
                $relativePath = substr($filePath, strlen($sourceDir) + 1);

                $zip->addFile($filePath, $relativePath);
            }
        }

        $zip->close();
    }

    /**
     * Отправить файл для скачивания
     */
    public function sendDownloadResponse(string $backupId): \yii\web\Response
    {
        $downloadInfo = $this->download($backupId);
        $archivePath = $downloadInfo['archive_path'];
        $archiveName = $downloadInfo['archive_name'];

        if (!file_exists($archivePath)) {
            throw new Exception("Архив не найден");
        }

        // Отправляем файл
        $response = Yii::$app->response;
        $response->sendFile($archivePath, $archiveName, [
            'mimeType' => 'application/zip',
            'inline' => false,
        ]);

        // Удаляем временный файл после отправки
        register_shutdown_function(function() use ($archivePath) {
            if (file_exists($archivePath)) {
                unlink($archivePath);
            }
            // Удаляем временную директорию если пуста
            $tempDir = dirname($archivePath);
            if (file_exists($tempDir) && count(scandir($tempDir)) <= 2) {
                FileHelper::removeDirectory($tempDir);
            }
        });

        return $response;
    }
}