<?php

namespace app\core\services\backup;

use app\core\helpers\SettingsHelper;
use Exception;
use FilesystemIterator;
use RecursiveDirectoryIterator;
use RecursiveIteratorIterator;
use Yii;
use yii\base\Component;
use yii\helpers\FileHelper;
use yii\db\Connection;
use yii\db\Schema;
use yii\web\Response;
use ZipArchive;

class BackupManagerOrigin extends Component
{
    public string $backupPath = '@app/backups';
    public int $maxBackups = 50;
    public int $maxSqlBackups = 30;
    public int $maxFullBackups = 7;

    public function __construct($config = [])
    {
        parent::__construct($config);
        $this->backupPath = SettingsHelper::getSetting('backup_path', '@app/backups');
        $this->maxBackups = SettingsHelper::getSetting('max_backups', 50);
        $this->maxSqlBackups = SettingsHelper::getSetting('max_sql_backups', 30);
        $this->maxFullBackups = SettingsHelper::getSetting('max_full_backups', 7);
    }

    /**
     * Создать новый бэкап
     */
    public function createBackup(string $type, string $description = null, string $createdBy = null): array
    {
        if (!in_array($type, ['sql', 'full'])) {
            throw new \InvalidArgumentException("Неправильный тип бэкапа: {$type}");
        }

        $timestamp = date('Y-m-d_H-i-s');
        $backupId = "backup_{$timestamp}_{$type}";
        $backupDir = Yii::getAlias("{$this->backupPath}/{$backupId}");

        FileHelper::createDirectory($backupDir);

        $manifest = [
            'id' => $backupId,
            'type' => $type,
            'description' => $description,
            'created_at' => date('c'),
            'created_by' => $createdBy,
            'system_info' => $this->getSystemInfo(),
        ];

        try {
            // Создаем бэкап БД (PHP-способом)
            if ($type === 'sql' || $type === 'full') {
                $manifest['database'] = $this->backupDatabase($backupDir);
            }

            // Архивируем файлы (PHP-способом)
            if ($type === 'full') {
                $manifest['files'] = $this->backupFiles($backupDir);
            }

            // Сохраняем манифест
            $manifestFile = "{$backupDir}/manifest.json";
            file_put_contents($manifestFile,
                json_encode($manifest, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE)
            );

            // Ротация
            $deletedBackups = $this->rotateBackups();
            $totalSize = $this->calculateDirectorySize($backupDir);

            return [
                'success' => true,
                'backup_id' => $backupId,
                'path' => $backupDir,
                'manifest' => $manifest,
                'deleted_old_backups' => $deletedBackups,
                'total_size' => $totalSize,
                'total_size_formatted' => $this->formatBytes($totalSize),
            ];

        } catch (Exception $e) {
            if (file_exists($backupDir)) {
                FileHelper::removeDirectory($backupDir);
            }
            throw $e;
        }
    }

    /**
     * Создать дамп БД с помощью PHP
     */
    private function backupDatabase(string $backupDir): array
    {
        $db = Yii::$app->db;
        $dumpFile = "{$backupDir}/database.sql";

        if ($db->driverName !== 'pgsql') {
            throw new Exception("Поддерживается только PostgreSQL");
        }

        $dumpContent = $this->generateSqlDump($db);

        if (file_put_contents($dumpFile, $dumpContent) === false) {
            throw new Exception("Не удалось записать дамп БД в файл");
        }

        $tablesCount = $this->countTables($db);

        return [
            'file' => 'database.sql',
            'size' => filesize($dumpFile),
            'tables_count' => $tablesCount,
            'encoding' => 'UTF-8',
            'generated_by' => 'php',
        ];
    }

    /**
     * Генерация SQL дампа средствами PHP
     */
    private function generateSqlDump(Connection $db): string
    {
        $output = "";

        // Заголовок
        $output .= "-- SQL Dump generated by PHP\n";
        $output .= "-- Date: " . date('Y-m-d H:i:s') . "\n";
        $output .= "-- Database: " . $this->getDbNameFromDsn($db->dsn) . "\n";
        $output .= "-- PHP Version: " . PHP_VERSION . "\n\n";

        // Отключаем проверки для ускорения
        $output .= "SET statement_timeout = 0;\n";
        $output .= "SET lock_timeout = 0;\n";
        $output .= "SET idle_in_transaction_session_timeout = 0;\n";
        $output .= "SET client_encoding = 'UTF8';\n";
        $output .= "SET standard_conforming_strings = on;\n";
        $output .= "SELECT pg_catalog.set_config('search_path', '', false);\n";
        $output .= "SET check_function_bodies = false;\n";
        $output .= "SET xmloption = content;\n";
        $output .= "SET client_min_messages = warning;\n";
        $output .= "SET row_security = off;\n\n";

        // Получаем все таблицы
        $tables = $this->getAllTables($db);

        foreach ($tables as $table) {
            // Создание таблицы
            $output .= $this->getTableCreateSql($db, $table) . ";\n\n";

            // Данные таблицы
            $output .= $this->getTableDataSql($db, $table);
        }

        // Индексы и констрейнты после данных
        foreach ($tables as $table) {
            $output .= $this->getTableIndexesSql($db, $table);
            $output .= $this->getTableConstraintsSql($db, $table);
        }

        // Последовательности (sequences)
        $output .= $this->getSequencesSql($db);

        return $output;
    }

    /**
     * Архивировать файлы с помощью PHP
     */
    private function backupFiles(string $backupDir): array
    {
        $uploadsPath = Yii::getAlias('@webroot/uploads');
        $eventsPath = Yii::getAlias('@webroot/uploads');
        $avatarsPath = Yii::getAlias('@webroot/images/avatars');
        $archiveFile = "{$backupDir}/uploads.zip";

        // Создаем ZIP архив средствами PHP
        $zip = new ZipArchive();

        if ($zip->open($archiveFile, ZipArchive::CREATE | ZipArchive::OVERWRITE) !== true) {
            throw new Exception("Не удалось создать ZIP архив");
        }

        $result_uploads = $this->addFolderToZip($zip, $uploadsPath);
        $result_events = $this->addFolderToZip($zip, $eventsPath);
        $result_avatars = $this->addFolderToZip($zip, $avatarsPath);

        $filesCount = $result_uploads['filesCount'] + $result_events['filesCount'] + $result_avatars['filesCount'];
        $totalSize = $result_uploads['totalSize'] + $result_events['totalSize'] + $result_avatars['totalSize'];

        $zip->close();

        if (!file_exists($archiveFile)) {
            throw new Exception("Архив создан, но файл отсутствует");
        }

        return [
            'file' => 'uploads.zip',
            'size' => filesize($archiveFile),
            'files_count' => $filesCount,
            'original_size' => $totalSize,
            'compression_ratio' => $totalSize > 0 ? round(filesize($archiveFile) / $totalSize * 100, 2) : 0,
            'format' => 'zip',
        ];
    }

    private function addFolderToZip(ZipArchive $zip, string $folder): array
    {
        $filesCount = 0;
        $totalSize = 0;

        if (file_exists($folder) && is_dir($folder)) {
            $iterator = new RecursiveIteratorIterator(
                new RecursiveDirectoryIterator($folder, RecursiveDirectoryIterator::SKIP_DOTS),
                RecursiveIteratorIterator::SELF_FIRST
            );

            foreach ($iterator as $item) {
                if ($item->isFile()) {
                    $filePath = $item->getRealPath();
                    $relativePath = substr($filePath, strlen($folder) + 1);

                    if ($zip->addFile($filePath, $relativePath)) {
                        $filesCount++;
                        $totalSize += $item->getSize();
                    }
                }
            }
        } else {
            // Создаем пустую папку
            FileHelper::createDirectory($folder);
            file_put_contents("{$folder}/README.txt", "Папка $folder была пуста на момент бэкапа");
            $zip->addFile("{$folder}/README.txt", "README.txt");
            $filesCount = 1;
            $totalSize = filesize("{$folder}/README.txt");
        }

        return [
          'filesCount' => $filesCount,
          'totalSize' => $totalSize,
        ];
    }

    public function sizeBackups()
    {
        $backupPath = Yii::getAlias($this->backupPath);

        if (!is_dir($backupPath)) {
            return 0;
        }

        $size = 0;
        $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($backupPath, FilesystemIterator::SKIP_DOTS)
        );

        foreach ($iterator as $file) {
            if ($file->isFile()) {
                $size += $file->getSize();
            }
        }

        return $size;
    }

    public function lastBackupsDate(): ?int
    {
        $backupPath = Yii::getAlias($this->backupPath);

        if (!is_dir($backupPath)) {
            return null;
        }

        return \app\core\helpers\FileHelper::getNewestFolderDate($backupPath);
    }

    /**
     * Получить список всех бэкапов
     */
    public function listBackups(): array
    {
        $backupPath = Yii::getAlias($this->backupPath);

        // Создаем папку если не существует
        if (!file_exists($backupPath)) {
            FileHelper::createDirectory($backupPath);
            return [];
        }

        $backupDirs = glob("{$backupPath}/backup_*", GLOB_ONLYDIR);
        $result = [];

        foreach ($backupDirs as $backupDir) {
            $manifestFile = "{$backupDir}/manifest.json";

            if (!file_exists($manifestFile)) {
                continue;
            }

            try {
                $manifest = json_decode(file_get_contents($manifestFile), true);

                if (!is_array($manifest)) {
                    continue;
                }

                $size = $this->calculateDirectorySize($backupDir);
                $backupId = basename($backupDir);

                $result[] = [
                    'id' => $backupId,
                    'path' => $backupDir,
                    'manifest' => $manifest,
                    'size' => $size,
                    'size_formatted' => $this->formatBytes($size),
                    'created_at' => strtotime($manifest['created_at'] ?? filemtime($backupDir)),
                    'type' => $manifest['type'] ?? 'unknown',
                ];
            } catch (Exception $e) {
                continue;
            }
        }

        // Сортируем по дате создания (новые сначала)
        usort($result, function($a, $b) {
            return $b['created_at'] <=> $a['created_at'];
        });

        return $result;
    }

    /**
     * Удалить бэкап
     */
    public function deleteBackup(string $backupId): bool
    {
        $backupDir = Yii::getAlias("{$this->backupPath}/{$backupId}");

        if (!file_exists($backupDir) || !is_dir($backupDir)) {
            return false;
        }

        try {
            FileHelper::removeDirectory($backupDir);
            return true;
        } catch (Exception $e) {
            return false;
        }
    }

    /**
     * Восстановить из бэкапа
     */
    public function restoreBackup(string $backupId, bool $restoreDatabase = true, bool $restoreFiles = true): array
    {
        $backupDir = Yii::getAlias("{$this->backupPath}/{$backupId}");

        if (!file_exists($backupDir)) {
            throw new Exception("Бэкап не найден: {$backupId}");
        }

        $manifestFile = "{$backupDir}/manifest.json";
        if (!file_exists($manifestFile)) {
            throw new Exception("Манифест бэкапа не найден");
        }

        $manifest = json_decode(file_get_contents($manifestFile), true);
        if (!is_array($manifest)) {
            throw new Exception("Неверный формат манифеста");
        }

        $type = $manifest['type'] ?? 'unknown';
        $steps = [];

        // Восстанавливаем БД
        if ($restoreDatabase && ($type === 'sql' || $type === 'full')) {
            $steps[] = $this->restoreDatabase($backupDir);
        }

        // Восстанавливаем файлы
        if ($restoreFiles && $type === 'full') {
            $steps[] = $this->restoreFiles($backupDir);
        }

        return [
            'success' => true,
            'backup_id' => $backupId,
            'type' => $type,
            'manifest' => $manifest,
            'steps' => $steps,
            'message' => 'Восстановление выполнено успешно',
        ];
    }

    /**
     * Восстановить БД из SQL
     */
    private function restoreDatabase(string $backupDir): array
    {
        $dumpFile = "{$backupDir}/database.sql";

        if (!file_exists($dumpFile)) {
            throw new Exception("Файл дампа БД не найден");
        }

        $db = Yii::$app->db;

        // Читаем файл и выполняем запросы
        $sqlContent = file_get_contents($dumpFile);

        // Разбиваем на отдельные запросы
        $queries = $this->splitSqlQueries($sqlContent);

        $executed = 0;
        $errors = [];

        // Отключаем проверки для ускорения
        $db->createCommand("SET session_replication_role = 'replica';")->execute();

        foreach ($queries as $query) {
            $query = trim($query);

            if (empty($query) || strpos($query, '--') === 0) {
                continue;
            }

            try {
                $db->createCommand($query)->execute();
                $executed++;
            } catch (Exception $e) {
                $errors[] = "Ошибка в запросе: " . substr($query, 0, 100) . "... - " . $e->getMessage();
            }
        }

        // Включаем проверки обратно
        $db->createCommand("SET session_replication_role = 'origin';")->execute();

        return [
            'step' => 'database',
            'success' => true,
            'queries_executed' => $executed,
            'errors' => $errors,
        ];
    }

    /**
     * Восстановить файлы из архива
     */
    private function restoreFiles(string $backupDir): array
    {
        $archiveFile = "{$backupDir}/uploads.zip";

        if (!file_exists($archiveFile)) {
            throw new Exception("Архив файлов не найден");
        }

        $uploadsPath = Yii::getAlias('@webroot/uploads');

        // Создаем бэкап текущих файлов
        $backupCurrentPath = Yii::getAlias('@runtime/uploads_backup_' . date('Y-m-d_H-i-s'));
        if (file_exists($uploadsPath)) {
            FileHelper::copyDirectory($uploadsPath, $backupCurrentPath);
        }

        // Очищаем папку uploads
        if (file_exists($uploadsPath)) {
            FileHelper::removeDirectory($uploadsPath);
        }

        // Создаем пустую папку
        FileHelper::createDirectory($uploadsPath);

        // Распаковываем ZIP архив
        $zip = new ZipArchive();

        if ($zip->open($archiveFile) !== true) {
            // Пытаемся восстановить из бэкапа
            if (file_exists($backupCurrentPath)) {
                FileHelper::copyDirectory($backupCurrentPath, $uploadsPath);
            }
            throw new Exception("Не удалось открыть ZIP архив");
        }

        $extracted = $zip->extractTo($uploadsPath);
        $zip->close();

        if (!$extracted) {
            // Пытаемся восстановить из бэкапа
            if (file_exists($backupCurrentPath)) {
                FileHelper::copyDirectory($backupCurrentPath, $uploadsPath);
            }
            throw new Exception("Не удалось распаковать архив");
        }

        // Удаляем временный бэкап
        if (file_exists($backupCurrentPath)) {
            FileHelper::removeDirectory($backupCurrentPath);
        }

        return [
            'step' => 'files',
            'success' => true,
            'archive' => $archiveFile,
            'destination' => $uploadsPath,
        ];
    }

    /**
     * Автоматическая ротация бэкапов
     */
    private function rotateBackups(): array
    {
        $allBackups = $this->listBackups();
        $deleted = [];

        // Разделяем бэкапы по типам
        $sqlBackups = [];
        $fullBackups = [];

        foreach ($allBackups as $backup) {
            if ($backup['type'] === 'sql') {
                $sqlBackups[] = $backup;
            } elseif ($backup['type'] === 'full') {
                $fullBackups[] = $backup;
            }
        }

        // Удаляем лишние SQL бэкапы (старые в начале массива после сортировки)
        if (count($sqlBackups) > $this->maxSqlBackups) {
            $toDelete = array_slice($sqlBackups, $this->maxSqlBackups);
            foreach ($toDelete as $backup) {
                if ($this->deleteBackup($backup['id'])) {
                    $deleted[] = $backup['id'];
                }
            }
        }

        // Удаляем лишние FULL бэкапы
        if (count($fullBackups) > $this->maxFullBackups) {
            $toDelete = array_slice($fullBackups, $this->maxFullBackups);
            foreach ($toDelete as $backup) {
                if ($this->deleteBackup($backup['id'])) {
                    $deleted[] = $backup['id'];
                }
            }
        }

        // Общий лимит на все бэкапы
        $totalBackups = count($sqlBackups) + count($fullBackups);
        if ($totalBackups > $this->maxBackups) {
            $allBackupsSorted = $allBackups;
            usort($allBackupsSorted, function($a, $b) {
                return $a['created_at'] <=> $b['created_at']; // старые в начале
            });

            $toDelete = array_slice($allBackupsSorted, 0, $totalBackups - $this->maxBackups);
            foreach ($toDelete as $backup) {
                if (!in_array($backup['id'], $deleted) && $this->deleteBackup($backup['id'])) {
                    $deleted[] = $backup['id'];
                }
            }
        }

        return $deleted;
    }

    /**
     * Получить информацию о системе
     */
    private function getSystemInfo(): array
    {
        $db = Yii::$app->db;

        return [
            'app' => [
                'name' => Yii::$app->name,
                'version' => '1.0.0',
                'environment' => YII_ENV,
                'debug' => YII_DEBUG,
            ],
            'server' => [
                'php_version' => PHP_VERSION,
                'yii_version' => Yii::getVersion(),
            ],
            'database' => [
                'driver' => $db->driverName,
                'tables_count' => $this->countTables($db),
            ],
            'paths' => [
                'backup' => Yii::getAlias($this->backupPath),
                'uploads' => Yii::getAlias('@webroot/uploads'),
            ],
            'timestamp' => time(),
            'date' => date('c'),
        ];
    }

    /**
     * Получить имя БД из DSN
     */
    private function getDbNameFromDsn(string $dsn): string
    {
        preg_match('/dbname=([^;]+)/', $dsn, $matches);
        return $matches[1] ?? 'unknown';
    }

    /**
     * Получить все таблицы БД
     */
    private function getAllTables(Connection $db): array
    {
        $sql = "SELECT table_name 
                FROM information_schema.tables 
                WHERE table_schema = 'public' 
                AND table_type = 'BASE TABLE'
                ORDER BY table_name";

        return $db->createCommand($sql)->queryColumn();
    }

    /**
     * Получить SQL для создания таблицы
     */
    private function getTableCreateSql(Connection $db, string $table): string
    {
        $sql = "SELECT 
                    'CREATE TABLE ' || quote_ident(table_name) || ' (' || 
                    string_agg(
                        quote_ident(column_name) || ' ' || 
                        data_type || 
                        CASE 
                            WHEN character_maximum_length IS NOT NULL 
                            THEN '(' || character_maximum_length || ')' 
                            ELSE '' 
                        END || 
                        CASE 
                            WHEN is_nullable = 'NO' THEN ' NOT NULL' 
                            ELSE '' 
                        END || 
                        CASE 
                            WHEN column_default IS NOT NULL 
                            THEN ' DEFAULT ' || column_default 
                            ELSE '' 
                        END,
                        ', '
                        ORDER BY ordinal_position
                    ) || 
                    ')' as create_sql
                FROM information_schema.columns 
                WHERE table_schema = 'public' 
                AND table_name = :table
                GROUP BY table_name";

        $result = $db->createCommand($sql, [':table' => $table])->queryOne();
        return $result['create_sql'] ?? "CREATE TABLE {$table} (id SERIAL PRIMARY KEY)";
    }

    /**
     * Получить данные таблицы в виде INSERT запросов
     */
    private function getTableDataSql(Connection $db, string $table): string
    {
        $output = "";

        try {
            $batchSize = 1000;
            $offset = 0;
            $hasData = false;

            while (true) {
                $sql = "SELECT * FROM \"{$table}\" LIMIT {$batchSize} OFFSET {$offset}";
                $rows = $db->createCommand($sql)->queryAll();

                if (empty($rows)) {
                    break;
                }

                $hasData = true;

                foreach ($rows as $row) {
                    $columns = [];
                    $values = [];

                    foreach ($row as $key => $value) {
                        $columns[] = "\"{$key}\"";
                        $values[] = $this->escapeSqlValue($value);
                    }

                    $columnsStr = implode(', ', $columns);
                    $valuesStr = implode(', ', $values);

                    $output .= "INSERT INTO \"{$table}\" ({$columnsStr}) VALUES ({$valuesStr});\n";
                }

                $offset += $batchSize;
            }

            if ($hasData) {
                $output .= "\n";
            }

        } catch (Exception $e) {
            $output .= "-- Error dumping data for table {$table}: " . $e->getMessage() . "\n\n";
        }

        return $output;
    }

    /**
     * Экранирование SQL значений
     */
    private function escapeSqlValue($value): string
    {
        if ($value === null) {
            return 'NULL';
        }

        if (is_numeric($value)) {
            return $value;
        }

        if (is_bool($value)) {
            return $value ? 'TRUE' : 'FALSE';
        }

        $value = str_replace("'", "''", $value);
        $value = str_replace("\\", "\\\\", $value);

        return "'{$value}'";
    }

    /**
     * Получить индексы таблицы
     */
    private function getTableIndexesSql(Connection $db, string $table): string
    {
        $output = "";

        $sql = "SELECT 
                    indexname, indexdef
                FROM pg_indexes
                WHERE schemaname = 'public' 
                AND tablename = :table
                AND indexname NOT LIKE '%_pkey'";

        $indexes = $db->createCommand($sql, [':table' => $table])->queryAll();

        foreach ($indexes as $index) {
            $output .= $index['indexdef'] . ";\n";
        }

        if (!empty($indexes)) {
            $output .= "\n";
        }

        return $output;
    }

    /**
     * Получить констрейнты таблицы
     */
    private function getTableConstraintsSql(Connection $db, string $table): string
    {
        $output = "";

        $sql = "SELECT
                    conname,
                    pg_get_constraintdef(oid) as condef
                FROM pg_constraint
                WHERE conrelid = :table::regclass
                AND contype = 'f'";

        $constraints = $db->createCommand($sql, [':table' => $table])->queryAll();

        foreach ($constraints as $constraint) {
            $output .= "ALTER TABLE \"{$table}\" ADD CONSTRAINT \"{$constraint['conname']}\" {$constraint['condef']};\n";
        }

        if (!empty($constraints)) {
            $output .= "\n";
        }

        return $output;
    }

    /**
     * Получить последовательности
     */
    private function getSequencesSql(Connection $db): string
    {
        $output = "";

        $sql = "SELECT 
                sequence_name,
                start_value, 
                minimum_value, 
                maximum_value, 
                increment, 
                cycle_option
            FROM information_schema.sequences 
            WHERE sequence_schema = 'public'";

        try {
            $sequences = $db->createCommand($sql)->queryAll();

            foreach ($sequences as $sequence) {
                $output .= "CREATE SEQUENCE \"{$sequence['sequence_name']}\"\n";
                $output .= "    START WITH {$sequence['start_value']}\n";
                $output .= "    INCREMENT BY {$sequence['increment']}\n";
                $output .= "    MINVALUE {$sequence['minimum_value']}\n";
                $output .= "    MAXVALUE {$sequence['maximum_value']}\n";
                $output .= "    " . ($sequence['cycle_option'] === 'YES' ? 'CYCLE' : 'NO CYCLE') . ";\n\n";
            }
        } catch (Exception $e) {
            // Пропускаем ошибки с последовательностями
            $output .= "-- Error getting sequences: " . $e->getMessage() . "\n\n";
        }

        return $output;
    }

    /**
     * Разбить SQL на отдельные запросы
     */
    private function splitSqlQueries(string $sql): array
    {
        $sql = preg_replace('/--.*$/m', '', $sql);

        $queries = [];
        $currentQuery = '';
        $inString = false;
        $stringChar = '';

        for ($i = 0; $i < strlen($sql); $i++) {
            $char = $sql[$i];

            if ($char === "'" || $char === '"') {
                if (!$inString) {
                    $inString = true;
                    $stringChar = $char;
                } elseif ($stringChar === $char && $sql[$i-1] !== '\\') {
                    $inString = false;
                }
            }

            $currentQuery .= $char;

            if ($char === ';' && !$inString) {
                $queries[] = $currentQuery;
                $currentQuery = '';
            }
        }

        if (!empty($currentQuery)) {
            $queries[] = $currentQuery;
        }

        return $queries;
    }

    /**
     * Посчитать количество таблиц в БД
     */
    private function countTables(Connection $db): int
    {
        try {
            if ($db->driverName === 'pgsql') {
                $sql = "SELECT COUNT(*) FROM information_schema.tables 
                        WHERE table_schema = 'public' 
                        AND table_type = 'BASE TABLE'";
                return (int) $db->createCommand($sql)->queryScalar();
            }
        } catch (Exception $e) {
        }

        return 0;
    }

    /**
     * Посчитать размер директории
     */
    private function calculateDirectorySize(string $directory): int
    {
        if (!file_exists($directory) || !is_dir($directory)) {
            return 0;
        }

        $size = 0;
        $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($directory, RecursiveDirectoryIterator::SKIP_DOTS)
        );

        foreach ($iterator as $file) {
            if ($file->isFile()) {
                $size += $file->getSize();
            }
        }

        return $size;
    }

    /**
     * Форматировать размер в байтах
     */
    public function formatBytes(int $bytes, int $precision = 2): string
    {
        $units = ['B', 'KB', 'MB', 'GB', 'TB'];

        $bytes = max($bytes, 0);
        $pow = floor(($bytes ? log($bytes) : 0) / log(1024));
        $pow = min($pow, count($units) - 1);

        $bytes /= pow(1024, $pow);

        return round($bytes, $precision) . ' ' . $units[$pow];
    }

    /**
     * Получить статистику по бэкапам
     */
    public function getStats(): array
    {
        $backups = $this->listBackups();

        $sqlCount = 0;
        $fullCount = 0;
        $totalSize = 0;
        $sqlSize = 0;
        $fullSize = 0;

        foreach ($backups as $backup) {
            $totalSize += $backup['size'];

            if ($backup['type'] === 'sql') {
                $sqlCount++;
                $sqlSize += $backup['size'];
            } elseif ($backup['type'] === 'full') {
                $fullCount++;
                $fullSize += $backup['size'];
            }
        }

        $backupPath = Yii::getAlias($this->backupPath);
        $diskFree = disk_free_space(dirname($backupPath));
        $diskTotal = disk_total_space(dirname($backupPath));

        return [
            'total' => count($backups),
            'sql_count' => $sqlCount,
            'full_count' => $fullCount,
            'size' => [
                'total' => $totalSize,
                'total_formatted' => $this->formatBytes($totalSize),
                'sql' => $sqlSize,
                'sql_formatted' => $this->formatBytes($sqlSize),
                'full' => $fullSize,
                'full_formatted' => $this->formatBytes($fullSize),
            ],
            'disk' => [
                'free' => $diskFree,
                'free_formatted' => $this->formatBytes($diskFree),
                'total' => $diskTotal,
                'total_formatted' => $this->formatBytes($diskTotal),
                'used_percent' => $diskTotal > 0 ? round(($diskTotal - $diskFree) / $diskTotal * 100, 2) : 0,
            ],
            'limits' => [
                'max_backups' => $this->maxBackups,
                'max_sql_backups' => $this->maxSqlBackups,
                'max_full_backups' => $this->maxFullBackups,
            ],
        ];
    }

    /**
     * Проверить требования системы
     */
    public function checkRequirements(): array
    {
        $requirements = [];

        // Проверка расширения ZIP
        $requirements['zip_extension'] = [
            'available' => extension_loaded('zip'),
            'message' => extension_loaded('zip')
                ? 'Расширение ZIP доступно'
                : 'Установите расширение ZIP'
        ];

        // Проверка директории для бэкапов
        $backupPath = Yii::getAlias($this->backupPath);
        $requirements['backup_directory'] = [
            'exists' => file_exists($backupPath),
            'writable' => is_writable($backupPath) || (!file_exists($backupPath) && is_writable(dirname($backupPath))),
            'path' => $backupPath,
        ];

        // Проверка папки uploads
        $uploadsPath = Yii::getAlias('@webroot/uploads');
        $requirements['uploads_directory'] = [
            'exists' => file_exists($uploadsPath),
            'writable' => is_writable($uploadsPath) || (!file_exists($uploadsPath) && is_writable(dirname($uploadsPath))),
            'path' => $uploadsPath,
        ];

        // Общая проверка
        $requirements['all_ok'] =
            $requirements['zip_extension']['available'] &&
            $requirements['backup_directory']['writable'];

        return $requirements;
    }

    /**
     * Скачать бэкап как архив
     */
    public function download(string $backupId): array
    {
        $backupDir = Yii::getAlias("{$this->backupPath}/{$backupId}");

        if (!file_exists($backupDir) || !is_dir($backupDir)) {
            throw new Exception("Бэкап не найден: {$backupId}");
        }

        // Создаем временный архив
        $tempDir = Yii::getAlias('@runtime/temp_backup_download');
        FileHelper::createDirectory($tempDir);

        $tempArchive = $tempDir . '/' . $backupId . '.tar.gz';

        // Архивируем папку бэкапа
        $this->createTarGzArchive($backupDir, $tempArchive);

        if (!file_exists($tempArchive)) {
            throw new Exception("Не удалось создать архив для скачивания");
        }

        return [
            'success' => true,
            'backup_id' => $backupId,
            'archive_path' => $tempArchive,
            'archive_name' => $backupId . '.tar.gz',
            'size' => filesize($tempArchive),
        ];
    }

    /**
     * Создать tar.gz архив средствами PHP
     */
    private function createTarGzArchive(string $sourceDir, string $outputFile): void
    {
        // Если есть команда tar, используем ее (быстрее)
        if (function_exists('exec')) {
            exec('which tar 2>/dev/null', $output, $returnCode);
            if ($returnCode === 0) {
                $command = sprintf(
                    'tar -czf %s -C %s . 2>&1',
                    escapeshellarg($outputFile),
                    escapeshellarg($sourceDir)
                );
                exec($command, $output, $returnCode);
                if ($returnCode === 0) {
                    return;
                }
            }
        }

        // Иначе создаем ZIP архив (менее эффективно, но работает везде)
        $this->createZipArchive($sourceDir, $outputFile);
    }

    /**
     * Создать ZIP архив (альтернатива tar.gz)
     */
    private function createZipArchive(string $sourceDir, string $outputFile): void
    {
        // Меняем расширение на .zip
        $outputFile = preg_replace('/\.tar\.gz$/', '.zip', $outputFile);

        $zip = new ZipArchive();

        if ($zip->open($outputFile, ZipArchive::CREATE | ZipArchive::OVERWRITE) !== true) {
            throw new Exception("Не удалось создать ZIP архив");
        }

        $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($sourceDir, RecursiveDirectoryIterator::SKIP_DOTS),
            RecursiveIteratorIterator::SELF_FIRST
        );

        foreach ($iterator as $item) {
            if ($item->isFile()) {
                $filePath = $item->getRealPath();
                $relativePath = substr($filePath, strlen($sourceDir) + 1);

                $zip->addFile($filePath, $relativePath);
            }
        }

        $zip->close();
    }

    /**
     * Отправить файл для скачивания
     */
    public function sendDownloadResponse(string $backupId): Response
    {
        $downloadInfo = $this->download($backupId);
        $archivePath = $downloadInfo['archive_path'];
        $archiveName = $downloadInfo['archive_name'];

        if (!file_exists($archivePath)) {
            throw new Exception("Архив не найден");
        }

        // Отправляем файл
        $response = Yii::$app->response;
        $response->sendFile($archivePath, $archiveName, [
            'mimeType' => $this->getMimeType($archivePath),
            'inline' => false,
        ]);

        // Удаляем временный файл после отправки
        register_shutdown_function(function() use ($archivePath) {
            if (file_exists($archivePath)) {
                unlink($archivePath);
            }
            // Удаляем временную директорию если пуста
            $tempDir = dirname($archivePath);
            if (file_exists($tempDir) && count(scandir($tempDir)) <= 2) {
                FileHelper::removeDirectory($tempDir);
            }
        });

        return $response;
    }

    /**
     * Получить MIME тип файла
     */
    private function getMimeType(string $filePath): string
    {
        $extension = strtolower(pathinfo($filePath, PATHINFO_EXTENSION));

        $mimeTypes = [
            'tar.gz' => 'application/gzip',
            'tgz' => 'application/gzip',
            'zip' => 'application/zip',
            'sql' => 'application/sql',
            'json' => 'application/json',
        ];

        return $mimeTypes[$extension] ?? 'application/octet-stream';
    }

    /**
     * Получить прямую ссылку для скачивания
     */
    public function getDownloadUrl(string $backupId): string
    {
        return Yii::$app->urlManager->createAbsoluteUrl(['/backup/download', 'id' => $backupId]);
    }
}