<?php

namespace app\core\services\event;

use app\core\entities\EventField;
use app\core\forms\event\EventFieldForm;
use app\core\repositories\event\EventFieldRepository;
use app\core\repositories\event\EventRepository;
use app\core\services\TransactionManager;
use DomainException;

class EventFieldService
{
    private EventRepository $eventRepository;
    private EventFieldRepository $fieldRepository;
    private TransactionManager $transaction;

    public function __construct(
        EventRepository $eventRepository,
        EventFieldRepository $fieldRepository,
        TransactionManager $transaction
    ) {
        $this->eventRepository = $eventRepository;
        $this->fieldRepository = $fieldRepository;
        $this->transaction = $transaction;
    }

    /**
     * Создание кастомного поля
     */
    public function create(string $eventId, EventFieldForm $form): EventField
    {
        $event = $this->eventRepository->get($eventId);

        // Проверка уникальности имени поля
        if ($this->fieldRepository->existsByName($eventId, $form->name)) {
            throw new DomainException('Поле с таким именем уже существует');
        }

        $sortOrder = $form->sort_order ?? $this->fieldRepository->getNextSortOrder($eventId);

        $field = EventField::create(
            $eventId,
            $form->name,
            $form->label,
            $form->type,
            $form->required,
            $sortOrder,
            $form->options
        );

        $this->transaction->wrap(function () use ($field) {
            $this->fieldRepository->save($field);
        });

        return $field;
    }

    /**
     * Редактирование поля
     */
    public function edit(string $id, EventFieldForm $form): void
    {
        $field = $this->fieldRepository->get($id);

        // Проверка уникальности имени (кроме текущего поля)
        if ($this->fieldRepository->existsByName($field->event_id, $form->name, $field->id)) {
            throw new DomainException('Поле с таким именем уже существует');
        }

        $field->edit(
            $form->label,
            $form->type,
            $form->required,
            $form->sort_order,
            $form->options
        );

        $this->transaction->wrap(function () use ($field) {
            $this->fieldRepository->save($field);
        });
    }

    /**
     * Удаление поля
     */
    public function remove(string $id): void
    {
        $field = $this->fieldRepository->get($id);
        $this->fieldRepository->remove($field);
    }

    /**
     * Изменение порядка полей
     */
    public function reorder(string $eventId, array $fieldIds): void
    {
        $fields = $this->fieldRepository->getByEvent($eventId);
        $fieldsById = [];

        foreach ($fields as $field) {
            $fieldsById[$field->id] = $field;
        }

        $this->transaction->wrap(function () use ($fieldIds, $fieldsById) {
            foreach ($fieldIds as $index => $fieldId) {
                if (isset($fieldsById[$fieldId])) {
                    $fieldsById[$fieldId]->sort_order = $index;
                    $this->fieldRepository->save($fieldsById[$fieldId]);
                }
            }
        });
    }

    /**
     * Создание стандартных полей
     */
    public function createDefaultFields(string $eventId): void
    {
        $defaultFields = $this->getDefaultFields();
        $sortOrder = $this->fieldRepository->getNextSortOrder($eventId);

        $this->transaction->wrap(function () use ($eventId, $defaultFields, $sortOrder) {
            foreach ($defaultFields as $fieldData) {
                $field = EventField::create(
                    $eventId,
                    $fieldData['name'],
                    $fieldData['label'],
                    $fieldData['type'],
                    $fieldData['required'],
                    $sortOrder,
                    $fieldData['options'] ?? null
                );

                $this->fieldRepository->save($field);
                $sortOrder++;
            }
        });
    }

    /**
     * Копирование полей из другого мероприятия
     */
    public function copyFields(string $sourceEventId, string $targetEventId): void
    {
        $sourceFields = $this->fieldRepository->getByEvent($sourceEventId);
        $sortOrder = $this->fieldRepository->getNextSortOrder($targetEventId);

        $this->transaction->wrap(function () use ($sourceFields, $targetEventId, $sortOrder) {
            foreach ($sourceFields as $sourceField) {
                $field = EventField::create(
                    $targetEventId,
                    $sourceField->name . '_copy',
                    $sourceField->label . ' (копия)',
                    $sourceField->type,
                    $sourceField->required,
                    $sortOrder,
                    $sourceField->getOptionsArray()
                );

                $this->fieldRepository->save($field);
                $sortOrder++;
            }
        });
    }

    /**
     * Стандартные поля
     */
    private function getDefaultFields(): array
    {
        return [
            [
                'name' => 'first_name',
                'label' => 'Имя',
                'type' => EventField::TYPE_TEXT,
                'required' => true,
            ],
            [
                'name' => 'last_name',
                'label' => 'Фамилия',
                'type' => EventField::TYPE_TEXT,
                'required' => true,
            ],
            [
                'name' => 'email',
                'label' => 'Email',
                'type' => EventField::TYPE_EMAIL,
                'required' => false,
            ],
            [
                'name' => 'phone',
                'label' => 'Телефон',
                'type' => EventField::TYPE_PHONE,
                'required' => false,
            ],
            [
                'name' => 'company',
                'label' => 'Компания',
                'type' => EventField::TYPE_TEXT,
                'required' => false,
            ],
            [
                'name' => 'position',
                'label' => 'Должность',
                'type' => EventField::TYPE_TEXT,
                'required' => false,
            ],
        ];
    }
}
