<?php

namespace app\core\services\event;

use app\core\entities\Event;
use app\core\entities\User;
use app\core\forms\event\EventCreateForm;
use app\core\forms\event\EventForm;
use app\core\repositories\event\EventRepository;
use app\core\repositories\user\UserRepository;
use app\core\services\TransactionManager;
use DomainException;
use Yii;
use yii\web\UploadedFile;

class EventService
{
    private EventRepository $eventRepository;
    private TransactionManager $transaction;
    private UserRepository $userRepository;

    public function __construct(
        EventRepository $eventRepository,
        TransactionManager $transaction,
        UserRepository $userRepository
    ) {
        $this->eventRepository = $eventRepository;
        $this->transaction = $transaction;
        $this->userRepository = $userRepository;
    }

    public function create(EventCreateForm $form): Event
    {
        $event = Event::create(
            $form->title,
            $form->slug,
            $form->description,
            $form->start_date ?? null,
            $form->end_date ?? null,
            Yii::$app->user->id,
            $form->short_description,
            $form->content ? json_decode($form->content, true) : null,
            $form->type,
            $form->location_type,
            $form->location,
            null, // image будет обрабатываться отдельно
            $form->registration_start ?? null,
            $form->registration_end ?? null,
            $form->max_participants || 0,
            $form->price,
            $form->currency,
            $form->settings ? json_decode($form->settings, true) : null,
            $form->meta_title,
            $form->meta_description,
            $form->meta_keywords,
            $form->color
        );

        $this->transaction->wrap(function () use ($event) {
            $this->eventRepository->save($event);
        });

        return $event;
    }

    public function edit(string $id, EventForm $form): void
    {
        $event = $this->eventRepository->get($id);

        $event->edit(
            $form->title,
            $form->slug,
            $form->description,
            $form->start_date ?? null,
            $form->end_date ?? null,
            $form->short_description,
            $form->content ? json_decode($form->content, true) : null,
            $form->type,
            $form->location_type,
            $form->location,
            $event->image, // сохраняем существующее изображение
            isset($form->registration_start) && $form->registration_start !== "" ? (int)$form->registration_start : null,
            isset($form->registration_end) && $form->registration_end !== "" ? (int)$form->registration_end : null,
            isset($form->max_participants) && $form->max_participants !== "" ? (int)$form->max_participants : 0,
            $form->price,
            $form->currency,
            $form->settings ? json_decode($form->settings, true) : null,
            $form->meta_title,
            $form->meta_description,
            $form->meta_keywords,
            $form->color
        );

        $event->status = $form->status;

        $this->transaction->wrap(function () use ($event) {
            $this->eventRepository->save($event);
        });
    }

    public function subscribe(string $id, string $uid): bool
    {
        $event = $this->eventRepository->get($id);
        $user = $this->userRepository->get($uid);

        $notify_array = $event->getNotifyEmailsArray();
        if (in_array($user->email, $notify_array)) {
          return false;
        }

        $notify_array[] = $user->email;
        $this->transaction->wrap(function () use ($event, $notify_array) {
            $event->setNotifyEmailsArray($notify_array);
            $this->eventRepository->save($event);
        });

        return true;
    }

    public function unSubscribe(string $id, string $uid): bool
    {
        $event = $this->eventRepository->get($id);
        $user = $this->userRepository->get($uid);

        $notify_array = $event->getNotifyEmailsArray();
        if (!in_array($user->email, $notify_array)) {
            return false;
        }

        $notify_array = array_filter($notify_array, function($item) use ($user) {
            return $item !== $user->email;
        });

        $this->transaction->wrap(function () use ($event, $notify_array) {
            $event->setNotifyEmailsArray($notify_array);
            $this->eventRepository->save($event);
        });

        return true;
    }

    public function publish(string $id): void
    {
        $event = $this->eventRepository->get($id);
        $event->publish();
        $this->eventRepository->save($event);
    }

    public function draft(string $id): void
    {
        $event = $this->eventRepository->get($id);
        $event->draft();
        $this->eventRepository->save($event);
    }

    public function cancel(string $id): void
    {
        $event = $this->eventRepository->get($id);
        $event->cancel();
        $this->eventRepository->save($event);
    }

    public function archive(string $id): void
    {
        $event = $this->eventRepository->get($id);
        $event->archive();
        $this->eventRepository->save($event);
    }

    public function moveToPending(string $id): void
    {
        $event = $this->eventRepository->get($id);
        $event->moveToPending();
        $this->eventRepository->save($event);
    }

    public function remove(string $id): void
    {
        $event = $this->eventRepository->get($id);
        $this->removeImage($event);
        $this->eventRepository->remove($event);
    }

    public function saveImportFile(Event $event, UploadedFile $file): array
    {
        $fileName = md5('event-import-' . $event->id . '-' . time()) . '.' . $file->extension;
        $path = Yii::getAlias('@webroot/events/' . $fileName);

        // Создаем директорию если не существует
        $dir = dirname($path);
        if (!is_dir($dir)) {
            mkdir($dir, 0755, true);
        }

        if ($file->saveAs($path)) {
            $headers = $this->extractHeaders($path, 'csv');
            return [$path, $headers];
        } else {
            throw new DomainException('Не удалось сохранить изображение');
        }
    }

    /**
     * Достаем заголовки столбцов при импорте csv/xml
     * @param $filePath
     * @param $extension
     * @return array
     */
    private function extractHeaders($filePath, $extension): array
    {
        $headers = [];

        if ($extension === 'csv') {
            if (($handle = fopen($filePath, "r")) !== FALSE) {
                $headers = fgetcsv($handle, 1000, ",");
                fclose($handle);
            }
        }
//        elseif ($extension === 'xml') {
//            $xml = simplexml_load_file($filePath);
//            $firstItem = $xml->participant[0] ?? $xml->row[0] ?? null;
//            if ($firstItem) {
//                foreach ($firstItem->children() as $child) {
//                    $headers[] = $child->getName();
//                }
//            }
//        }

        return explode(';', $headers[0]);
    }

    public function setImage(Event $event, UploadedFile $image): void
    {
        $fileName = md5('event-' . $event->id . '-' . time()) . '.' . $image->extension;
        $path = Yii::getAlias('@webroot/events/' . $fileName);

        // Создаем директорию если не существует
        $dir = dirname($path);
        if (!is_dir($dir)) {
            mkdir($dir, 0755, true);
        }

        if ($image->saveAs($path)) {
            // Удаляем старое изображение если есть
            if ($event->image && file_exists(Yii::getAlias('@webroot/events/' . $event->image))) {
                unlink(Yii::getAlias('@webroot/events/' . $event->image));
            }

            $event->image = $fileName;
            $this->eventRepository->save($event);
        } else {
            throw new DomainException('Не удалось сохранить изображение');
        }
    }

    public function removeImage(Event $event): void
    {
        if ($event->image && file_exists(Yii::getAlias('@webroot/events/' . $event->image))) {
            unlink(Yii::getAlias('@webroot/events/' . $event->image));
        }

        $event->image = null;
        $this->eventRepository->save($event);
    }

    public function updateSettings(string $id, array $settings): void
    {
        $event = $this->eventRepository->get($id);
        $currentSettings = $event->settings ? json_decode($event->settings, true) : [];
        $mergedSettings = array_merge($currentSettings, $settings);

        $event->settings = json_encode($mergedSettings, JSON_UNESCAPED_UNICODE);
        $this->eventRepository->save($event);
    }

    public function setSetting($id, $key, $value): void
    {
        $event = $this->eventRepository->get($id);
        $settings = $event->settings ? json_decode($event->settings, true) : [];
        $settings[$key] = $value;
        $event->settings = json_encode($settings, JSON_UNESCAPED_UNICODE);
        $this->eventRepository->save($event);
    }

    public function getSetting($id, $key, $default)
    {
        $event = $this->eventRepository->get($id);
        $settings = $event->settings ? json_decode($event->settings, true) : [];
        return $settings[$key] ?? $default;
    }

//    public function updateRegistrationSettings(string $id, array $settings): void
//    {
//        $event = $this->eventRepository->get($id);
//
//        $event->registration_requires_approval = $settings['requires_approval'] ?? false;
//        $event->registration_notification_emails = $settings['notification_emails'] ?? null;
//        $event->registration_custom_form = $settings['custom_form'] ?? false;
//        $event->registration_success_message = $settings['success_message'] ?? null;
//
//        $this->eventRepository->save($event);
//    }

//    public function getRegistrationStats(string $id): array
//    {
//        return $this->eventRepository->getRegistrationStats($id);
//    }
//
//    public function exportRegistrations(string $id, string $format = 'csv'): string
//    {
//        $event = $this->eventRepository->getWithRegistrations($id);
//        $registrations = $event->registrations;
//
//        // Логика экспорта в CSV/Excel
//        // Возвращаем путь к файлу
//    }
}
