<?php

namespace app\core\services\event;

use app\core\entities\Event;
use app\core\entities\Member;
use app\core\forms\event\MemberCreateForm;
use app\core\forms\event\MemberForm;
use app\core\helpers\MailHelper;
use app\core\helpers\SettingsHelper;
use app\core\repositories\event\EventRepository;
use app\core\repositories\event\MemberRepository;
use app\core\services\TransactionManager;
use DomainException;
use Yii;
use yii\web\UploadedFile;

class MemberService
{
    private MemberRepository $memberRepository;
    private EventRepository $eventRepository;
    private TransactionManager $transaction;
    private EventService $eventService;

    public function __construct(
        MemberRepository $memberRepository,
        EventRepository $eventRepository,
        TransactionManager $transaction,
        EventService $eventService
    ) {
        $this->memberRepository = $memberRepository;
        $this->eventRepository = $eventRepository;
        $this->transaction = $transaction;
        $this->eventService = $eventService;
    }

    /**
     * Регистрация участника (публичная)
     */
    public function register(
        string $eventId,
        string $firstName,
        string $lastName,
        ?string $email = null,
        ?string $phone = null,
        ?string $middleName = null,
        ?string $company = null,
        ?string $position = null,
        ?string $city = null,
        ?string $country = null,
        ?array $formData = null,
        bool $notify = false
    ): Member {
        $event = $this->eventRepository->get($eventId);

        // Проверка возможности регистрации
        $this->validateRegistration($event);

        // Проверка дубликата по email (если email указан)
        if ($email && $this->memberRepository->existsForEventAndEmail($eventId, $email)) {
            throw new DomainException('Участник с таким email уже зарегистрирован на это мероприятие');
        }

        $member = Member::create(
            $eventId,
            $firstName,
            $lastName,
            $email,
            $phone,
            $middleName,
            $company,
            $position,
            $city,
            $country,
            $formData,
            Yii::$app->request->userIP,
            Yii::$app->request->userAgent,
            false // не админская регистрация
        );

        // Автоматическое подтверждение если включено
        if ($event->auto_confirm) {
            $member->confirm();
        }

        $this->transaction->wrap(function () use ($member, $event, $notify) {
            $this->memberRepository->save($member);

            // Отправка уведомлений
            if ($notify) {
                $this->sendNotifications($member, $event);
            }
        });

        return $member;
    }

    /**
     * Регистрация участника администратором
     */
    public function registerByAdmin(
        string $eventId,
        string $firstName,
        string $lastName,
        ?string $email = null,
        ?string $phone = null,
        ?string $middleName = null,
        ?string $company = null,
        ?string $position = null,
        ?string $city = null,
        ?string $country = null,
        ?string $notes = null
    ): Member {
        $member = Member::create(
            $eventId,
            $firstName,
            $lastName,
            $email,
            $phone,
            $middleName,
            $company,
            $position,
            $city,
            $country,
            null, // form_data для админской регистрации не используется
            null, // IP
            null, // User Agent
            true // админская регистрация
        );

        if ($notes) {
            $member->notes = $notes;
        }

        $this->transaction->wrap(function () use ($member) {
            $this->memberRepository->save($member);
        });

        return $member;
    }

    /**
     * Редактирование участника
     */
    public function edit(string $id, MemberForm $form): void
    {
        $member = $this->memberRepository->get($id);

        $member->edit(
            $form->first_name,
            $form->last_name,
            $form->email,
            $form->phone,
            $form->middle_name,
            $form->company,
            $form->position,
            $form->city,
            $form->country,
            $form->bio,
            $form->documents
        );

        $this->transaction->wrap(function () use ($member) {
            $this->memberRepository->save($member);
        });
    }

    /**
     * Подтверждение регистрации
     */
    public function confirm(string $id): void
    {
        $member = $this->memberRepository->get($id);
        $member->confirm();
        $this->memberRepository->save($member);
    }

    /**
     * Отмена регистрации
     */
    public function cancel(string $id): void
    {
        $member = $this->memberRepository->get($id);
        $member->cancel();
        $this->memberRepository->save($member);
    }

    /**
     * Отметка присутствия
     */
    public function markAttended(string $id): void
    {
        $member = $this->memberRepository->get($id);
        $member->markAttended();
        $this->memberRepository->save($member);
    }

    public function unmarkAttended(string $id): void
    {
        $member = $this->memberRepository->get($id);
        $member->unmarkAttended();
        $this->memberRepository->save($member);
    }

    /**
     * Check-in по коду
     */
    public function checkin(string $id): void
    {
        $member = $this->memberRepository->get($id);
        $this->markAttended($member->id);
    }

    /**
     * Удаление участника
     */
    public function remove(string $id): void
    {
        $member = $this->memberRepository->get($id);
        $this->deletePhotoFile($member);
        $this->memberRepository->remove($member);
    }

    /**
     * Загрузка фото
     */
    public function setPhoto(string $id, UploadedFile $file): void
    {
        $member = $this->memberRepository->get($id);

        $fileName = md5('member-' . $member->id . '-' . time()) . '.' . $file->extension;
        $path = Yii::getAlias('@webroot/uploads/members/' . $fileName);

        $dir = dirname($path);
        if (!is_dir($dir)) {
            mkdir($dir, 0755, true);
        }

        if ($file->saveAs($path)) {
            $this->deletePhotoFile($member);
            $member->setPhoto($fileName);
            $this->memberRepository->save($member);
        } else {
            throw new DomainException('Не удалось сохранить фото');
        }
    }

    /**
     * Удаление фото
     */
    public function removePhoto(string $id): void
    {
        $member = $this->memberRepository->get($id);
        $this->deletePhotoFile($member);
        $member->setPhoto(null);
        $this->memberRepository->save($member);
    }

    /**
     * Добавление документа
     */
    public function addDocument(string $id, array $document): void
    {
        $member = $this->memberRepository->get($id);
        $member->addDocument($document);
        $this->memberRepository->save($member);
    }

    /**
     * Получение статистики
     */
    public function getStats(string $eventId): array
    {
        return $this->memberRepository->getStats($eventId);
    }

    /**
     * Экспорт участников
     */
    public function export(string $eventId, string $format = 'csv'): string
    {
        $event = $this->eventRepository->get($eventId);
        $members = $event->members;

        // Логика экспорта в CSV/Excel
        // Возвращаем путь к файлу
        return $this->exportToCsv($event, $members);
    }

    /**
     * Валидация возможности регистрации
     */
    private function validateRegistration($event): void
    {
        if (!$event->isRegistrationOpen()) {
            throw new DomainException('Регистрация на это мероприятие закрыта');
        }

        if (!$this->eventRepository->hasAvailableSeats($event->id)) {
            throw new DomainException('На мероприятии нет свободных мест');
        }
    }

    /**
     * Удаление файла фото
     */
    private function deletePhotoFile(Member $member): void
    {
        if ($member->photo && file_exists(Yii::getAlias('@webroot/uploads/members/' . $member->photo))) {
            unlink(Yii::getAlias('@webroot/uploads/members/' . $member->photo));
        }
    }

    /**
     * Отправка уведомлений
     */
    private function sendNotifications(Member $member, Event $event): void
    {
        // Отправка email участнику
        if ($member->email) {
            $this->sendConfirmationEmail($member, $event);
        }

        // Уведомление организаторам
        $notificationEmails = $event->getNotifyEmailsArray();
        if (!empty($notificationEmails)) {
            $this->sendOrganizerNotification($member, $event, $notificationEmails);
        }
    }

    /**
     * Отправка email подтверждения
     */
    private function sendConfirmationEmail(Member $member, $event): void
    {
        // Отправка уведомления участнику
//        if ($form->email && filter_var($form->email, FILTER_VALIDATE_EMAIL)) {
//            MailHelper::send()->compose(
//                ['html' => '@app/mail/event/event-member-notification-html', 'text' => '@app/mail/event/event-member-notification-text'],
//                [
//                    'event' => $event,
//                    'member' => $member,
//                    'notify' => $this->eventService->getSetting($event->id, 'tpl.mail.notify', ''),
//                ]
//            )
//                ->setTo($form->email)
//                ->setFrom([$senderEmail => $senderName])
//                ->setSubject('Регистрация подтверждена: ' . $event->title)
//                ->send();
//
//        }
        // Отправка уведомлений подписанным администраторам
//        $emails = $event->notify_emails;
//        $emails = $emails ? json_decode($emails, true) : ['error-202@yandex.ru'];
//        foreach ($emails as $email) {
//            MailHelper::send()->compose(
//                ['html' => '@app/mail/event/event-registration-notification-html', 'text' => '@app/mail/event/event-registration-notification-text'],
//                [
//                    'event' => $event,
//                    'member' => $member,
//                ]
//            )
//                ->setTo($email)
//                ->setFrom([$senderEmail => $senderName])
//                ->setSubject('Новая регистрация: ' . $event->title)
//                ->send();
//        }

        $senderEmail = SettingsHelper::getSetting('from_email', Yii::$app->params['senderEmail']);
        $senderName = SettingsHelper::getSetting('from_name', Yii::$app->params['senderName']);

        MailHelper::send()->compose(
            //['html' => '@app/mail/event/event-registration-html', 'text' => '@app/mail/event/event-registration-text'],
            ['html' => '@app/mail/event/event-member-notification-html', 'text' => '@app/mail/event/event-member-notification-text'],
            [
                'member' => $member,
                'event' => $event,
                'notify' => $this->eventService->getSetting($event->id, 'tpl.mail.notify', ''),
            ]
        )
            ->setTo($member->email)
            ->setFrom([$senderEmail => $senderName])
            ->setSubject('Регистрация подтверждена: ' . $event->title)
            ->send();
    }

    /**
     * Отправка уведомления организаторам
     */
    private function sendOrganizerNotification(Member $member, $event, array $emails): void
    {
        $senderEmail = SettingsHelper::getSetting('from_email', Yii::$app->params['senderEmail']);
        $senderName = SettingsHelper::getSetting('from_name', Yii::$app->params['senderName']);

        Yii::$app->mailer->compose(
            ['html' => '@app/mail/event/event-registration-notification-html', 'text' => '@app/mail/event/event-registration-notification-text'],
            [
            'member' => $member,
            'event' => $event,
        ])
            ->setTo($emails)
            ->setFrom([$senderEmail => $senderName])
            ->setSubject('Новая регистрация на мероприятие: ' . $event->title)
            ->send();
    }

    /**
     * Экспорт в CSV
     */
    private function exportToCsv($event, $members): string
    {
        $fileName = 'export-' . $event->slug . '-' . date('Y-m-d') . '.csv';
        $filePath = Yii::getAlias('@webroot/exports/' . $fileName);

        $dir = dirname($filePath);
        if (!is_dir($dir)) {
            mkdir($dir, 0755, true);
        }

        $handle = fopen($filePath, 'w');

        // Добавляем BOM для корректного отображения в Excel
        fwrite($handle, chr(0xEF) . chr(0xBB) . chr(0xBF));

        // Заголовки с кодировкой UTF-8
        fputcsv($handle, [
            'Фамилия', 'Имя', 'Email', 'Телефон',
            'Компания', 'Должность', 'Статус', 'Дата регистрации'
        ], ';', '"', '\\');

        foreach ($members as $member) {
            fputcsv($handle, [
                $member->last_name,
                $member->first_name,
                $member->email ?? '',
                $member->phone ?? '',
                $member->company ?? '',
                $member->position ?? '',
                Member::getStatuses()[$member->status],
                date('d.m.Y H:i', $member->registered_at)
            ], ';', '"', '\\');
        }

        fclose($handle);

        return $filePath;
    }
}
