<?php
namespace app\core\services\update;

use RecursiveDirectoryIterator;
use RecursiveIteratorIterator;
use Yii;
use ZipArchive;

class ArchiveBuilder
{
    public function build($version, $changedFiles, $deletedFiles, $migrations, $manifest): string
    {
        $tempDir = $this->createTempDirectory();

        try {
            // 1. Создаем структуру
            $this->createArchiveStructure($tempDir);

            // 2. Записываем манифест
            file_put_contents(
                $tempDir . '/manifest.json',
                json_encode($manifest, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES)
            );

            // 3. Копируем измененные файлы
            //$this->copyChangedFiles($changedFiles, $tempDir);
            echo "  Copying changed files...\n";
            $copiedCount = $this->copyChangedFiles($changedFiles, $tempDir);
            echo "  Copied files: {$copiedCount}\n";

            // 4. Копируем миграции
            $this->copyMigrations($migrations, $tempDir);

            echo "  Creating version file...\n";
            $this->copyVersionFile($tempDir, $version);

            // 5. Создаем delete.list
            if (!empty($deletedFiles)) {
                file_put_contents(
                    $tempDir . '/delete.list',
                    implode("\n", $deletedFiles)
                );
            }

            // 6. Создаем ZIP
            $archivePath = $this->createZipArchive($tempDir, $version);

            // 7. Обновляем хэш в манифесте
            $this->updateManifestHash($archivePath, $manifest);

            return $archivePath;

        } finally {
            // Очистка временной директории
            $this->removeDirectory($tempDir);
        }
    }

    private function createTempDirectory(): string
    {
        $tempDir = sys_get_temp_dir() . '/update_gen_' . uniqid();
        mkdir($tempDir, 0755, true);
        return $tempDir;
    }

    private function createArchiveStructure($tempDir): void
    {
        mkdir($tempDir . '/files', 0755, true);
        mkdir($tempDir . '/migrations', 0755, true);
    }

    private function copyVersionFile($tempDir, $version): void
    {
        $destination = $tempDir . '/files/config/version.php';

        // Создаем директорию config если нужно
        $destDir = dirname($destination);
        if (!is_dir($destDir)) {
            mkdir($destDir, 0755, true);
        }

        // Создаем файл version.php с целевой версией
        $content = "<?php\n// Auto-generated by update system\nreturn '{$version}';\n";
        file_put_contents($destination, $content);

        echo "    Created version.php with version: {$version}\n";
    }

    private function copyChangedFiles($files, $tempDir): int
    {
        $basePath = Yii::getAlias('@app');
        $copied = 0;

        foreach ($files as $file) {
            $source = $basePath . '/' . $file;
            $destination = $tempDir . '/files/' . $file;

            if ($file === 'config/version.php') {
                echo "    Skipping config/version.php (will be created separately)\n";
                continue;
            }

            if (!file_exists($source)) {
                echo "    WARNING: Source file not found: {$file}\n";
                continue;
            }

            // Создаем директории если нужно
            $destDir = dirname($destination);
            if (!is_dir($destDir)) {
                mkdir($destDir, 0755, true);
            }

            if (copy($source, $destination)) {
                $copied++;
                if ($copied <= 5) { // Показываем первые 5 файлов
                    echo "    Copied: {$file}\n";
                }
            } else {
                echo "    ERROR: Failed to copy: {$file}\n";
            }

            //if (file_exists($source)) {
            //    copy($source, $destination);
            //}
        }
        echo "    Total copied: {$copied} files\n";
        return $copied;
    }

    private function copyMigrations($migrations, $tempDir): void
    {
        $migrationDir = Yii::getAlias('@app/migrations');

        foreach ($migrations as $migration) {
            $source = $migrationDir . '/' . $migration;
            $destination = $tempDir . '/migrations/' . $migration;

            if (file_exists($source)) {
                copy($source, $destination);
            }
        }
    }

    private function createZipArchive($tempDir, $version): string
    {
        $outputDir = Yii::getAlias('@app/runtime/updates');

        if (!is_dir($outputDir)) {
            mkdir($outputDir, 0755, true);
        }

        $archiveName = $version . '-update.zip';
        $archivePath = $outputDir . '/' . $archiveName;

        $zip = new ZipArchive();

        if ($zip->open($archivePath, ZipArchive::CREATE | ZipArchive::OVERWRITE) === true) {
            $this->addFolderToZip($zip, $tempDir, '');
            $zip->close();
        } else {
            throw new \Exception("Failed to create ZIP archive");
        }

        return $archivePath;
    }

    private function addFolderToZip($zip, $folder, $parentFolder = ''): void
    {
        $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($folder, RecursiveDirectoryIterator::SKIP_DOTS),
            RecursiveIteratorIterator::SELF_FIRST
        );

        foreach ($iterator as $file) {
            if ($file->isFile()) {
                $filePath = $file->getPathname();
                $relativePath = $parentFolder . str_replace($folder . '/', '', $filePath);
                $zip->addFile($filePath, $relativePath);
            }
        }
    }

    private function updateManifestHash($archivePath, $manifest): void
    {
        $hash = hash_file('sha256', $archivePath);
        $manifest['hash']['value'] = $hash;

        // Обновляем манифест в ZIP
        $zip = new ZipArchive();
        if ($zip->open($archivePath) === true) {
            $zip->deleteName('manifest.json');
            $zip->addFromString(
                'manifest.json',
                json_encode($manifest, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES)
            );
            $zip->close();
        }
    }

    private function removeDirectory($dir): void
    {
        if (!is_dir($dir)) {
            return;
        }

        $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($dir, RecursiveDirectoryIterator::SKIP_DOTS),
            RecursiveIteratorIterator::CHILD_FIRST
        );

        foreach ($iterator as $file) {
            if ($file->isDir()) {
                rmdir($file->getPathname());
            } else {
                unlink($file->getPathname());
            }
        }

        rmdir($dir);
    }
}