<?php
namespace app\core\services\update;

use RecursiveDirectoryIterator;
use RecursiveIteratorIterator;
use SplFileInfo;
use Yii;
use ZipArchive;

class DistributionCreator
{
    private $version;
    private array $excludeConfig = [
        'excludeDirs' => ['@app/vendor']
    ];
    private $fileCount = 0;

    public function __construct(array $config = [])
    {
        foreach ($config as $key => $value) {
            if (property_exists($this, $key)) {
                $this->$key = $value;
            }
        }
    }

    /**
     * Создает полный дистрибутив приложения
     */
    public function create(): string
    {
        echo "Создание дистрибутива для версии: {$this->version}\n";

        echo "Сканирование файлов проекта... ";
        $scanner = $this->createFileScanner();
        $allFiles = $this->scanAllFiles($scanner);
        echo "готово\n";

        $tempDir = $this->createTempDirectory();
        echo "Создана временная папка: $tempDir\n";

        try {
            echo "Копирование файлов во временную папку... ";
            $this->copyFilesToTemp($allFiles, $tempDir);
            echo "готово\n";

            // Устанавливаем зависимости во временной папке
            $this->installComposerDependencies($tempDir);

            // Создаем манифест
            $manifest = $this->createManifest($allFiles);
            file_put_contents(
                $tempDir . '/manifest.json',
                json_encode($manifest, JSON_PRETTY_PRINT)
            );
            echo "Создан manifest.json\n";

            $archivePath = $this->createDistributionZip($tempDir);

            echo "\nДистрибутив создан: " . basename($archivePath) . "\n";
            echo "Файлов: {$this->fileCount}\n";
            echo "Размер: " . filesize($archivePath) . " bytes\n";

            return $archivePath;

        } finally {
            $this->removeDirectory($tempDir);
        }
    }

    /**
     * Получает информацию о дистрибутиве
     */
    public function getInfo(): array
    {
        $archivePath = $this->getArchivePath();

        if (!file_exists($archivePath)) {
            throw new \Exception("Distribution not found: {$this->version}");
        }

        return [
            'version' => $this->version,
            'archive_path' => $archivePath,
            'size' => filesize($archivePath),
            'created_at' => date('Y-m-d H:i:s', filemtime($archivePath)),
            'file_count' => $this->fileCount,
        ];
    }

    /**
     * Создает FileScanner с исключениями
     */
    private function createFileScanner(): FileScanner
    {
        $scanner = new FileScanner();

        $exclusionManager = new ExclusionManager();
        $configExclusions = $exclusionManager->getExclusions();

        $scanner->addExcludeDirectories($configExclusions['directories'] ?? []);
        $scanner->addExcludeFiles($configExclusions['files'] ?? []);
        $scanner->addExcludePatterns($configExclusions['patterns'] ?? []);

        if (!empty($this->excludeConfig['excludeDirs'])) {
            $scanner->addExcludeDirectories($this->excludeConfig['excludeDirs']);
        }

        if (!empty($this->excludeConfig['excludeFiles'])) {
            $scanner->addExcludeFiles($this->excludeConfig['excludeFiles']);
        }

        if (!empty($this->excludeConfig['excludePatterns'])) {
            $scanner->addExcludePatterns($this->excludeConfig['excludePatterns']);
        }

        return $scanner;
    }

    /**
     * Сканирует все файлы
     */
    private function scanAllFiles(FileScanner $scanner): array
    {
        $basePath = Yii::getAlias('@app');
        $files = [];

        $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($basePath, RecursiveDirectoryIterator::SKIP_DOTS),
            RecursiveIteratorIterator::SELF_FIRST
        );

        foreach ($iterator as $item) {
            if ($scanner->shouldExclude($item)) {
                continue;
            }

            if ($item->isFile()) {
                $relativePath = str_replace($basePath . DIRECTORY_SEPARATOR, '', $item->getPathname());
                $files[] = [
                    'path' => $relativePath,
                    'full_path' => $item->getPathname(),
                    'size' => $item->getSize(),
                ];
            }
        }

        echo "Total files found: " . count($files) . "\n";
        return $files;
    }

    /**
     * Копирует файлы во временную директорию
     */
    private function copyFilesToTemp(array $files, string $tempDir): void
    {
        echo "Копирование файлов...\n";

        foreach ($files as $fileInfo) {
            $source = $fileInfo['full_path'];
            $destination = $tempDir . '/' . $fileInfo['path'];

            $destDir = dirname($destination);
            if (!is_dir($destDir)) {
                mkdir($destDir, 0755, true);
            }

            if (copy($source, $destination)) {
                $this->fileCount++;

                if ($this->fileCount % 500 === 0) {
                    echo "Скопировано файлов: {$this->fileCount}\n";
                }
            }
        }
    }

    /**
     * Создает манифест
     */
    private function createManifest(array $files): array
    {
        $totalSize = 0;
        foreach ($files as $file) {
            $totalSize += $file['size'];
        }

        return [
            'version' => $this->version,
            'created_at' => date('c'),
            'total_files' => count($files),
            'total_size' => $totalSize,
            'exclude_config' => $this->excludeConfig,
        ];
    }

    /**
     * Создает ZIP архив
     */
    private function createDistributionZip(string $tempDir): string
    {
        $outputDir = Yii::getAlias('@app/runtime/distributions');

        if (!is_dir($outputDir)) {
            mkdir($outputDir, 0755, true);
        }

        $archiveName = $this->version . '-distribution.zip';
        $archivePath = $outputDir . '/' . $archiveName;

        if (is_file($archivePath)) {
            unlink($archivePath);
        }

        $zip = new ZipArchive();

        if ($zip->open($archivePath, ZipArchive::CREATE | ZipArchive::OVERWRITE) === true) {
            $this->addFolderToZip($zip, $tempDir, '');
            $zip->close();
        } else {
            throw new \Exception("Failed to create ZIP archive");
        }

        return $archivePath;
    }

    /**
     * Добавляет папку в ZIP
     */
    private function addFolderToZip($zip, $folder, $parentFolder = ''): void
    {
        $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($folder, RecursiveDirectoryIterator::SKIP_DOTS),
            RecursiveIteratorIterator::SELF_FIRST
        );

        foreach ($iterator as $file) {
            if ($file->isFile()) {
                $filePath = $file->getPathname();
                $relativePath = $parentFolder . str_replace($folder . '/', '', $filePath);
                $zip->addFile($filePath, $relativePath);
            }
        }
    }

    /**
     * Получает путь к архиву
     */
    private function getArchivePath(): string
    {
        $outputDir = Yii::getAlias('@app/runtime/distributions');
        $archiveName = $this->version . '-distribution.zip';
        return $outputDir . '/' . $archiveName;
    }

    /**
     * Получает путь к манифесту
     */
    private function getManifestPath(): string
    {
        $outputDir = Yii::getAlias('@app/runtime/distributions');
        $manifestName = $this->version . '-manifest.json';
        return $outputDir . '/' . $manifestName;
    }

    /**
     * Создает временную директорию
     */
    private function createTempDirectory(): string
    {
        $tempDir = sys_get_temp_dir() . '/dist_' . uniqid();
        mkdir($tempDir, 0755, true);
        return $tempDir;
    }

    /**
     * Удаляет директорию рекурсивно
     */
    private function removeDirectory($dir): void
    {
        if (!is_dir($dir)) {
            return;
        }

        $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($dir, RecursiveDirectoryIterator::SKIP_DOTS),
            RecursiveIteratorIterator::CHILD_FIRST
        );

        foreach ($iterator as $file) {
            if ($file->isDir()) {
                rmdir($file->getPathname());
            } else {
                unlink($file->getPathname());
            }
        }

        rmdir($dir);
    }

    /**
     * Вывод в консоль
     */
    private function stdout($message): void
    {
        echo $message;
    }

    /**
     * Устанавливает production зависимости Composer во временной папке
     */
    private function installComposerDependencies(string $tempDir): void
    {
        echo "Установка Composer зависимостей... ";

        $oldCwd = getcwd();
        chdir($tempDir);

        try {
            $command = 'composer install --no-dev --optimize-autoloader --classmap-authoritative --no-scripts --no-interaction 2>&1';
            exec($command, $output, $returnCode);

            if ($returnCode !== 0) {
                echo "ОШИБКА!\n";
                throw new \Exception("Composer failed: " . implode("\n", array_slice($output, -10)));
            }

            echo "готово\n";
        } finally {
            chdir($oldCwd);
        }
    }
}
