<?php
namespace app\core\services\update;

use Exception;
use RecursiveDirectoryIterator;
use RecursiveIteratorIterator;
use SplFileInfo;
use Yii;

class FileScanner
{
    private array $exclusions = [
        'directories' => [],
        'files' => [],
        'patterns' => [],
    ];

    // Сохраняет состояние файлов текущей версии
    public function saveState($version): void
    {
        $state = [
            'version' => $version,
            'timestamp' => time(),
            'files' => $this->scanCurrentFiles(),
        ];

        $stateFile = $this->getStateFilePath($version);
        file_put_contents($stateFile, json_encode($state, JSON_PRETTY_PRINT));

        echo "State saved to: {$stateFile}\n";
        echo "Files indexed: " . count($state['files']) . "\n";
    }

    // Получает измененные файлы между версиями
    public function getChangedFiles($fromVersion, $toVersion): array
    {
        $oldState = $this->loadState($fromVersion);
        $currentFiles = $this->scanCurrentFiles();

        $changed = [];
        foreach ($currentFiles as $file => $hash) {
            if (!isset($oldState['files'][$file]) || $oldState['files'][$file] !== $hash) {
                $changed[] = $file;
            }
        }

        return $changed;
    }

    // Получает удаленные файлы
    public function getDeletedFiles($fromVersion, $toVersion): array
    {
        $oldState = $this->loadState($fromVersion);
        $currentFiles = $this->scanCurrentFiles();

        $deleted = [];
        foreach ($oldState['files'] as $file => $hash) {
            if (!isset($currentFiles[$file])) {
                // Проверяем, что файл действительно существует на диске
                $fullPath = Yii::getAlias('@app') . '/' . $file;
                if (file_exists($fullPath)) {
                    $deleted[] = $file;
                }
            }
        }

        return $deleted;
    }

    // Сканирует текущие файлы с учетом исключений
    private function scanCurrentFiles(): array
    {
        $basePath = Yii::getAlias('@app');
        $files = [];

        $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($basePath, RecursiveDirectoryIterator::SKIP_DOTS),
            RecursiveIteratorIterator::SELF_FIRST
        );

        foreach ($iterator as $file) {
            if ($this->shouldExclude($file)) {
                continue;
            }

            if ($file->isFile()) {
                $relativePath = str_replace($basePath . DIRECTORY_SEPARATOR, '', $file->getPathname());
                $files[$relativePath] = hash_file('sha256', $file->getPathname());
            }
        }

        return $files;
    }

    // Проверяет, нужно ли исключить файл/папку
    public function shouldExclude(SplFileInfo $file): bool
    {
        $path = $file->getPathname();
        $normalizedPath = str_replace('\\', '/', $path);

        $relativePath = str_replace(Yii::getAlias('@app') . '/', '', $path);

        // Проверка исключенных папок
        foreach ($this->exclusions['directories'] as $idx => $dir) {
            $dirPath = Yii::getAlias($dir);
            $normalizedDirPath = str_replace('\\', '/', $dirPath);

            if (str_starts_with($normalizedPath, $normalizedDirPath . '/')) {
                return true;
            }
        }

        // Проверка конкретных файлов
        if ($file->isFile()) {
            foreach ($this->exclusions['files'] as $excludedFile) {
                $excludedPath = Yii::getAlias($excludedFile);
                $normalizedExcludedPath = str_replace('\\', '/', $excludedPath);

                if ($normalizedPath === $normalizedExcludedPath) {
                    return true;
                }
            }
        }

        // Проверка regex паттернов
        if ($file->isFile()) {
            $relativePath = str_replace(
                str_replace('\\', '/', Yii::getAlias('@app')) . '/',
                '',
                $normalizedPath
            );

            foreach ($this->exclusions['patterns'] as $pattern) {
                if (preg_match($pattern, $relativePath)) {
                    return true;
                }
            }
        }

        return false;
    }

    // Загружает сохраненное состояние версии
    private function loadState($version)
    {
        $stateFile = $this->getStateFilePath($version);

        if (!file_exists($stateFile)) {
            throw new Exception("State file not found for version: {$version}");
        }

        $data = json_decode(file_get_contents($stateFile), true);

        if (json_last_error() !== JSON_ERROR_NONE) {
            throw new Exception("Invalid state file for version: {$version}");
        }

        return $data;
    }

    // Путь к файлу состояния
    private function getStateFilePath($version): string
    {
        $stateDir = Yii::getAlias('@app/runtime/update_states');

        if (!is_dir($stateDir)) {
            mkdir($stateDir, 0755, true);
        }

        return $stateDir . '/' . $version . '.json';
    }

    // Методы для добавления исключений
    public function addExcludeDirectories(array $dirs): void
    {
        $this->exclusions['directories'] = array_merge(
            $this->exclusions['directories'],
            $dirs
        );
    }

    public function addExcludeFiles(array $files): void
    {
        $this->exclusions['files'] = array_merge(
            $this->exclusions['files'],
            $files
        );
    }

    public function addExcludePatterns(array $patterns): void
    {
        $this->exclusions['patterns'] = array_merge(
            $this->exclusions['patterns'],
            $patterns
        );
    }
}