<?php

namespace app\core\services\update;

use Yii;

class UpdateGenerator
{
    private $fromVersion;
    private $toVersion;
    private array $excludeConfig = [];

    public function __construct(array $config = [])
    {
        foreach ($config as $key => $value) {
            if (property_exists($this, $key)) {
                $this->$key = $value;
            }
        }
    }

    /**
     * Основной метод генерации архива обновления
     */
    public function generate(): string
    {
        echo "Step 1: Scanning files...\n";
        $fileScanner = $this->createFileScanner();
        $changedFiles = $fileScanner->getChangedFiles($this->fromVersion, $this->toVersion);
        $deletedFiles = $fileScanner->getDeletedFiles($this->fromVersion, $this->toVersion);

        echo "Step 2: Collecting migrations...\n";
        $migrationCollector = new MigrationCollector();
        $migrations = $migrationCollector->getMigrationsBetween(
            $this->fromVersion,
            $this->toVersion
        );

        echo "Step 3: Creating temporary version file...\n";
        $this->createNewVersionFile();

        echo "Step 4: Creating manifest...\n";
        $manifest = $this->createManifest($changedFiles, $deletedFiles, $migrations);

        echo "Step 5: Building archive...\n";
        $archiveBuilder = new ArchiveBuilder();
        $archivePath = $archiveBuilder->build(
            $this->toVersion,
            $changedFiles,
            $deletedFiles,
            $migrations,
            $manifest
        );

        echo "Step 6: Restoring original version file...\n";
        $this->restoreOldVersionFile();

        return $archivePath;
    }

    private function createNewVersionFile(): void
    {
        $configFile = Yii::getAlias('@app/config/version.php');
        $backupFile = $configFile . '.backup';

        // Делаем backup текущего файла версии
        if (file_exists($configFile)) {
            copy($configFile, $backupFile);
            echo "  Created backup: " . basename($backupFile) . "\n";
        }

        // Создаем временный файл с новой версией
        $content = "<?php\nreturn '{$this->toVersion}';\n";
        file_put_contents($configFile, $content);

        echo "  Created temporary version file: {$this->toVersion}\n";
    }

    private function restoreOldVersionFile(): void
    {
        $configFile = Yii::getAlias('@app/config/version.php');
        $backupFile = $configFile . '.backup';

        if (file_exists($backupFile)) {
            copy($backupFile, $configFile);
            unlink($backupFile);
            echo "  Restored original version file\n";
        }
    }

    /**
     * Получает diff между версиями
     */
    public function getDiff()
    {
        $fileScanner = $this->createFileScanner();

        return [
            'changed' => $fileScanner->getChangedFiles($this->fromVersion, $this->toVersion),
            'deleted' => $fileScanner->getDeletedFiles($this->fromVersion, $this->toVersion),
            'migrations' => (new MigrationCollector())
                ->getMigrationsBetween($this->fromVersion, $this->toVersion),
        ];
    }

    /**
     * Создает манифест обновления
     */
    private function createManifest($changedFiles, $deletedFiles, $migrations)
    {
        return [
            'version' => $this->toVersion,
            'previous_version' => $this->fromVersion,
            'generated_at' => date('c'),
            'description' => "Update from {$this->fromVersion} to {$this->toVersion}",
            'statistics' => [
                'changed_files' => count($changedFiles),
                'deleted_files' => count($deletedFiles),
                'migrations' => count($migrations),
            ],
            'requirements' => [
                'min_version' => $this->fromVersion,
            ],
            'hash' => [
                'algorithm' => 'sha256',
                // Здесь будет реальный хэш после создания архива
            ],
        ];
    }

    /**
     * Создает FileScanner с учетом исключений
     */
    private function createFileScanner(): FileScanner
    {
        $scanner = new FileScanner();

        // Добавляем исключения из конфига
        $exclusionManager = new ExclusionManager();
        $configExclusions = $exclusionManager->getExclusions();

        $scanner->addExcludeDirectories($configExclusions['directories'] ?? []);
        $scanner->addExcludeFiles($configExclusions['files'] ?? []);
        $scanner->addExcludePatterns($configExclusions['patterns'] ?? []);

        // Динамические исключения из параметров
        if (!empty($this->excludeConfig['excludeDirs'])) {
            $scanner->addExcludeDirectories($this->excludeConfig['excludeDirs']);
        }

        if (!empty($this->excludeConfig['excludeFiles'])) {
            $scanner->addExcludeFiles($this->excludeConfig['excludeFiles']);
        }

        if (!empty($this->excludeConfig['excludePatterns'])) {
            $scanner->addExcludePatterns($this->excludeConfig['excludePatterns']);
        }

        return $scanner;
    }
}