<?php
namespace app\core\services\user;

use app\core\entities\User;
use app\core\forms\user\UserCreateForm;
use app\core\forms\user\UserForm;
use app\core\repositories\user\UserRepository;
use app\core\services\RoleManager;
use app\core\services\TransactionManager;
use DomainException;
use Ramsey\Uuid\Uuid;
use Yii;
use yii\base\Exception;
use yii\db\StaleObjectException;

class UserService
{
    private UserRepository $user_repository;
    private RoleManager $roles;
    private TransactionManager $transaction;

    /**
     * UserManageService constructor.
     *
     * @param UserRepository $user_repository
     * @param RoleManager $roles
     * @param TransactionManager $transaction
     */
    public function __construct(
        UserRepository $user_repository,
        RoleManager $roles,
        TransactionManager $transaction
    )
    {
        $this->user_repository  = $user_repository;
        $this->roles       = $roles;
        $this->transaction = $transaction;
    }

    /**
     * @throws \yii\db\Exception
     * @throws Exception
     */
    public function create(UserCreateForm $form, $role = null): User
    {
        $user = User::create(
            $form->username,
            $form->email,
            $form->password
        );
        $this->transaction->wrap(function () use ($user, $form, $role) {
            $this->user_repository->save($user);
            $this->roles->assign($user->id, $role ?? 'user');
        });
        return $user;
    }

    /**
     * @throws Exception
     * @throws \yii\db\Exception
     */
    public function edit($id, UserForm $form): void
    {
        $user = $this->user_repository->get($id);
        $user->edit(
            $form->username,
            $form->email,
            $form->password
        );

        // Сам себе статус не может изменить
        if ($user->id != Yii::$app->user->id) {
            $user->status = $form->status;
        }

        $this->transaction->wrap(function () use ($user, $form) {
            $this->user_repository->save($user);
            // Сам себе роль не может изменить
            if ($user->id != Yii::$app->user->id) {
                $this->roles->assign($user->id, $form->role);
            }
        });
    }

    /**
     * @throws \Exception
     */
    public function assignRole($id, $role): void
    {
        $user = $this->user_repository->get($id);
        $this->roles->assign($user->id, $role);
    }

    /**
     * @throws StaleObjectException|Throwable
     */
    public function remove($id): void
    {
        $user = $this->user_repository->get($id);
        $this->user_repository->remove($user);
    }

    public function markDeleted($id): void
    {
        $user = $this->user_repository->get($id);
        $user->status = User::STATUS_DELETED;
        $this->user_repository->save($user);
    }

    public function repair($id): void
    {
        $user = $this->user_repository->get($id);
        $user->status = User::STATUS_ACTIVE;
        $this->user_repository->save($user);
    }

    public function setSetting($id, $key, $value): void
    {
        $user = $this->user_repository->get($id);
        $settings = $user->settings ? json_decode($user->settings, true) : [];
        $settings[$key] = $value;
        $user->settings = json_encode($settings, JSON_UNESCAPED_UNICODE);
        $this->user_repository->save($user);
    }

    public function loginToken(User $user, $time = 0): void
    {
        $user->generateLoginToken($time);
        $this->user_repository->save($user);
    }

    public function authByToken(string $token): User
    {
        $user = $this->user_repository->findByLoginToken($token);
        if (!$user) {
            throw new DomainException(Yii::t('user', 'Token not found.'));
        }
        $user->status = User::STATUS_ACTIVE;
        $user->clearLoginToken();
        $this->user_repository->save($user);
        return $user;
    }

    public function setAvatar(User $user, $user_pic): array
    {
        $fileName = md5('avatar-' . $user_pic->baseName . time()) . '.' . $user_pic->extension;
        $path = Yii::getAlias('@runtime/' . $fileName);

        if ($user_pic->size > 15 * 1024 * 1024) { // 3Mb
            return [
                'result' => 'error',
                'message' => 'Размер файла не должен превышать 15 Мб'
            ];
        }

        $user_pic->saveAs($path);

        // fix orientation
        if ($user_pic->extension === 'jpg' || $user_pic->extension === 'png') {
            $deg = 0;
            $ext = $user_pic->extension;
            if (extension_loaded('exif')) {
                $exif = @exif_read_data($path);
            }
            $orientation = $exif['Orientation'] ?? 1;
            if (isset($orientation) && $orientation != 1){
                switch ($orientation) {
                    case 3:
                        $deg = 180;
                        break;
                    case 6:
                        $deg = 270;
                        break;
                    case 8:
                        $deg = 90;
                        break;
                }

                if ($deg) {

                    // If png
                    if ($ext == "png") {
                        $img_new = imagecreatefrompng($path);
                        $img_new = imagerotate($img_new, $deg, 0);

                        // Save rotated image
                        imagepng($img_new,$path);
                    }else {
                        $img_new = imagecreatefromjpeg($path);
                        $img_new = imagerotate($img_new, $deg, 0);

                        // Save rotated image
                        imagejpeg($img_new,$path,80);
                    }
                }
            }
        }

        $user->user_pic = basename(Yii::$app->avatar->update($user->username . '_' . $user->id, null, $path));
        if (file_exists($path)) {
            unlink($path);
        }

        $this->user_repository->save($user);
        return [
            'result' => 'success'
        ];
    }
}
