<?php
/**
 * ID-PASS Installer - Secure Version with Step Navigation
 */
session_start();

// Включаем отладку
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Определяем пути
define('ROOT_PATH', dirname(__DIR__));
define('INSTALL_PATH', ROOT_PATH . '/install');
define('CONFIG_PATH', ROOT_PATH . '/config');
define('INSTALL_LOCK_FILE', ROOT_PATH . '/.installed');

// ==================== БЕЗОПАСНОСТЬ ====================

// Если есть файл .installed - редирект на главную
if (file_exists(INSTALL_LOCK_FILE)) {
    header('Location: /');
    exit;
}

// Заголовки безопасности
header('X-Frame-Options: DENY');
header('X-Content-Type-Options: nosniff');
header('X-XSS-Protection: 1; mode=block');

function loadYii2Only() {
    $yiiPath = ROOT_PATH . '/vendor/yiisoft/yii2';
    require_once ROOT_PATH . '/vendor/yiisoft/yii2/Yii.php';

    // Подключаем BaseYii
    require_once $yiiPath . '/BaseYii.php';

    // Автозагрузка только Yii2 классов
    spl_autoload_register(function($className) use ($yiiPath) {
        // Только yii\ пространство имен
        if (strpos($className, 'yii\\') === 0) {
            $file = $yiiPath . '/' . str_replace('\\', '/', substr($className, 4)) . '.php';
            if (file_exists($file)) {
                require_once $file;
                return true;
            }
        }
        // Или классы из папки vendor/yiisoft (например, yii\composer)
        if (strpos($className, 'yii\\') === 0 || strpos($className, 'Yiisoft\\') === 0) {
            $file = ROOT_PATH . '/vendor/' . str_replace('\\', '/', $className) . '.php';
            if (file_exists($file)) {
                require_once $file;
                return true;
            }
        }
        return false;
    }, true, true);

    // Инициализируем Yii
    Yii::$classMap = [];
    Yii::$container = new \yii\di\Container();
}

// Токен CSRF
function generateCsrfToken() {
    if (empty($_SESSION['csrf_token'])) {
        $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
    }
    return $_SESSION['csrf_token'];
}

function validateCsrfToken($token) {
    if (empty($_SESSION['csrf_token']) || !hash_equals($_SESSION['csrf_token'], $token)) {
        throw new Exception('Недействительный CSRF токен');
    }
    return true;
}

// Валидация входных данных
function sanitizeInput($input) {
    if (is_array($input)) {
        return array_map('sanitizeInput', $input);
    }
    return htmlspecialchars(strip_tags(trim($input)), ENT_QUOTES, 'UTF-8');
}

// Проверка домена
function validateDomain($domain) {
    if (empty($domain)) {
        return '';
    }
    // Более простая проверка URL
    if (filter_var($domain, FILTER_VALIDATE_URL)) {
        return $domain;
    }
    // Если нет протокола, добавляем http://
    if (!preg_match('/^https?:\/\//', $domain)) {
        $domain = 'http://' . $domain;
    }
    if (filter_var($domain, FILTER_VALIDATE_URL)) {
        return $domain;
    }
    throw new Exception('Некорректный формат домена');
}

// Проверка email
function validateEmail($email) {
    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        throw new Exception('Некорректный email адрес');
    }
    return $email;
}

// Проверка порта
function validatePort($port) {
    $port = (int)$port;
    if ($port < 1 || $port > 65535) {
        throw new Exception('Некорректный номер порта');
    }
    return $port;
}

// ==================== УПРАВЛЕНИЕ ШАГАМИ ====================

$steps = [
        1 => [
                'title' => 'Требования',
                'requires' => [],
                'can_skip_to' => true
        ],
        2 => [
                'title' => 'База данных',
                'requires' => [],
                'can_skip_to' => true
        ],
        3 => [
                'title' => 'Параметры',
                'requires' => ['db_config'],
                'can_skip_to' => false
        ],
        4 => [
                'title' => 'Структура БД',
                'requires' => ['db_config', 'app_params'],
                'can_skip_to' => false
        ],
        5 => [
                'title' => 'Завершение',
                'requires' => [],
                'can_skip_to' => true
        ]
];

// Инициализация структуры сессии для установки
if (!isset($_SESSION['install_data'])) {
    $_SESSION['install_data'] = [
            'is_real_installation' => false,
            'current_step' => 1,
            'completed_steps' => [],
            'db_config' => null,
            'app_params' => null,
            'migration_result' => null,
            'admin_result' => null,
            'install_complete' => false
    ];
}

// Функция проверки возможности перехода на шаг
function canAccessStep($step, $sessionData) {
    global $steps;

    if (!isset($steps[$step])) {
        return false;
    }

    if ($steps[$step]['can_skip_to']) {
        return true;
    }

    foreach ($steps[$step]['requires'] as $requiredData) {
        if (!isset($sessionData[$requiredData])) {
            return false;
        }
    }

    return true;
}

// Получаем запрошенный шаг
$requestedStep = isset($_GET['step']) ? (int)$_GET['step'] : 1;
$requestedStep = max(1, min(5, $requestedStep));

// Проверяем доступность шага
if (!canAccessStep($requestedStep, $_SESSION['install_data'])) {
    $currentStep = 1;
    foreach ($steps as $stepNum => $stepConfig) {
        if (canAccessStep($stepNum, $_SESSION['install_data'])) {
            $currentStep = $stepNum;
            break;
        }
    }
} else {
    $currentStep = $requestedStep;
}

// Обновляем текущий шаг в сессии
$_SESSION['install_data']['current_step'] = $currentStep;

// ==================== ТЕСТОВЫЕ ДАННЫЕ ====================

function getTestData() {
    return [
            'is_real_installation' => false,
            'current_step' => 5,
            'completed_steps' => [1, 2, 3, 4, 5],
            'db_config' => [
                    'host' => 'localhost',
                    'port' => '5432',
                    'name' => 'idpass_db',
                    'user' => 'idpass_user',
                    'pass' => '********'
            ],
            'app_params' => [
                    'generalDomain' => 'https://example.id-pass.ru',
                    'adminEmail' => 'admin@example.com',
                    'senderEmail' => 'noreply@example.com',
                    'senderName' => 'ID-PASS'
            ],
            'migration_result' => [
                    'migrations_count' => 15,
                    'tables_count' => 23,
                    'migrations' => [
                            'm000000_000000_base',
                            'm130524_201442_init',
                            'm190124_110200_add_verification_token',
                            'm200312_204345_rbac_tables',
                            'm210101_000001_create_events_table',
                            'm210101_000002_create_participants_table',
                    ],
                    'tables' => [
                            'users', 'auth_assignment', 'auth_item', 'auth_item_child',
                            'auth_rule', 'migration', 'events', 'participants',
                            'profiles', 'settings', 'logs', 'notifications'
                    ]
            ],
            'admin_result' => [
                    'success' => true,
                    'username' => 'admin',
                    'email' => 'admin@example.com',
                    'password' => 'a1b2c3d4e5f67890',
                    'user_id' => '550e8400-e29b-41d4-a716-446655440000'
            ],
            'install_complete' => true
    ];
}

function getStep5Data() {
    // Если установка завершена и есть реальные данные в сессии
    if (isset($_SESSION['install_data']['install_complete']) &&
            $_SESSION['install_data']['install_complete'] === true) {
        return [
                'is_real' => true,
                'data' => $_SESSION['install_data']
        ];
    }

    // Проверяем наличие файла блокировки установки
    $isInstalled = file_exists(INSTALL_LOCK_FILE);

    // Если установка завершена, но данных в сессии нет (например, после перезагрузки)
    if ($isInstalled) {
        return [
                'is_real' => true,
                'data' => [
                        'is_real_installation' => true,
                        'install_complete' => true,
                        'app_params' => [
                                'generalDomain' => (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? "https" : "http") . "://" . $_SERVER['HTTP_HOST'],
                                'adminEmail' => 'admin@' . $_SERVER['HTTP_HOST'],
                        ],
                        'db_config' => [
                                'host' => 'localhost',
                                'name' => 'idpass',
                        ],
                        'admin_result' => [
                                'success' => false,
                                'error' => 'Данные администратора не сохранены в сессии',
                                'instructions' => 'Для создания администратора выполните команду: php yii user/create-admin',
                                'alternative' => 'Или войдите в систему и создайте пользователя через интерфейс'
                        ]
                ]
        ];
    }

    // Иначе показываем тестовые данные
    return [
            'is_real' => false,
            'data' => getTestData()
    ];
}

// ==================== ОСНОВНЫЕ ФУНКЦИИ ====================

function isInstallationComplete() {
    if (!defined('CONFIG_PATH') || !defined('INSTALL_LOCK_FILE')) {
        return false;
    }

    if (file_exists(INSTALL_LOCK_FILE)) {
        return true;
    }

    if (file_exists(CONFIG_PATH . '/db.php') &&
            file_exists(CONFIG_PATH . '/params_local.php')) {
        return true;
    }

    return false;
}

// Проверяем установку
if (isInstallationComplete()) {
    // Если установка завершена и мы не на 5 шаге, редиректим на 5
    if ($currentStep != 5) {
        header('Location: ?step=5');
        exit;
    }
}

function checkRequirements() {
    $requirements = [
            'PHP Version >= 8.3' => version_compare(PHP_VERSION, '8.3.0', '>='),
            'PHP Расширение PDO PostgreSQL' => extension_loaded('pdo_pgsql'),
            'PHP Расширение JSON' => extension_loaded('json'),
            'PHP Расширение MBString' => extension_loaded('mbstring'),
            'PHP Расширение cURL' => extension_loaded('curl'),
            'PHP Расширение ZIP' => extension_loaded('zip'),
            //'Yii2 установлен (vendor/yiisoft/yii2/Yii.php)' => file_exists(ROOT_PATH . '/vendor/yiisoft/yii2/Yii.php'),
            'Папка config доступна для записи' => is_writable(CONFIG_PATH) || (!file_exists(CONFIG_PATH) && is_writable(dirname(CONFIG_PATH))),
            'Папка сервиса доступна для записи' => is_writable(ROOT_PATH),
    ];

    return $requirements;
}

function testDatabaseConnection($dbConfig) {
    if (empty($dbConfig['host']) || empty($dbConfig['name']) || empty($dbConfig['user'])) {
        throw new Exception('Не заполнены обязательные параметры БД');
    }

    $dsn = "pgsql:host={$dbConfig['host']};port={$dbConfig['port']};dbname={$dbConfig['name']}";

    try {
        $pdo = new PDO($dsn, $dbConfig['user'], $dbConfig['pass']);
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        $pdo->setAttribute(PDO::ATTR_EMULATE_PREPARES, false);

        $version = $pdo->query('SHOW server_version')->fetchColumn();
        $majorVersion = (int) explode('.', $version)[0];
        if ($majorVersion < 12) {
            throw new Exception("Требуется PostgreSQL 12 или выше. У вас установлена версия $version");
        }

        return $pdo;
    } catch (PDOException $e) {
        throw new Exception('Ошибка подключения к БД: ' . $e->getMessage());
    }
}

function executeYiiMigrations($dbConfig) {
    // Проверяем наличие Yii2
    if (!file_exists(ROOT_PATH . '/vendor/yiisoft/yii2/Yii.php')) {
        throw new Exception('Yii2 не найден. Выполните: composer install');
    }

    // Подключаем Yii2
    loadYii2Only();

    // Временный конфиг для установки с authManager
    $config = [
            'id' => 'installer',
            'basePath' => ROOT_PATH,
            'components' => [
                    'db' => [
                            'class' => 'yii\db\Connection',
                            'dsn' => "pgsql:host={$dbConfig['host']};port={$dbConfig['port']};dbname={$dbConfig['name']}",
                            'username' => $dbConfig['user'],
                            'password' => $dbConfig['pass'],
                            'charset' => 'utf8',
                    ],
                    'authManager' => [
                            'class' => 'yii\rbac\DbManager',
                            'db' => 'db',
                    ],
            ],
    ];

    // Создаем приложение в консольном режиме
    $app = new \yii\console\Application($config);

    try {
        // ===== ПРОВЕРКА ПЕРЕД МИГРАЦИЯМИ =====
        // Получаем ВСЕ таблицы в базе (кроме системных)
        $existingTables = $app->db->createCommand("
            SELECT table_name 
            FROM information_schema.tables 
            WHERE table_schema = 'public'
            AND table_name NOT LIKE 'pg_%'
            AND table_name NOT IN ('spatial_ref_sys')
            ORDER BY table_name
        ")->queryColumn();

        // Если есть таблицы - БД не пустая
        if (!empty($existingTables)) {
            $message = "❌ База данных НЕ ПУСТАЯ!\n\n";
            $message .= "Обнаружены таблицы (" . count($existingTables) . "):\n";

            //foreach ($existingTables as $table) {
            //    $message .= "  • $table\n";
            //}

            $message .= "\nПеред продолжением необходимо:\n";
            $message .= "1. Удалить все таблицы из базы данных\n";
            $message .= "2. Или использовать другую, ПУСТУЮ базу данных\n\n";
            //$message .= "Как очистить базу:\n";
            //$message .= "1. Подключиться к PostgreSQL:\n";
            //$message .= "   psql -U {$dbConfig['user']} -d {$dbConfig['name']} -h {$dbConfig['host']}\n\n";
            //$message .= "2. Выполнить команды:\n";
            //$message .= "   DROP SCHEMA public CASCADE;\n";
            //$message .= "   CREATE SCHEMA public;\n";
            //$message .= "   GRANT ALL ON SCHEMA public TO {$dbConfig['user']};\n";
            //$message .= "   GRANT ALL ON SCHEMA public TO public;\n";

            throw new Exception($message);
        }

    } catch (\Exception $e) {
        throw new Exception("Проверка базы данных: " . $e->getMessage());
    }

    // ===== ВЫПОЛНЕНИЕ МИГРАЦИЙ =====
    $controller = new \yii\console\controllers\MigrateController('migrate', $app);
    $controller->interactive = false;
    $controller->color = false;

    try {
        // Запускаем миграции
        ob_start();
        $controller->runAction('up', ['interactive' => false]);
        $output = ob_get_clean();

        // Получаем список примененных миграций
        $appliedMigrations = $app->db->createCommand("
            SELECT version FROM {{%migration}} ORDER BY apply_time
        ")->queryColumn();

        // Получаем список созданных таблиц
        $tables = $app->db->createCommand("
            SELECT table_name 
            FROM information_schema.tables 
            WHERE table_schema = 'public'
            ORDER BY table_name
        ")->queryColumn();

        return [
                'success' => true,
                'migrations' => $appliedMigrations,
                'tables' => $tables,
                'migrations_count' => count($appliedMigrations),
                'tables_count' => count($tables),
                'output' => $output
        ];

    } catch (\Exception $e) {
        throw new Exception("Ошибка выполнения миграций: " . $e->getMessage());
    }
}

function createAdminUser($dbConfig, $params) {
    // Проверяем наличие Yii2
    if (!file_exists(ROOT_PATH . '/vendor/yiisoft/yii2/Yii.php')) {
        throw new Exception('Yii2 не найден. Выполните: composer install');
    }

    // Подключаем Yii2
    loadYii2Only();

    // Конфигурация приложения
    $config = [
            'id' => 'installer',
            'basePath' => ROOT_PATH,
            'components' => [
                    'db' => [
                            'class' => 'yii\db\Connection',
                            'dsn' => "pgsql:host={$dbConfig['host']};port={$dbConfig['port']};dbname={$dbConfig['name']}",
                            'username' => $dbConfig['user'],
                            'password' => $dbConfig['pass'],
                            'charset' => 'utf8',
                    ],
                    'authManager' => [
                            'class' => 'yii\rbac\DbManager',
                            'db' => 'db',
                    ],
            ],
    ];

    // Создаем приложение
    $app = new \yii\console\Application($config);

    try {
        // Генерируем случайный пароль
        $password = bin2hex(random_bytes(8)); // 16 символов

        // Создаем пользователя напрямую через SQL
        $authKey = Yii::$app->security->generateRandomString();
        $passwordHash = Yii::$app->security->generatePasswordHash($password);
        $timestamp = time();

        $userId = Yii::$app->db->createCommand("
            INSERT INTO users (id, username, email, auth_key, password_hash, status, created_at, updated_at)
            VALUES (gen_random_uuid(), :username, :email, :auth_key, :password_hash, 10, :time, :time)
            RETURNING id
        ", [
                ':username' => 'admin',
                ':email' => $params['adminEmail'],
                ':auth_key' => $authKey,
                ':password_hash' => $passwordHash,
                ':time' => $timestamp
        ])->queryScalar();

        // Назначаем роль admin
        Yii::$app->db->createCommand("
            INSERT INTO auth_assignment (item_name, user_id, created_at)
            VALUES ('admin', :user_id, :time)
        ", [
                ':user_id' => $userId,
                ':time' => $timestamp
        ])->execute();

        return [
                'success' => true,
                'username' => 'admin',
                'email' => $params['adminEmail'],
                'password' => $password,
                'user_id' => $userId
        ];

    } catch (\Exception $e) {
        throw new Exception("Ошибка создания администратора: " . $e->getMessage());
    }
}

function saveConfigFiles($dbConfig, $params) {
    // Конфиг базы данных
    $dbConfigContent = <<<PHP
<?php

return [
    'class' => 'yii\db\Connection',
    'dsn' => 'pgsql:host={$dbConfig['host']};port={$dbConfig['port']};dbname={$dbConfig['name']}',
    'username' => '{$dbConfig['user']}',
    'password' => '{$dbConfig['pass']}',
    'charset' => 'utf8',
    'schemaMap' => [
        'pgsql' => [
            'class' => 'yii\db\pgsql\Schema',
            'defaultSchema' => 'public'
        ]
    ],
    'on afterOpen' => function(\$event) {
        \$event->sender->createCommand("SET timezone = 'UTC'")->execute();
    }
];
PHP;

    // Параметры приложения
    $params['jwtExpired'] = 86400 * 30;
    $params['jwtIssuer'] = $params['generalDomain'];
    $params['jwtAudience'] = $params['generalDomain'];
    $params['apiFatherUrl'] = 'https://shop.id-pass.ru';
    $params['demo'] = false;

    $paramsContent = "<?php\n\nreturn " . var_export($params, true) . ";\n";

    // Создаем директорию config если её нет
    if (!is_dir(CONFIG_PATH)) {
        mkdir(CONFIG_PATH, 0755, true);
    }

    // Записываем файлы
    file_put_contents(CONFIG_PATH . '/db.php', $dbConfigContent);
    file_put_contents(CONFIG_PATH . '/params_local.php', $paramsContent);

    // Создаем файл блокировки установки
    file_put_contents(INSTALL_LOCK_FILE, json_encode([
            'installed_at' => date('Y-m-d H:i:s'),
            'admin_email' => $params['adminEmail'],
            'domain' => $params['generalDomain'],
            'note' => 'Установка завершена. Данные администратора были показаны во время установки.'
    ], JSON_PRETTY_PRINT));

    return true;
}

// ==================== ОБРАБОТКА ====================

$error = '';
$success = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        if (!isset($_POST['csrf_token'])) {
            throw new Exception('Отсутствует CSRF токен');
        }
        validateCsrfToken($_POST['csrf_token']);

        switch ($currentStep) {
            case 2:
                $dbConfig = [
                        'host' => sanitizeInput($_POST['db_host'] ?? 'localhost'),
                        'port' => validatePort($_POST['db_port'] ?? '5432'),
                        'name' => sanitizeInput($_POST['db_name'] ?? 'idpass'),
                        'user' => sanitizeInput($_POST['db_user'] ?? 'idpass_user'),
                        'pass' => $_POST['db_pass'] ?? '',
                ];

                testDatabaseConnection($dbConfig);
                $_SESSION['install_data']['db_config'] = $dbConfig;
                $success = 'Подключение успешно!';
                header('Location: ?step=3');
                exit;

            case 3:
                if (!isset($_SESSION['install_data']['db_config'])) {
                    throw new Exception('Конфигурация базы данных не найдена');
                }

                $currentDomain = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? "https" : "http") . "://" . $_SERVER['HTTP_HOST'];

                $params = [
                        'generalDomain' => validateDomain($_POST['general_domain'] ?? $currentDomain),
                        'adminEmail' => validateEmail($_POST['admin_email'] ?? ''),
                        'senderEmail' => validateEmail($_POST['sender_email'] ?? $_POST['admin_email'] ?? ''),
                        'senderName' => sanitizeInput($_POST['sender_name'] ?? 'ID-PASS'),
                        'jwtSecretKey' => 'base64:' . base64_encode(random_bytes(64)),
                ];

                $_SESSION['install_data']['app_params'] = $params;
                $success = 'Параметры сохранены!';
                header('Location: ?step=4');
                exit;

            case 4:
                // Сбрасываем предыдущие результаты
                unset($_SESSION['install_data']['migration_result']);
                unset($_SESSION['install_data']['admin_result']);
                unset($_SESSION['install_data']['install_complete']);

                if (!isset($_SESSION['install_data']['db_config'])) {
                    throw new Exception('Конфигурация базы данных не найдена');
                }

                // Выполняем миграции
                $migrationResult = executeYiiMigrations($_SESSION['install_data']['db_config']);
                $_SESSION['install_data']['migration_result'] = $migrationResult;

                // Создаем администратора
                try {
                    $adminResult = createAdminUser(
                            $_SESSION['install_data']['db_config'],
                            $_SESSION['install_data']['app_params']
                    );
                    $_SESSION['install_data']['admin_result'] = $adminResult;
                } catch (Exception $e) {
                    $_SESSION['install_data']['admin_result'] = [
                            'success' => false,
                            'error' => $e->getMessage()
                    ];
                }

                // После успешного создания администратора
                $_SESSION['install_data']['install_complete'] = true;
                $_SESSION['install_data']['is_real_installation'] = true;

                $success = 'Миграции успешно выполнены!';
                //header('Location: ?step=5');
                //exit;

                // Принудительно сохраняем сессию
                session_write_close();

                // Редирект с параметром для обновления
                header('Location: ?step=5&r=' . time());
                exit;

            case 5:
                if (!isset($_SESSION['install_data']['db_config']) ||
                        !isset($_SESSION['install_data']['app_params'])) {
                    throw new Exception('Данные установки не найдены');
                }

                // Сохраняем конфигурационные файлы
                saveConfigFiles(
                        $_SESSION['install_data']['db_config'],
                        $_SESSION['install_data']['app_params']
                );

                // Отмечаем установку как завершенную
                // $_SESSION['install_data']['install_complete'] = true;
                // $_SESSION['install_data']['is_real_installation'] = true;

                $success = 'Установка завершена!';
                header('Location: ?step=5');
                exit;
        }
    } catch (Exception $e) {
        $error = 'Ошибка: ' . $e->getMessage();
    }
}

// ==================== ВЫВОД HTML ====================
?>
<!DOCTYPE html>
<html lang="ru">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Установка ID-PASS</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            display: flex;
            justify-content: center;
            align-items: center;
            padding: 20px;
        }
        .installer-container {
            background: white;
            border-radius: 20px;
            box-shadow: 0 20px 60px rgba(0,0,0,0.3);
            width: 100%;
            max-width: 900px;
            overflow: hidden;
        }
        .header {
            background: linear-gradient(135deg, #4f46e5 0%, #7c3aed 100%);
            color: white;
            padding: 30px;
            text-align: center;
            position: relative;
        }
        .header h1 { font-size: 28px; margin-bottom: 10px; }
        .header .subtitle { opacity: 0.9; font-size: 16px; margin-bottom: 15px; }
        .dev-mode {
            position: absolute;
            top: 15px;
            right: 15px;
            background: rgba(255,255,255,0.2);
            padding: 5px 15px;
            border-radius: 20px;
            font-size: 12px;
            font-weight: bold;
        }
        .steps-nav {
            display: flex;
            background: #f1f5f9;
            padding: 0;
            position: relative;
        }
        .step {
            flex: 1;
            text-align: center;
            padding: 15px;
            position: relative;
            color: #64748b;
            font-weight: 500;
            cursor: pointer;
            transition: all 0.3s;
            text-decoration: none;
            display: block;
        }
        .step:hover {
            background: rgba(255,255,255,0.3);
        }
        .step.active {
            color: #4f46e5;
            background: white;
            cursor: default;
        }
        .step.completed {
            color: #10b981;
        }
        .step.disabled {
            opacity: 0.5;
            cursor: not-allowed;
        }
        .step-number {
            display: inline-block;
            width: 30px;
            height: 30px;
            line-height: 30px;
            border-radius: 50%;
            background: #e2e8f0;
            margin-right: 10px;
            color: #64748b;
        }
        .step.active .step-number {
            background: #4f46e5;
            color: white;
        }
        .step.completed .step-number {
            background: #10b981;
            color: white;
        }
        .step.disabled .step-number {
            background: #cbd5e1;
        }
        .step-link {
            text-decoration: none;
            color: inherit;
            display: block;
        }
        .content { padding: 40px; }
        .form-group { margin-bottom: 25px; }
        label {
            display: block;
            margin-bottom: 8px;
            color: #334155;
            font-weight: 500;
            font-size: 14px;
        }
        input[type="text"],
        input[type="password"],
        input[type="email"],
        input[type="number"] {
            width: 100%;
            padding: 12px 16px;
            border: 2px solid #e2e8f0;
            border-radius: 10px;
            font-size: 16px;
            transition: all 0.3s;
        }
        input:focus {
            outline: none;
            border-color: #4f46e5;
            box-shadow: 0 0 0 3px rgba(79, 70, 229, 0.1);
        }
        .btn {
            background: linear-gradient(135deg, #4f46e5 0%, #7c3aed 100%);
            color: white;
            border: none;
            padding: 14px 30px;
            border-radius: 10px;
            font-size: 16px;
            font-weight: 500;
            cursor: pointer;
            transition: all 0.3s;
            display: inline-flex;
            align-items: center;
            gap: 8px;
            text-decoration: none;
        }
        .btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 10px 20px rgba(79, 70, 229, 0.3);
        }
        .btn-secondary { background: #64748b; }
        .btn-success { background: linear-gradient(135deg, #10b981 0%, #059669 100%); }
        .btn-warning { background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%); }
        .btn-danger { background: linear-gradient(135deg, #dc2626 0%, #b91c1c 100%); }
        .error {
            background: #fee2e2;
            border: 1px solid #fca5a5;
            color: #dc2626;
            padding: 15px;
            border-radius: 10px;
            margin-bottom: 20px;
        }
        .success {
            background: #d1fae5;
            border: 1px solid #a7f3d0;
            color: #059669;
            padding: 15px;
            border-radius: 10px;
            margin-bottom: 20px;
        }
        .warning {
            background: #fef3c7;
            border: 1px solid #f59e0b;
            color: #d97706;
            padding: 15px;
            border-radius: 10px;
            margin-bottom: 20px;
        }
        .actions {
            display: flex;
            justify-content: space-between;
            margin-top: 30px;
            padding-top: 20px;
            border-top: 1px solid #e2e8f0;
        }
        .requirement-list { list-style: none; }
        .requirement-list li {
            padding: 10px 0;
            border-bottom: 1px solid #e2e8f0;
            display: flex;
            align-items: center;
            gap: 10px;
        }
        .requirement-list li:last-child { border-bottom: none; }
        .status-ok { color: #10b981; }
        .status-error { color: #dc2626; }
        .migration-list { margin: 20px 0; }
        .migration-item {
            padding: 10px;
            border: 1px solid #e2e8f0;
            border-radius: 8px;
            margin-bottom: 10px;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        .migration-status {
            padding: 4px 12px;
            border-radius: 20px;
            font-size: 12px;
            font-weight: 500;
        }
        .status-success { background: #d1fae5; color: #059669; }
        .table-list {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(200px, 1fr));
            gap: 10px;
            margin: 20px 0;
        }
        .table-item {
            background: #f8fafc;
            border: 1px solid #e2e8f0;
            border-radius: 8px;
            padding: 10px;
            text-align: center;
        }
        .completion-message { text-align: center; padding: 40px 0; }
        .completion-message h2 { color: #059669; margin-bottom: 20px; }
        .instructions {
            background: #f8fafc;
            border-radius: 10px;
            padding: 20px;
            margin-top: 30px;
        }
        .instructions h3 { margin-bottom: 15px; color: #334155; }
        .instructions ul { padding-left: 20px; color: #64748b; }
        .instructions li { margin-bottom: 8px; }
        code {
            background: #f1f5f9;
            padding: 2px 6px;
            border-radius: 4px;
            font-family: monospace;
        }
        .debug-info {
            background: #fef3c7;
            border: 1px solid #f59e0b;
            padding: 15px;
            border-radius: 10px;
            margin-top: 20px;
            font-size: 12px;
        }
        .debug-info h4 {
            color: #d97706;
            margin-bottom: 10px;
        }
        .debug-info pre {
            background: #fef9c3;
            padding: 10px;
            border-radius: 5px;
            overflow: auto;
            font-family: monospace;
        }
        .btn:disabled {
            opacity: 0.6;
            cursor: not-allowed;
            transform: none !important;
            box-shadow: none !important;
        }
        .password-box {
            background: #f8fafc;
            border: 2px dashed #e2e8f0;
            border-radius: 10px;
            padding: 20px;
            margin: 20px 0;
            text-align: center;
        }
        .password-display {
            font-size: 24px;
            font-weight: bold;
            font-family: monospace;
            color: #059669;
            margin: 15px 0;
            padding: 15px;
            background: white;
            border-radius: 8px;
            border: 2px solid #a7f3d0;
            letter-spacing: 2px;
        }
        .password-warning {
            color: #dc2626;
            font-weight: bold;
            margin: 15px 0;
            padding: 10px;
            background: #fee2e2;
            border-radius: 8px;
            border: 1px solid #fca5a5;
        }
        .copy-btn {
            background: #3b82f6;
            color: white;
            border: none;
            padding: 8px 20px;
            border-radius: 6px;
            cursor: pointer;
            margin-top: 10px;
            font-size: 14px;
        }
        .copy-btn:hover {
            background: #2563eb;
        }
        .installed-link {
            display: inline-block;
            margin-top: 20px;
            padding: 15px 30px;
            background: linear-gradient(135deg, #10b981 0%, #059669 100%);
            color: white;
            text-decoration: none;
            border-radius: 10px;
            font-weight: bold;
            font-size: 18px;
        }
        .installed-link:hover {
            transform: translateY(-2px);
            box-shadow: 0 10px 20px rgba(16, 185, 129, 0.3);
        }
    </style>
</head>
<body>
<div class="installer-container">
    <div class="header">
        <div class="dev-mode">РЕЖИМ РАЗРАБОТКИ</div>
        <h1>Установка ID-PASS</h1>
        <div class="subtitle">Система управления событиями и участниками</div>
        <div class="subtitle" style="font-size: 14px; opacity: 0.7;">
            Вы можете перейти на любой шаг для корректировки дизайна
        </div>
    </div>

    <div class="steps-nav">
        <?php foreach ($steps as $stepNumber => $stepConfig):
            $isActive = $stepNumber == $currentStep;
            $isCompleted = isset($_SESSION['install_data']['completed_steps']) &&
                    in_array($stepNumber, $_SESSION['install_data']['completed_steps']);
            $isAccessible = canAccessStep($stepNumber, $_SESSION['install_data']);
            $cssClass = '';

            if ($isActive) {
                $cssClass = 'active';
            } elseif ($isCompleted) {
                $cssClass = 'completed';
            } elseif (!$isAccessible) {
                $cssClass = 'disabled';
            }
            ?>
            <?php if ($isAccessible): ?>
            <a href="?step=<?php echo $stepNumber; ?>" class="step <?php echo $cssClass; ?>">
                <span class="step-number"><?php echo $stepNumber; ?></span>
                <div class="step-name"><?php echo $stepConfig['title']; ?></div>
            </a>
        <?php else: ?>
            <div class="step <?php echo $cssClass; ?>">
                <span class="step-number"><?php echo $stepNumber; ?></span>
                <div class="step-name"><?php echo $stepConfig['title']; ?></div>
            </div>
        <?php endif; ?>
        <?php endforeach; ?>
    </div>

    <div class="content">
        <?php if ($error): ?>
            <div class="error" style="white-space: pre-wrap;"><?php echo htmlspecialchars($error); ?></div>
        <?php endif; ?>

        <?php if ($success): ?>
            <div class="success"><?php echo htmlspecialchars($success); ?></div>
        <?php endif; ?>

        <?php if ($currentStep != 5): ?>
            <form method="POST">
                <input type="hidden" name="csrf_token" value="<?php echo generateCsrfToken(); ?>">

                <?php if ($currentStep == 1): ?>
                    <h2>Проверка требований</h2>
                    <p>Перед началом установки убедитесь, что выполнены все системные требования:</p>

                    <ul class="requirement-list">
                        <?php
                        $requirements = checkRequirements();
                        $allOk = true;
                        foreach ($requirements as $req => $status):
                            if (!$status) $allOk = false;
                            ?>
                            <li>
                                <?php if ($status): ?>
                                    <span class="status-ok">✓</span>
                                <?php else: ?>
                                    <span class="status-error">✗</span>
                                <?php endif; ?>
                                <?php echo htmlspecialchars($req); ?>
                            </li>
                        <?php endforeach; ?>
                    </ul>

                    <div class="actions">
                        <div>
                            <button type="button" class="btn btn-warning" onclick="location.reload()">
                                ↻ Обновить проверку
                            </button>
                        </div>
                        <?php if ($allOk): ?>
                            <a href="?step=2" class="btn">
                                Продолжить →
                            </a>
                        <?php else: ?>
                            <button type="button" class="btn" disabled>
                                Исправьте требования
                            </button>
                        <?php endif; ?>
                    </div>

                <?php elseif ($currentStep == 2): ?>
                    <h2>Настройка базы данных PostgreSQL</h2>
                    <p>Укажите параметры подключения к базе данных:</p>

                    <?php
                    $dbConfig = $_SESSION['install_data']['db_config'] ?? [
                            'host' => 'localhost',
                            'port' => '5432',
                            'name' => 'idpass',
                            'user' => 'idpass_user',
                            'pass' => ''
                    ];
                    ?>

                    <div class="form-group">
                        <label for="db_host">Хост</label>
                        <input type="text" id="db_host" name="db_host"
                               value="<?php echo htmlspecialchars($dbConfig['host']); ?>" required>
                    </div>

                    <div class="form-group">
                        <label for="db_port">Порт</label>
                        <input type="number" id="db_port" name="db_port"
                               value="<?php echo htmlspecialchars($dbConfig['port']); ?>" required>
                    </div>

                    <div class="form-group">
                        <label for="db_name">Имя базы данных</label>
                        <input type="text" id="db_name" name="db_name"
                               value="<?php echo htmlspecialchars($dbConfig['name']); ?>" required>
                        <small style="color: #64748b; font-size: 12px;">База данных должна быть создана заранее</small>
                    </div>

                    <div class="form-group">
                        <label for="db_user">Пользователь</label>
                        <input type="text" id="db_user" name="db_user"
                               value="<?php echo htmlspecialchars($dbConfig['user']); ?>" required>
                    </div>

                    <div class="form-group">
                        <label for="db_pass">Пароль</label>
                        <input type="password" id="db_pass" name="db_pass"
                               value="<?php echo htmlspecialchars($dbConfig['pass']); ?>"
                                <?php echo empty($dbConfig['pass']) ? '' : 'placeholder="(уже установлен)"'; ?>>
                    </div>

                    <div class="actions">
                        <a href="?step=1" class="btn btn-secondary">← Назад</a>
                        <button type="submit" class="btn">Проверить подключение →</button>
                    </div>

                <?php elseif ($currentStep == 3): ?>
                    <h2>Настройка параметров приложения</h2>

                    <?php
                    $currentDomain = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? "https" : "http") . "://" . $_SERVER['HTTP_HOST'];
                    $params = $_SESSION['install_data']['app_params'] ?? [
                            'generalDomain' => $currentDomain,
                            'adminEmail' => '',
                            'senderEmail' => '',
                            'senderName' => 'ID-PASS'
                    ];
                    ?>

                    <div class="form-group">
                        <label for="general_domain">Домен приложения</label>
                        <input type="text" id="general_domain" name="general_domain"
                               value="<?php echo htmlspecialchars($params['generalDomain']); ?>" required>
                        <small style="color: #64748b; font-size: 12px;">URL, по которому будет доступно приложение</small>
                    </div>

                    <div class="form-group">
                        <label for="admin_email">Email администратора</label>
                        <input type="email" id="admin_email" name="admin_email"
                               value="<?php echo htmlspecialchars($params['adminEmail']); ?>" required>
                    </div>

                    <div class="form-group">
                        <label for="sender_email">Email отправителя</label>
                        <input type="email" id="sender_email" name="sender_email"
                               value="<?php echo htmlspecialchars($params['senderEmail']); ?>">
                        <small style="color: #64748b; font-size: 12px;">Если не указан, будет использован email администратора</small>
                    </div>

                    <div class="form-group">
                        <label for="sender_name">Имя отправителя</label>
                        <input type="text" id="sender_name" name="sender_name"
                               value="<?php echo htmlspecialchars($params['senderName']); ?>" required>
                    </div>

                    <div class="actions">
                        <a href="?step=2" class="btn btn-secondary">← Назад</a>
                        <button type="submit" class="btn">Сохранить параметры →</button>
                    </div>

                <?php elseif ($currentStep == 4): ?>
                    <?php
                    // Очищаем результаты миграций ПРИ ЛЮБОМ заходе на шаг 4
                    // Это сработает и при обновлении страницы (GET)
                    unset($_SESSION['install_data']['migration_result']);
                    unset($_SESSION['install_data']['admin_result']);
                    unset($_SESSION['install_data']['install_complete']);
                    ?>

                    <h2>Подготовка структуры базы данных</h2>

                    <?php if (isset($_SESSION['install_data']['migration_result'])): ?>
                        <?php $result = $_SESSION['install_data']['migration_result']; ?>
                        <div class="success">
                            <h3>✅ База данных подготовлена!</h3>
                            <!-- <p>Выполнено миграций: <?php echo $result['migrations_count'] ?? 0; ?></p>
                            <p>Создано таблиц: <?php echo $result['tables_count'] ?? 0; ?></p> -->

                            <!-- <?php if (!empty($result['migrations'])): ?>
                                <div style="margin-top: 15px;">
                                    <h4>Примененные миграции:</h4>
                                    <div class="migration-list">
                                        <?php foreach ($result['migrations'] as $migration): ?>
                                            <div class="migration-item">
                                                <span><?php echo htmlspecialchars($migration); ?></span>
                                                <span class="migration-status status-success">✅ Выполнена</span>
                                            </div>
                                        <?php endforeach; ?>
                                    </div>
                                </div>
                            <?php endif; ?>

                            <?php if (!empty($result['tables'])): ?>
                                <div style="margin-top: 15px;">
                                    <h4>Созданные таблицы:</h4>
                                    <div class="table-list">
                                        <?php foreach ($result['tables'] as $table): ?>
                                            <div class="table-item"><?php echo htmlspecialchars($table); ?></div>
                                        <?php endforeach; ?>
                                    </div>
                                </div>
                            <?php endif; ?> -->
                        </div>

                        <div class="actions">
                            <a href="?step=3" class="btn btn-secondary">← Назад</a>
                            <a href="?step=5" class="btn btn-success">Продолжить →</a>
                        </div>
                    <?php else: ?>
                        <p style="margin-top: 20px;">Сейчас будет создана структура базы данных.</p>

                        <div class="warning" style="margin-top: 20px;">
                            <h3>⚠️ Внимание!</h3>
                            <p>Перед созданием структуры будет проверено, что база данных ПУСТАЯ.</p>
                            <p>Если в базе уже есть таблицы, установка прервется.</p>
                        </div>

                        <!-- <div class="instructions">
							<h3>Что будет выполнено:</h3>
							<ul>
								<li>Проверка что база данных пустая</li>
								<li>Все миграции из папки <code>migrations/</code></li>
								<li>Автоматическое создание таблиц RBAC</li>
								<li>Создание таблицы для отслеживания миграций</li>
							</ul>
							<p><strong>Если нужно очистить базу:</strong></p>
							<code style="display: block; padding: 10px; margin: 10px 0; background: #f1f5f9;">
								psql -U <?php echo htmlspecialchars($_SESSION['install_data']['db_config']['user'] ?? 'idpass_user'); ?> -d <?php echo htmlspecialchars($_SESSION['install_data']['db_config']['name'] ?? 'idpass'); ?><br>
								DROP SCHEMA public CASCADE;<br>
								CREATE SCHEMA public;
							</code>
						</div> -->

                        <div class="actions">
                            <a href="?step=3" class="btn btn-secondary">← Назад</a>
                            <button type="submit" class="btn btn-success">Выполнить проверку и миграции →</button>
                        </div>
                    <?php endif; ?>

                <?php endif; ?>
            </form>
        <?php else: // ШАГ 5 ?>

        <?php if ($currentStep == 5): ?>

        <?php endif; ?>

        <?php
        $step5Data = getStep5Data();
        $data = $step5Data['data'];
        $isRealData = $step5Data['is_real'];
        $isInstalled = file_exists(INSTALL_LOCK_FILE);
        ?>

            <div class="completion-message">
                <?php if ($isRealData && $data['is_real_installation']): ?>

                    <div>
                        <svg style="width: 100px; height: 100px; color: #059669;" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" viewBox="0 0 24 24"><path d="M12 2C6.48 2 2 6.48 2 12s4.48 10 10 10s10-4.48 10-10S17.52 2 12 2zm0 18c-4.41 0-8-3.59-8-8s3.59-8 8-8s8 3.59 8 8s-3.59 8-8 8zm4.59-12.42L10 14.17l-2.59-2.58L6 13l4 4l8-8z" fill="currentColor"></path></svg>
                    </div>

                    <h2>Установка <?php echo $isInstalled ? 'завершена' : 'почти завершена'; ?>!</h2>
                    <p>ID-PASS успешно <?php echo $isInstalled ? 'установлен и настроен' : 'настроен'; ?>.</p>

                    <?php if (isset($data['admin_result']['success']) && $data['admin_result']['success']): ?>
                        <div class="password-box">
                            <h3>🔐 Данные для входа администратора</h3>
                            <p><strong>Логин (email):</strong> <?php echo htmlspecialchars($data['admin_result']['email']); ?></p>
                            <div class="password-display" id="adminPassword"><?php echo htmlspecialchars($data['admin_result']['password']); ?></div>

                            <div style="display: flex; align-items: center; justify-content: center;">
                                <button class="copy-btn" onclick="copyPassword()" style="display: flex; align-items: center; justify-content: center; font-size: 1.2em; gap: 5px;">
                                    <svg style="width: 25px;" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" viewBox="0 0 24 24"><path d="M16 1H4c-1.1 0-2 .9-2 2v14h2V3h12V1zm3 4H8c-1.1 0-2 .9-2 2v14c0 1.1.9 2 2 2h11c1.1 0 2-.9 2-2V7c0-1.1-.9-2-2-2zm0 16H8V7h11v14z" fill="currentColor"></path></svg>
                                    Скопировать пароль
                                </button>
                            </div>

                            <div class="password-warning">
                                ⚠️ СОХРАНИТЕ ЭТОТ ПАРОЛЬ! Он больше не будет показан.
                            </div>
                        </div>
                    <?php else: ?>
                        <div class="warning">
                            <h3>⚠️ Администратор не создан автоматически</h3>
                            <p>Для создания администратора выполните команду:</p>
                            <code style="display: block; padding: 10px; margin: 10px 0; background: #f1f5f9;">
                                php yii user/create-admin
                            </code>
                        </div>
                    <?php endif; ?>

                <?php else: ?>
                    <h2>📋 Предпросмотр завершения установки</h2>
                    <div class="warning">
                        <strong>ℹ️ Тестовые данные:</strong> Это предпросмотр. Для реальной установки пройдите все шаги.
                    </div>

                    <?php if (isset($data['admin_result']['success']) && $data['admin_result']['success']): ?>
                        <div class="password-box" style="background: #fef3c7; border-color: #f59e0b;">
                            <h3>🔐 Данные для входа администратора (тестовые)</h3>
                            <p><strong>Логин (email):</strong> <?php echo htmlspecialchars($data['admin_result']['email']); ?></p>
                            <div class="password-display" style="color: #d97706; border-color: #fbbf24;"><?php echo htmlspecialchars($data['admin_result']['password']); ?></div>
                            <p style="color: #d97706; font-style: italic;">Это тестовый пароль, в реальной установке он будет другим.</p>
                        </div>
                    <?php endif; ?>
                <?php endif; ?>

                <div class="instructions" style="text-align: left;">
                    <h3>Информация об установке:</h3>
                    <ul>
                        <li><strong>Приложение:</strong> <?php echo htmlspecialchars($data['app_params']['generalDomain'] ?? 'https://example.id-pass.ru'); ?></li>
                        <li><strong>База данных:</strong> <?php echo htmlspecialchars($data['db_config']['name'] ?? 'idpass_db'); ?>@<?php echo htmlspecialchars($data['db_config']['host'] ?? 'localhost'); ?></li>
                        <!-- <?php if (isset($data['migration_result']['migrations_count'])): ?>
                            <li><strong>Выполнено миграций:</strong> <?php echo $data['migration_result']['migrations_count']; ?></li>
                        <?php endif; ?> -->
                        <?php if (isset($data['migration_result']['tables_count'])): ?>
                            <li><strong>Создано таблиц:</strong> <?php echo $data['migration_result']['tables_count']; ?></li>
                        <?php endif; ?>
                    </ul>

                    <h3 style="margin-top: 20px;">Дальнейшие действия:</h3>
                    <ul>
                        <?php if ($isRealData && $data['is_real_installation']): ?>
                            <li>Используйте данные администратора для входа в систему</li>
                            <li>Смените пароль администратора после входа</li>
                            <?php if (!$isInstalled): ?>
                                <li>Нажмите "Завершить установку" для сохранения конфигурационных файлов</li>
                            <?php else: ?>
                                <li>Для повторной настройки очитите базу данных, удалите файлы и распкуйте архив снова</li>
                            <?php endif; ?>
                        <?php else: ?>
                            <li>Для начала установки перейдите на шаг 1</li>
                            <li>После установки вы получите реальные данные для входа</li>
                        <?php endif; ?>
                    </ul>
                </div>

                <?php if ($isRealData && $data['is_real_installation']): ?>
                    <?php if ($isInstalled): ?>
                        <div style="margin-top: 30px;">
                            <a href="<?php echo htmlspecialchars($data['app_params']['generalDomain'] ?? '/'); ?>"
                               class="installed-link" target="_blank">
                                Перейти в приложение
                            </a>
                            <?php if (isset($data['admin_result']['success']) && $data['admin_result']['success']): ?>
                                <!-- <button type="button" class="btn btn-danger" onclick="clearSession()" style="margin-left: 10px;">
                                    Скрыть пароль
                                </button> -->
                            <?php endif; ?>
                        </div>
                    <?php else: ?>
                        <div style="margin-top: 30px;">
                            <form method="POST">
                                <input type="hidden" name="csrf_token" value="<?php echo generateCsrfToken(); ?>">
                                <button type="submit" class="btn btn-success" style="padding: 15px 40px; font-size: 18px;">
                                    Завершить установку
                                </button>
                                <p style="margin-top: 10px; color: #64748b; font-size: 14px;">
                                    Сохранит конфигурационные файлы и создаст файл блокировки
                                </p>
                            </form>
                        </div>
                    <?php endif; ?>
                <?php else: ?>
                    <div style="margin-top: 30px; display: flex; gap: 10px; justify-content: center;">
                        <a href="?step=1" class="btn btn-secondary">
                            ← Начать установку
                        </a>
                        <?php if (isset($_SESSION['install_data']['db_config']) &&
                                isset($_SESSION['install_data']['app_params']) &&
                                isset($_SESSION['install_data']['migration_result'])): ?>
                            <form method="POST" style="display: inline;">
                                <input type="hidden" name="csrf_token" value="<?php echo generateCsrfToken(); ?>">
                                <button type="submit" class="btn btn-success">
                                    Завершить установку
                                </button>
                            </form>
                        <?php else: ?>
                            <button type="button" class="btn" disabled title="Сначала пройдите предыдущие шаги">
                                Завершить установку (недоступно)
                            </button>
                        <?php endif; ?>
                    </div>
                <?php endif; ?>
            </div>

        <?php if ($isRealData && $data['is_real_installation'] && isset($data['admin_result']['success']) && $data['admin_result']['success']): ?>
            <script>
                // Автоматически очищаем сессию после показа пароля (через 5 минут)
                setTimeout(function() {
                    fetch('?clear_session=1').then(() => {
                        console.log('Сессия очищена');
                    });
                }, 300000); // 5 минут
            </script>
        <?php endif; ?>

        <?php endif; ?>
    </div>
</div>

<script>
    function copyPassword() {
        const passwordElement = document.getElementById('adminPassword');
        const password = passwordElement.textContent.trim();

        navigator.clipboard.writeText(password).then(function() {
            const btn = document.querySelector('.copy-btn');
            const originalText = btn.textContent;
            btn.textContent = 'Скопировано!';
            btn.style.background = '#10b981';

            setTimeout(function() {
                btn.textContent = originalText;
                btn.style.background = '#3b82f6';
            }, 2000);
        }).catch(function(err) {
            alert('Не удалось скопировать пароль: ' + err);
        });
    }

    function clearSession() {
        if (confirm('Вы уверены, что хотите скрыть пароль? После этого он больше не будет показан.')) {
            fetch('?clear_session=1').then(function() {
                location.reload();
            });
        }
    }

    // Обработка очистки сессии
    <?php
    if (isset($_GET['clear_session']) && $currentStep == 5) {
        // Очищаем только данные администратора
        if (isset($_SESSION['install_data']['admin_result'])) {
            unset($_SESSION['install_data']['admin_result']['password']);
            $_SESSION['install_data']['admin_result']['password_hidden'] = true;
        }
    }
    ?>
</script>
</body>
</html>